/*
 *  ALSA sequencer Timer
 *  Copyright (c) 1998-1999 by Frank van de Pol <frank@vande-pol.demon.nl>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
#ifndef __SND_SEQ_TIMER_H
#define __SND_SEQ_TIMER_H

#include "../../include/timer.h"
#include "../../include/seq_kernel.h"

typedef struct {
	/* ... tempo / offset / running state */

	//	
	//
	//	   Different sync types (for slave operation):
	//	   
	//		midi <- seq. clock
	//		  "  <- midi clock  (adjust tempo)
	//		
	//		seq. clock <- hw timer
	//		     "     <- MTC
	//		     "     <- SMPTE
	//
	
	int			running:1;	/* running state of queue */	

	unsigned int		tempo;		/* current tempo, us/tick */
	int			ppq;		/* time resolution, ticks/quarter */

	snd_seq_tick_time_t	tempo_tick;	/* tick of last tempo change */
	snd_seq_real_time_t	tempo_time;	/* time of last tempo change */

	snd_seq_tick_time_t	cur_tick;	/* current tick */
	snd_seq_real_time_t	cur_time;	/* current time */
	
	unsigned long		tick_time;	/* time per tick in nsec */
	unsigned long		cur_tick_time;	/* current time per tick in nsec */

#ifdef use_seqsync
	snd_seq_tick_time_t	sync_tmp;	/* tmp var for sync */
#endif

	int			type;		/* timer type */
	int			timer_no;	/* timer number */
	int			slave_type;	/* slave type */
	int			resolution;	/* resolution in Hz */
	snd_timer_instance_t	*timeri;	/* timer instance */
	unsigned long		ticks;		/* ticks per period */

#ifdef use_seqsync
	int			midi_client;	/* timer source - client */
	int			midi_port;	/* timer source - port */
	
	long int		sync_tick_resolution;  /* resolution per 10ms midi tick --FP:What's this for ???? */
	long int		sync_real_resolution;  /* resolution per midi tick (nanoseconds) */
#endif

	struct timeval 		last_update;	 /* time of last clock update, used for interpolation */

	spinlock_t lock;
} seq_timer_t;


/* create new timer (constructor) */
extern seq_timer_t *snd_seq_timer_new(void);

/* delete timer (destructor) */
extern void snd_seq_timer_delete(seq_timer_t **tmr);



/* compare timestamp between events */
/* return 1 if a >= b; otherwise return 0 */
static inline int snd_seq_compare_tick_time(snd_seq_tick_time_t *a, snd_seq_tick_time_t *b)
{
	/* compare ticks */
	return (*a >= *b);
}

static inline int snd_seq_compare_real_time(snd_seq_real_time_t *a, snd_seq_real_time_t *b)
{
	/* compare real time */
	if (a->tv_sec > b->tv_sec)
		return 1;
	if ((a->tv_sec == b->tv_sec) && (a->tv_nsec >= b->tv_nsec))
		return 1;
	return 0;
}


static inline void snd_seq_sanity_real_time(snd_seq_real_time_t *tm)
{
	while (tm->tv_nsec >= 1000000000) {
		/* roll-over */
		tm->tv_nsec -= 1000000000;
                tm->tv_sec++;
        }
}


/* increment timestamp */
static inline void snd_seq_inc_real_time(snd_seq_real_time_t *tm, snd_seq_real_time_t *inc)
{
	tm->tv_sec  += inc->tv_sec;
	tm->tv_nsec += inc->tv_nsec;
	snd_seq_sanity_real_time(tm);
}

/* called by timer isr */
extern int snd_seq_timer_open(queue_t *q);
extern int snd_seq_timer_close(queue_t *q);
extern int snd_seq_timer_midi_open(queue_t *q);
extern int snd_seq_timer_midi_close(queue_t *q);
extern void snd_seq_timer_defaults(seq_timer_t *tmr);
extern void snd_seq_timer_reset(seq_timer_t *tmr);
extern void snd_seq_timer_stop(seq_timer_t *tmr);
extern void snd_seq_timer_start(seq_timer_t *tmr);
extern void snd_seq_timer_continue(seq_timer_t *tmr);
extern int snd_seq_timer_set_tempo(seq_timer_t *tmr, int tempo);
extern int snd_seq_timer_set_ppq(seq_timer_t *tmr, int ppq);
extern int snd_seq_timer_set_position_tick(seq_timer_t *tmr, snd_seq_tick_time_t position);
extern int snd_seq_timer_set_position_time(seq_timer_t *tmr, snd_seq_real_time_t position);
extern snd_seq_real_time_t snd_seq_timer_get_cur_time(seq_timer_t *tmr);
extern snd_seq_tick_time_t snd_seq_timer_get_cur_tick(seq_timer_t *tmr);

                        
#endif
