/***************************************************************************
                          kaspabase.h  -  description
                             -------------------
    begin                : Tue Sep 7 1999
    copyright            : (C) 1999 by Jan Mueller
    email                : janmueller7@hotmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef KASPABASE_H
#define KASPABASE_H
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include "kaspasql.h"
#include <stdio.h>


/** Exception class. Indicates that a deletion failed because the record or one of
	  the record linked is locked. */
class BaseDeleteLock: public KaspaErr {
 public:
  BaseDeleteLock(char *s=0): KaspaErr(s) {};
};


/** Base class for fields. A 'Field' is abstract but the derived classes are holding the
		information of one field read from the db. Field keeps track of changings.
 */
class Field {
 public:
 	Field();
  virtual ~Field();
  /** Abstract method. Sets data to default values. */
  virtual void empty()=0;
  /** Set changed state. */
  void setChanged();
  /** set unchanged state. */
  void setUnchanged();
  /** Returns state. */
  bool hasChanged() const;
 private:
  bool changed;	
};


/** A Data Field - that means a simple buffer... */
class DataField: public Field {
 public:	
  /** Copy constructor */
  DataField(DataField& T);
  /** Initializes data with 0. */
  DataField();
  /** Ciao... */
  ~DataField();

  DataField& operator=(const DataField& T);

  /** Set Data to 0. */
  virtual void empty();
  /** Returns length of buffer. */
  long length() const;
  /** Returns pointer to the internal buffer. */
  const unsigned char *get() const;
  /** Set data. The buffer is copied.*/
  void set(const unsigned char *buf, long l);
  /** Set Data. Takes the buffer. */
  void take(unsigned char *buf, long l);

private:
  unsigned char *d;
  long len;
  /** Deletes Data, if necessary, and set to 0. */
  void del();

};


/** A Text Field... */
class TextField: public Field {
 public:	
  /** Copy constructor */
  TextField(TextField& T);
  /** Initializes text with "". */
  TextField();
  /** Ciao... */
  virtual ~TextField();

	TextField& TextField::operator=(TextField &T);
  /** Set Text to "", one byte is reserved by new. */
  virtual void empty();
  /** Returns pointer to the internal string. */
  const char *get() const;
	/** Returns a string with "'" expanded to "\'". The string has to be deleted by the caller. */
	char *getSql() const;
	void getSql(Str &s) const;
  /** Set text. The text is copied.*/
  void set(const char *text);
	/** Appends a string... */
	void append(const char *text);
	/** Set text. The string is not copied, instead the pointer is taken. */
  void take(char *text);
 private:
  char *t;
  /** Deletes Text, if necessary (t!=0), and set to 0. */
  void del();
};

/** An Int Field. */
class IntField: public Field {
 public:	
  IntField(IntField& I);
  /** Initializes with 0. */
  IntField();
  /** set to 0 */
  virtual void empty();
	/** Returns the value as numeric. */
  int get() const;
	/** Returns the value as string. */
  void get(Str& s) const;
	/** Set value. */
  void set(int j);
	/** Set value. */
  void set(const char *s);
 private:
  int i;
};

/** An Bool Field. */
class BoolField: public Field {
 public:	

/*
	BoolField& operator=(BoolField& f);
  BoolField(BoolField& I); */
  /** Initializes with 0. */
  BoolField();
  /** set to 0 */
  virtual void empty();
	/** Adds "t" for true or "f" for false to the string. */
  void get(Str &s) const;
	/** Returns the value */
  bool get() const;
	/** Set value */
  void set(bool j);
	/** set value, "t" for true and "f" for false. */
  void set(const char *s);
 private:
  bool i;
};

/** An Oid (Object-ID) Field */
class OidField: public Field {
 public:	
//  OidField(OidField& O);
  /** Initializes with InvalidOid. */
  OidField();
  /** set to InvalidOid. */
  virtual void empty();
	/** returns the value */
  Oid get() const;
	/** Appends the value to the string. */
  void get(Str& s) const;
	/** set value */
  void set(Oid i);
	/** set value */
  void set(const char *s);
 private:
  Oid o;
};


class KaspaBase;

/** It is unnecessary to read the whole record for an overview.
		This class holds an title and the corresponding Oid and table.
		They are read-only.
 		The records are linked via "next", deleting the first record
		deletes the whole chain.
		There are derived classes which include other, table-specific
		information.
*/
class Title {
friend class KaspaBase;
  OidField  o;
	TextField title;
	TextField table;
protected:
	Title *next;
public:
	/** Returns the Oid of the row */
	Oid getOid() { return o.get(); }
	/** Returns the 'title' of the row. This may be for an author firstname+lastname */
	virtual const char *getTitle() const { return title.get(); };
	/** The table where the row is stored. */
	virtual const char *getType() const { return table.get(); };
	/** The next Title in the linked list */
  virtual Title *getNext() const { return next; };
	/** Appends a Title or a chain of Titles at the end of the linked list (follows the links) */
	void append(Title *t) {
		Title *m;
		for(m=this; m->getNext(); m=m->getNext());
		m->next=t;
	}

  /** Initializes next with 0; */
  Title() { next=0; }
  Title(Title &t) {
		next=0L;
		o.set(t.getOid());
		title.set(t.getTitle());
		table.set(t.getType());
	}
  /** Destructor. Deletes via next the whole chain. */
  virtual ~Title() { delete next; }
};

class PublTitle: public Title {
friend class KaspaBase;
	IntField	entrytype;
	TextField year;
public:
	int getEntryType() { return entrytype.get(); }
	const char *getYear() { return year.get(); }
  virtual PublTitle *getNext() { return (PublTitle *)next; }
};

class PartTitle: public Title {
friend class KaspaBase;
public:
  virtual PartTitle *getNext() { return (PartTitle *)next; }
};

class AuthorName: public Title {
	friend class KaspaBase;
	TextField firstname;
	TextField lastname;

public:
  virtual AuthorName *getNext() { return (AuthorName *)next; }
	const char *getFirstName() { return firstname.get(); }
	const char *getLastName() { return lastname.get(); }
};

class PartDataName: public Title {
friend class KaspaBase;
	TextField filename;
public:
  virtual PartDataName *getNext() { return (PartDataName *)next; }
	const char *getFileName() { return filename.get(); }
};

class PublisherName: public Title {
friend class KaspaBase;
	TextField address;
public:
  virtual PublisherName *getNext() { return (PublisherName *)next; }
	const char *getAddress() { return address.get(); }
};

class JournalName: public Title {
friend class KaspaBase;
	TextField address;
public:
  virtual JournalName *getNext() { return (JournalName *)next; }
	const char *getAddress() { return address.get(); }
};

class NoteTitle: public Title {
friend class KaspaBase;
	OidField publno;
  OidField parent;
public:
  virtual NoteTitle *getNext() { return (NoteTitle *)next; }
	Oid getPublNo() { return publno.get(); }
  Oid getParent() { return parent.get(); }
};



/** This is the abstract base class for the classes holding the rows of the tables.
		It's a linked list.  If a record is deleted, all linked records are deleted, too.
*/
struct KaspaRec {
  OidField id;
  KaspaRec *next;

  /** Constructs new record. next=0.
   */
  KaspaRec():next(0) { }
  virtual ~KaspaRec() {}

  /* All changed-flags are set false. */
  virtual void unchanged()=0;
	/* Initilizes all fields with their standart(empty)-values. */
  virtual void empty()=0;
};

/** Container for a publication.
*/
struct PublRec: public KaspaRec {
  TextField title;	        TextField subtitle;       IntField  edition;				TextField bibtex;
  TextField editor;         TextField howpublished;   TextField organization;   OidField  publisher;
  TextField year;           TextField pages;          TextField translator;     TextField volume;
  TextField number;         TextField month;          TextField series;         TextField type;
  TextField key;            TextField orgtitle;       OidField  orgpublisher;		TextField orgyear;
  TextField isbn_issn;      TextField hidingplace;    TextField memo;           IntField  entrytype;
	BoolField createbibtex;   TextField created;        TextField modified;

  /** The strings are initialized with @ref KaspaRec::emptyStr,
      Oids with InvalidOid, numbers with 0.
  */
  virtual void empty() {
    title.empty();  	    	subtitle.empty();  	  edition.empty();
    editor.empty();      		howpublished.empty(); organization.empty();
    publisher.empty();			year.empty();        	pages.empty();
    translator.empty();  		volume.empty();      	number.empty();
    month.empty();      		series.empty();      	type.empty();
    key.empty();        		orgtitle.empty();    	orgpublisher.empty();
    orgyear.empty();   			isbn_issn.empty();   	hidingplace.empty();
    memo.empty();       		entrytype.empty();    bibtex.empty();
		createbibtex.empty();		created.empty();			modified.empty();
  }

  /** All fields are marked unchanged. */
  virtual void unchanged() {
    title.setUnchanged();  	    	subtitle.setUnchanged();   		edition.setUnchanged(); 	
    publisher.setUnchanged();   	year.setUnchanged();       		pages.setUnchanged();  	
    month.setUnchanged();       	series.setUnchanged();     		type.setUnchanged();    	
    orgyear.setUnchanged();     	isbn_issn.setUnchanged();	 		hidingplace.setUnchanged();
    editor.setUnchanged();      	howpublished.setUnchanged();  organization.setUnchanged();
    translator.setUnchanged();  	volume.setUnchanged();      	number.setUnchanged();	  	
    key.setUnchanged();         	orgtitle.setUnchanged();     	memo.setUnchanged();		
    entrytype.setUnchanged();   	orgpublisher.setUnchanged();  bibtex.setUnchanged();
		createbibtex.setUnchanged();	modified.setUnchanged();			created.setUnchanged();
  }
};


/** Container for an author.
*/
struct AuthorRec: public KaspaRec {
  TextField firstname;  TextField lastname;
  TextField born;       TextField died;
  TextField pseudonym;  TextField memo;
  TextField country;    DataField picture;
	TextField modified;		TextField created;

  virtual void empty() {
    firstname.empty();	lastname.empty(); pseudonym.empty();
    country.empty();    born.empty();    	died.empty();
    memo.empty();    	  picture.empty();	modified.empty();
		created.empty();
  }

  virtual void unchanged() {
    lastname.setUnchanged();    firstname.setUnchanged();   country.setUnchanged();
    pseudonym.setUnchanged();   picture.setUnchanged();     born.setUnchanged();
    died.setUnchanged();        memo.setUnchanged();				modified.setUnchanged();
		created.setUnchanged();
  }
};


/** Containerclass for notes */
struct NoteRec: public KaspaRec {

  TextField title;  	IntField  type;
  TextField memo;   	OidField  publno;
	TextField modified;	TextField created;

  virtual void empty() {
    title.empty(); 		type.empty(); 		memo.empty(); publno.empty();
		modified.empty();	created.empty();
  }

  virtual void unchanged() {
    title.setUnchanged();  		type.setUnchanged();
    publno.setUnchanged(); 		memo.setUnchanged();
		modified.setUnchanged();	created.setUnchanged();
  }
};


/** Containerclass for parts */
struct PartRec: public KaspaRec {
  TextField title;  				IntField  partno;	  TextField memo;
  OidField  publno; 				TextField pages;  	OidField  intro;
  TextField key;						TextField language; TextField bibtex;
	BoolField  createbibtex; 	TextField modified;	TextField created;


  virtual void empty() {
    title.empty();  			memo.empty();   	pages.empty();
    key.empty();     			partno.empty();  	publno.empty();
    intro.empty();    		language.empty(); bibtex.empty();
		createbibtex.empty(); created.empty();	modified.empty();
  }

  virtual void unchanged() {
    title.setUnchanged(); 				partno.setUnchanged();  	publno.setUnchanged();
    intro.setUnchanged(); 			 	memo.setUnchanged();    	key.setUnchanged();
    pages.setUnchanged();  				language.setUnchanged(); 	bibtex.setUnchanged();
		createbibtex.setUnchanged(); 	modified.setUnchanged();	created.setUnchanged();
  }
};


/** Containerclass for partdatas */
struct PartDataRec: public KaspaRec {
  TextField filename;  TextField file;
  TextField type;      OidField  partno;
	TextField modified;	 TextField created;
	TextField text;      TextField memo;	

  virtual void empty() {
    filename.empty();  file.empty();
    type.empty();      partno.empty();
		created.empty();	 modified.empty();
    text.empty();      memo.empty();
  }

  virtual void unchanged() {
    filename.setUnchanged();  file.setUnchanged();
    partno.setUnchanged();    type.setUnchanged();
    text.setUnchanged();      memo.setUnchanged();
  }
};

/** Containerclass for publisher */
struct PublisherRec: public KaspaRec {
  TextField name;  		TextField city; 	TextField serie;
	TextField modified;	TextField created;

  virtual void empty() {
    name.empty(); city.empty(); serie.empty();
		created.empty(); modified.empty();
  }

  virtual void unchanged() {
    name.setUnchanged();
    city.setUnchanged();
		serie.setUnchanged();
		modified.setUnchanged();
		created.setUnchanged();
  }
};

struct LockTableItem {
	Oid id;
	LockTableItem *next;
};


/** Supports read/writing/ins./del. of records, asyncronous querys, locking,
    hides LO-Interface.
    There are four methods for each table. E.g. table author:

    @ref getAuthor returns authors matching a sql where clause (e.g. "where name='foo').
    The returned authors have the type @ref AuthorRec. AuthorRec consists of various fields:
    @ref TextField, @ref IntField, @ref OidField, @ref DataField.
    The field "name" is from type TextField. The name can be accessed or (un)marked as
    changed. Toggeling the changed state has no effect on the id field which
    identifies the tuple in the DBMS.
    The records are chained. If one record is deleted, the destructor deletes
    all following records, too.

    @ref insertAuthor inserts a new empty tuple and returns the id of the tuple.

    @ref deleteAuthor deletes tuples matching a sql where clause.

    @ref updateAuthor takes a sql where clause and takes one @ref AuthorRec.
    The tuples matching the where clause are updated with the fields marked as changed.
    These fields are marked as unchanged again.

    An overview of some tables is supported:
    @ref getAuthorNames reads the id's, the firstnames and lastnames of the authors
    matching a sql where clause. The authors are stored in a record of type @ref AuthorName.

    @ref lockObj locks an object (e.g. a tuple or a LO). The object is unlocked by the

    @ref unlockObj method. Locking means cooperative locking and
    is based on Massimo Dal Zotto's userlock-mechanism. It can be found in the
    contrib-directory of postgresql.

    @ref setCallback sets a function, which is called while a query is
    processed by the backend.

    @ref isWorking determines, if a query is pending.

    The underlying Class @ref Sql uses execpetions. Therefore the methods defined in this
    class could throw more or less all exceptions defined in @ref Sql.
*/
class KaspaBase : protected Sql {
 private:
  KaspaBase(const KaspaBase&);
	KaspaBase& operator=(const KaspaBase&);	

	static LockTableItem *locktable;

  /** Reads an oid from a field in a table matching where clause.
      The read oid is unlinked.
  */
  void deleteLo(const char *table, const char *field, const char *where);

  /** Reads an large object o into a string. Last character will be '0'. Returns never null.
			If there is no LO an empty string is return.*/
  char *lo2str(const Oid o);

  /** Reads an large object o into a buffer. The buffer is terminated by a zero.
			*len takes the length of the buffer.
  */
  char *lo2buf(const Oid o, long *len);

  /** Takes a string s containing an oid, returns the oid of the created LO. */
  Oid str2lo(const char *s);

  /** Writes a buffer buf of length len to a large object. Returns the oid. */
  Oid buf2lo(const char *buf, const long len);

  /** A large object is read into a @ref DataField <d>. */
  void lo2buf(const Oid o, DataField& d);

  /** Deletes Tuples in a table matching where clause. */
  void deleteTuples(const char *table, const char *where);

  /** Class @ref Sql (KaspaBase is derived from @ref Sql) calls idle()
      while a query is pending.
      This implementation calls the callback function set by @ref setCallBack.
      If the callback function was not set, idle() returns by default TRUE.
  */

  /** An internal method used by @ref getLinks. */
  void buildGetLinkClause(const char *table, Oid obj, Str& s);

	/**	Gets the time in postges timestamp format. */
	const char *getTime();
 public:
//	void cancel() { Sql::cancel(); }
//  virtual bool idle() { if(callback) return callback(); else return TRUE;}
  virtual bool idle() throw() { return true; }

  /** Uses an established connection to initialize. */
  KaspaBase(Sql *conn);

  /** Opens database with name dbname. */
  KaspaBase(const char *dbname);

  /** Closes connection. */
  virtual ~KaspaBase() throw();

  /** Locks an object. Objects can be tuples or large objects.
      The method returns TRUE on success.
      Locking means cooperative locking, nothing prevents other
      processes to write on the object.
      The methods take the object id o. It is possible to lock the
      same object by the same process
      several times. @ref unlockObj has to be called as many times to unlock again.
      If the process ends/dies, the lock is removed automatically.
  */
  bool lockObj(Oid o);

  /** Unlocks an object with id o. Returns TRUE on success, FALSE
      if the object was not locked by the process.
  */
  bool unlockObj(Oid o);

	bool isLocked(const char *table, const char *where);
	bool isLocked(Oid o);

  /* Returns TRUE, if a query is processed. */
  virtual bool isWorking() { return Sql::isWorking(); }

  Oid getNo(Str tab, Oid o);

  void movePublAuthorUp(Oid author, Oid publ);
  void movePublAuthorDown(Oid author, Oid publ);
  void movePartAuthorUp(Oid author, Oid publ);
  void movePartAuthorDown(Oid author, Oid publ);

  void renumber_publauthors(Oid publ);
  void renumber_partauthors(Oid publ);


  /** Creates new Author. Return Oid of the new author. */
  Oid newAuthor();

  /** Deletes Author matching where clause. */
  void deleteAuthor(Oid author);

  /** Reads one author from database matching SQL-WHERE clause where
      The returned list has to be deleted by the caller. @see AuthorRec
  */
  AuthorRec *getAuthor(const char *where);

  /** Updates author matching sql where clause.
      Only the fields with the changed state are updated.
      These fields are marked as unchanged again.
      @see AuthorRec
  */
  void updateAuthor(AuthorRec *author, const char *where, bool block=false);

  /** Same as above, @see newAuthor @see PublRec */
  Oid  newPubl();
  void deletePubl(Oid o);
  void deletePubl(const char *where);
  PublRec *getPubl(const char *where);
  void updatePubl(PublRec *publ, const char *where, bool block=false);

  /** Same as above. @see newAuthor @see PartRec */
  Oid  newPart(Oid publ);
  void deletePart(const char *where);
  PartRec *getPart(const char *where);
  void updatePart(PartRec *part, const char *where, bool block=false);

  /** Same as above, @see newAuthor @see PartDataRec */
  Oid  newPartData(Oid part);
  void deletePartData(const char *where);
  PartDataRec *getPartData(const char *where);
  void updatePartData(PartDataRec *partdata, const char *where, bool block=false);

  /** Same as above, @see newAuthor @see PartDataRec */
  Oid  newPublisher();
  void deletePublisher(const char *where);
  PublisherRec *getPublisher(const char *where);
  void updatePublisher(PublisherRec *publisher, const char *where, bool block=false);

  /** Same as above, @see newAuthor @see NoteRec */
  Oid newNote(Oid o=InvalidOid);
  void deleteNote(const char *where);
  NoteRec *getNote(const char *where);
  void updateNote(NoteRec *note, const char *note, bool block=false);
  void moveNote(Oid id, Oid parent, Oid sibling);

  /** Inserts a new connection between an author with id <author>
      and a publication with id publ. */
  void newPublAuthor(Oid publ, Oid author);

  /** Deletes connection matching where. */
  void deletePublAuthor(Oid publ, Oid author);
  void deletePublAuthor(const char *where);

  /** Inserts a new connection between an author with id author
      and a part with id part. */
  void newPartAuthor(Oid part, Oid author);

  /** Deletes connection matching where. */
  void deletePartAuthor(const char *where);
  void deletePartAuthor(Oid part, Oid author);

  /** Inserts a new link (between a note and an object, e.g. authors, publications,
      parts, partdata, other notes. The note has the id note, the object has the id obj
      and is located in table.
  */		
  void newLink(Oid note, Oid obj, const char *table);

  /** Deletes a link between a note and an object. The link matches where. */
  void deleteLink(const char *where);

  /** Get all objects associated with an object via a link.
      The method returns an @ref Title.It uses @ref getPublTitles, @ref getAuthorNames,
      @ref getPartTitles, @ref getPartDataTitles and @ref getNoteTitles to set up
      the returned linked list. */
  Title *getLinks(Oid obj);

 public:
  /** Returns a list with publications (o=oid, s1=name and s2=type of publ.)
      The publications match the sql where clause where.
      The caller has to delete the returned linked list of type @ref PublTitle.
  */
  PublTitle *getPublTitles(const char *where);

  /** Returns a list with the name and the prename of an author
      (o=oid, s1=prename and s2=name.)
      @see AuthorName
  */
  AuthorName *getAuthorNames(const char *where);

  /** Returns notes. o=Oid of a note, s1=title of a note, s2="".
      @see NoteTitle
  */
  NoteTitle *getNoteTitles(const char *where);

  /** Returns notes. o=Oid, s1=title, s2=publicationno. */
//  OssRec *getNotePublNo(const char *where);

  /** Returns the titles parts (o=oid, s1=title, s2="") */
  PartTitle *getPartTitles(const char *where, bool fullpath=false);

  /** Returns the file names of the files stored in the partdata table.
      (o=oid, s1=filename, s2="")
  */
  PartDataName *getPartDataNames(const char *where, bool fullpath=false);
  /** Returns the names of the publishers stored in the publisher table.
  		(o=oid, s1=name, s2="")
	*/
	PublisherName *getPublisherNames(const char *where);
	
	JournalName *getJournalNames(const char *where);
	
  /**  */
//	void search(const char *where, AuthorName **an, PartTitle **pt,
//				PublTitle **pbt, NoteTitle **nt, PartDataName **pdt);
  /**  */
  int bibTexIndex(Oid author, const char *year, Oid publ);
  /**  */
  bool createBibTex(const char *filename);
};

#endif











