/***************************************************************************
                          cmudview.h
                      -------------------
    description		 : class providing a colored output view
    begin		 : Wed Sep 8 1999
    copyright            : (C) 1999 by Kmud Developer Team
    email                : kmud-devel@kmud.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   * 
 *                                                                         *
 ***************************************************************************/


#ifndef CMUDVIEW_H
#define CMUDVIEW_H

#include <qtableview.h>
#include <qlist.h>
#include <qpainter.h>
#include <qkeycode.h>
#include <qfontmetrics.h>
#include "kmuddoc.h"
#include "ansi.h"

/** A chunk of text with defined color/font settings */
struct MudViewChunkData
{
	QString s;
	QColor fgc;
	QColor bgc;
	QFont font;
};

/** Struct handling one line of text consisting of a list of MudViewChunkData's */
struct MudViewLineData
{
	QList<MudViewChunkData> chunks;
	int lineHeight;
};

/**
  * Class providing a colored output view.
  * Interprets ansi/vt100 escape sequences.
  *@author Stephan Uhlmann, Andre Alexander Bell
  */

class CMudView : public QTableView  {

	Q_OBJECT

public: 
	/** constructor */
	CMudView(KmudDoc* document, QWidget * parent=0, const char * name=0);
	/** destructor */
	~CMudView();

	/** returns the length of a line in pixels */
	int lineLength(int line);
	/** returns the total number of lines */
	int numLines();
	/** appends a string with a specified color/background/font to the view.
			automagically interprets ansi/vt100 escape sequences and adapts the color/font settings
			returns the incoming string but without ANSI codes etc.
		*/
	QString append(QString s, QColor fgc, QColor bgc, QFont font);

	/** sets the default foreground color to use */
	void setForeground(QColor fgc);
	/** sets the default background color to use */
	void setBackground(QColor bgc);
	/** sets the actual background color to use */
	void setActualBackground(QColor bgc);
	/** sets the default font to use */
	void setFont(QFont font);

	/** Sets the interpretation type */
	void setInterpretType(int i);
	
	// sets/gets debug highlight color
	void setDebugHighlightColor(QColor color);
	QColor getDebugHighlightColor();
	
	/** sets the colors for ANSI */
	void setOutputANSIColor(int ANSIColor, QColor color);

	/** sets background colors */
	void setOutputBgColor(int BgColor, QColor color);
	
	/** gets the default foreground color to use */
	QColor getForeground();
	/** gets the default background color to use */
	QColor getBackground();
	/** gets the default font to use */
	QFont getFont();

	int getInterpretType();
	
	/** sets the colors for ANSI */
	QColor getOutputANSIColor(int ANSIColor);
	
	/** gets colors for background */
	QColor getOutputBgColor(int BgColor);
	
	/** sets maximum number of lines in the view */
	void setMaxLines(int);

	/** sets useBeep -- use \007 */
	void setUseBeep(bool);
	/** gets useBeep -- use \007 */
	bool getUseBeep();

	/** returns the size in pixels of the vertical scrollbar */
	int getVScrollBarWidth();
	/** enables/disables the vertical scrollbar  */
	void setVScrollBarVisible(bool);
	/** copies the marked text in the X-Clipboard */
	void copyMarkedText ();
	/** returns the Character at the Position (x, y) */
	virtual char getCharacterAt (int row, int col);
	/** returns the length of the specified row */
	virtual int getLineLength (int row);

	
signals:
	/** signal emitted when text in the widget chaged */
	void textChanged();
	void vSliderAtBottomChanged(bool);

protected:
	/** paints the cell (line) in the specified row/column
	  * using the given QPainter
	  */
	void paintCell(QPainter*, int row, int col );
	/** return the height of the specified row in pixels */
	int cellHeight(int row);
	/** return the width of the specified column in pixels */
	int cellWidth(int col);
	/** redefined resizeEvent to set the cellWidth to the total width */
	virtual void resizeEvent(QResizeEvent *);
	/** overwrites the mousePressEvent-Method from QWidget
	implements the start of marking a text-area */
	virtual void mousePressEvent (QMouseEvent* e);
	/** overwrites the mouseMoveEvent from QWidget
	implements the marking of a area */
	virtual void mouseMoveEvent (QMouseEvent* e);
	/** overwrites the mouseDoubleClickEvent inherited from QWidget
	marks the mord the mouse points to */
	virtual void mouseDoubleClickEvent (QMouseEvent* e);


private:
	/** initializes the view, called in constructor and when clearscreen
	escape sequence was received */
	void init();
	/** gets the number of the character in the line row and on the position xPos */
	int getCharacterNumber (int Row, int xPos);
	/** returns the text in the marked area */
	QString getMarkedText ();
	/** returns true, if the specified row is marked */
	int rowMarked (int Row);
	/** Flush a chunk so we can start working on a new one. */
	void flushChunk(QString &t, QColor foreground, QColor background, QFont font);

	QList< MudViewLineData > data;

	QColor defaultForeground;
	QColor defaultBackground;
	QFont defaultFont;

	int interpretType;

	QColor DEBUG_HIGHLIGHT;
	
	QColor FANSIColor[ANSICOLOR_COUNT];
	
	QColor FBgColor[BGCOLOR_COUNT];
	
	QColor actualForeground;
	QColor actualBackground;
	QFont actualFont;

	int maxLines;
	/** starting column of the marked area */
	int markedAreaStartCol;
	/** column of the ending of the marked area */
	int markedAreaEndCol;
	/** row of the end of the marked area */
	int markedAreaEndRow;
	/** starting row of the marked area */
	int markedAreaStartRow;
	int lineX;

	bool useDebugHighlight;

	/** true when vertical slider at bottom */
	bool vSliderAtBottom;

	/** the Document */
	KmudDoc* doc;

	/** To take advantage of the \007 char */
	bool useBeep;
	/** is set, if some text is marked */
	bool markedText;

  	// variables for wrapping output.
  	QFontMetrics *fontMetric;
	QWidget *myParent;
	QString oldLine;
	
	int currentMaxWidth;
    	
public slots:
	/** Scrolls around when certain keys are pressed in CMudLineEdit. */
	void slotScrollKey(QKeyEvent *);

	void slotVSliderMoved(int);
};

#endif
