/*
 This file is part of GNU Taler
 (C) 2021-2023 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

import { useTranslationContext } from "@gnu-taler/web-util/browser";
import { Fragment, h, VNode } from "preact";
import { useState } from "preact/hooks";
import {
  useEntityContext,
  useEntityDataContext,
} from "../../context/entity.js";

interface Props {
  onSuppress: (id: any) => void;
}

export function CardTable({ onSuppress }: Props): any {
  const data = useEntityDataContext();
  const [rowSelection, rowSelectionHandler] = useState<string | undefined>(
    undefined,
  );
  const { i18n } = useTranslationContext();
  const { title, endpoint, entity } = useEntityContext();

  return (
    <div class="card has-table">
      <header class="card-header">
        <p class="card-header-title">
          <span class="icon">
            <i class="mdi mdi-shopping" />
          </span>
          <i18n.Translate>{title}</i18n.Translate>
        </p>
        <div class="card-header-icon" aria-label="more options"></div>
      </header>
      <div class="card-content">
        <div class="b-table has-pagination">
          <div class="table-wrapper has-mobile-cards">
            {data.data[0][endpoint] !== undefined &&
            data.data[0][endpoint].length != 0 ? (
              <Table data={data.data[0][endpoint]} onSuppress={onSuppress} />
            ) : (
              <EmptyTable />
            )}
          </div>
        </div>
      </div>
    </div>
  );
}

interface TableProps {
  data: any;
  onSuppress: (id: any) => void;
}

function Table({ data, onSuppress }: TableProps): VNode {
  const { i18n } = useTranslationContext();
  const { entity } = useEntityContext();
  type Entity = typeof entity;
  let count = 0;

  return (
    <div class="table-container">
      <table class="table is-fullwidth is-striped is-hoverable is-fullwidth">
        <thead>
          <tr>
            {Object.keys(data[0]).map((i: Entity) => {
              const paramName =
                i[0].toUpperCase() + i.replace("_", " ").slice(1, i.count);
              return (
                <Fragment key={count.toString() + i}>
                  <th>
                    <i18n.Translate>{paramName}</i18n.Translate>
                  </th>
                </Fragment>
              );
            })}
          </tr>
        </thead>
        <tbody>
          {data.map((key: Entity, value: string) => {
            return (
              <tr>
                {Object.keys(data[0]).map((i: Entity) => {
                  return (
                    <Fragment>
                      <td>{key[i] == false ? "false" : key[i]}</td>
                    </Fragment>
                  );
                })}
                <td class="is-actions-cell right-sticky">
                  <div class="buttons is-right">
                    <span
                      class="has-tooltip-bottom"
                      data-tooltip={i18n.str`suppress`}
                    >
                      <button
                        class="button is-small is-success "
                        type="button"
                        onClick={(): void => onSuppress(key["row_id"])}
                      >
                        {<i18n.Translate>Suppress</i18n.Translate>}
                      </button>
                    </span>
                  </div>
                </td>
              </tr>
            );
          })}
        </tbody>
      </table>
    </div>
  );
}

function EmptyTable(): VNode {
  const { i18n } = useTranslationContext();
  return (
    <div class="content has-text-grey has-text-centered">
      <p>
        <span class="icon is-large">
          <i class="mdi mdi-emoticon-happy mdi-48px" />
        </span>
      </p>
      <p>
        <i18n.Translate>There are no entries yet</i18n.Translate>
      </p>
    </div>
  );
}
