/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Author: Charles Kerr <charles@rebelbase.com>
 *
 * Copyright (C) 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <pan/base/debug.h>

#include <pan/filters/filter-manager.h>
#include <pan/filters/filter-top.h>

#include <pan/articlelist.h>
#include <pan/article-list-toolbar-mediator.h>
#include <pan/article-toolbar.h>

/**
 * article-toolbar has information necessary for knowing how the articles
 * in the articlelist should be filtered, so each time a group is set in
 * articlelist this module updates the toolbar's filter.  The result is
 * fired to here, and then forwarded to articlelist_set_filter().
 *
 * When a group is being loaded then the articlelist is already being built,
 * so to prevent the list from being built twice we have this flag that
 * prevents a rebuild when the toolbar's filter change is triggered by
 * an already-updating articlelist.
 */
static gboolean rebuild = TRUE;

static gint
article_toolbar_changed_cb (gpointer call_obj, gpointer call_arg, gpointer data)
{
	Group * group;
	ArticleToolbarCallbackStruct * cbs = (ArticleToolbarCallbackStruct *) call_obj;
	debug_enter ("article_toolbar_changed_cb");

	/* update the articlelist */
	articlelist_set_filter (cbs->filter, cbs->filter_show, rebuild);
	rebuild = TRUE;

	/* update the group's filter stats for posterity */
	group = articlelist_get_group ();
	if (group != NULL) {
		const gchar * filter_name = NULL;
		if (cbs->filter!=NULL && IS_FILTER_TOP(cbs->filter))
			filter_name = FILTER_TOP(cbs->filter)->name;
		group_set_filter (group, cbs->filter_bits, cbs->filter_show, filter_name);
	}

	debug_exit ("article_toolbar_changed_cb");
	return 0;
}

static gint
articlelist_group_changed_cb (gpointer call_obj, gpointer call_arg, gpointer data)
{
	const Group * group;
	guint bits = ~0;
	gulong show = FILTER_SHOW_MATCHES_AND_REFERENCES;
	const gchar * name = NULL;
	debug_enter ("articlelist_group_changed_cb");

	/* determine which name, bits, & show to use */
	group = GROUP(call_arg);
	if (group != NULL) {
		bits = group->filter_bits;
		show = group->filter_show;
		name = group->filter_name;
	}

	/* notify the toolbar to update */
	rebuild = FALSE;
	article_toolbar_set_filter (name, bits, show);

	debug_exit ("articlelist_group_changed_cb");
	return 0;
}

void
article_list_toolbar_mediator_init (void)
{
	pan_callback_add (article_toolbar_get_filter_changed_callback(),
	                  article_toolbar_changed_cb,
	                  NULL);
	pan_callback_add (articlelist_get_group_changed_callback(),
	                  articlelist_group_changed_cb,
	                  NULL);

}
