// -*- Mode: C++; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 4 -*-
/*
 *  ORBit-C++: C++ bindings for ORBit.
 *
 *  Copyright (C) 1999 Phil Dawes
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Authors: Phil Dawes <philipd@users.sourceforge.net>
 *           Andreas Kloeckner <ak@ixion.net>
 *           Gerg rdi <cactus@cactus.rulez.org>
 */

#ifndef ORBITCPP_COMPOUND_SEQUENCE_HH
#define ORBITCPP_COMPOUND_SEQUENCE_HH

#include "orbitcpp/orb-cpp/orbitcpp_types.h"
#include "orbitcpp/orb-cpp/orbitcpp_exception.h"
#include "orbitcpp/orb-cpp/orbitcpp_sequence.h"
#include <orbit/orb-core/corba-defs.h>
#include <glib.h> // for g_assert

namespace _orbitcpp {

    template <class CPPElem, class CElem, class CElemSeq, CORBA_TypeCode seq_typecode>
    struct seq_seq_traits : public simple_seq_traits<CPPElem, CElem, CElemSeq, seq_typecode> {
	typedef CPPElem value_t;
	typedef CElem c_value_t;

	static void pack_elem (const value_t &cpp_elem, c_value_t &c_elem)
	{
	    cpp_elem._orbitcpp_pack(c_elem);
	}
        static void unpack_elem (value_t &cpp_elem, const c_value_t &c_elem)
        {
	    cpp_elem._orbitcpp_unpack(c_elem);
        }
    };

    template <class CPPType, class CPPElem, class CElem, class CElemSeq, CORBA_TypeCode seq_typecode>
    struct type_seq_traits : public simple_seq_traits<CPPElem, CElem, CElemSeq, seq_typecode> {
	typedef CPPElem value_t;
	typedef CElem c_value_t;

	static void pack_elem (const value_t &cpp_elem, c_value_t &c_elem)
	{
		CPPType::pack_elem(cpp_elem, c_elem);

	}
        static void unpack_elem (value_t &cpp_elem, const c_value_t &c_elem)
        {
		CPPType::unpack_elem(cpp_elem, c_elem);
        }
    };

    template<typename CPPElem, typename SeqTraits>
    class CompoundSeqBase: public SequenceBase<typename SeqTraits::value_t,
	typename SeqTraits::c_value_t,
        typename SeqTraits::c_seq_t>
    {
    public: // public requirement may be a GCC 3.3 bug
	typedef SeqTraits  traits_t;
	typedef typename traits_t::value_t   value_t;
	typedef typename traits_t::c_value_t c_value_t;
	typedef typename traits_t::c_seq_t   c_seq_t;
	
	typedef SequenceBase<value_t, c_value_t, c_seq_t> Super;
	typedef Super super_t;
	
	typedef CompoundSeqBase<CPPElem, SeqTraits> self_t;

	typedef typename Super::size_t   size_t;
	typedef typename Super::index_t  index_t;
	typedef typename Super::buffer_t buffer_t;
	
    public:
	// Empty constructor
	explicit CompoundSeqBase (size_t max = 0):
	    Super (max)
	    {
	    }
	
	// Create sequence from flat buffer
	CompoundSeqBase (size_t max,
			 size_t length, buffer_t buffer,
			 CORBA::Boolean release = false):
	    Super (max, length, buffer, release)
	    {
	    }
		
	// Copying
	CompoundSeqBase (const self_t &other):
	    Super (other)
	    {
	    }
	
    protected:
        c_seq_t* alloc_c () const {
	    return traits_t::alloc_c ();
	}
	
	c_value_t* alloc_c_buf (size_t length) const {
	    return traits_t::alloc_c_buf (length);
	}

	void pack_elem (const value_t &cpp_value, c_value_t &c_value) const {
	    traits_t::pack_elem (cpp_value, c_value);
	}
	
	void unpack_elem (value_t &cpp_value, const c_value_t &c_value) const {
	    traits_t::unpack_elem (cpp_value, c_value);
	}
    };
    
    template<typename CPPElem, typename SeqTraits=typename CPPElem::SeqTraits>
    class CompoundUnboundedSeq: public CompoundSeqBase<CPPElem, SeqTraits>
    {
    public: // public requirement may be a GCC 3.3 bug
	typedef SeqTraits  traits_t;
	typedef typename traits_t::c_value_t c_value_t;
	typedef typename traits_t::c_seq_t   c_seq_t;
	
	typedef CompoundSeqBase<CPPElem, SeqTraits> Super;
	typedef typename Super::value_t value_t;
	typedef SequenceBase<value_t, c_value_t, c_seq_t> super_t;
	
	
	typedef typename Super::size_t   size_t;
	typedef typename Super::index_t  index_t;
	typedef typename Super::buffer_t buffer_t;
	
    public:
	// Empty constructor
	explicit CompoundUnboundedSeq (size_t max = 0):
	    Super (max)
	    {
	    }
	
	// Create sequence from flat buffer
	CompoundUnboundedSeq (size_t max,
			 size_t length, buffer_t buffer,
			 CORBA::Boolean release = false):
	    Super (max, length, buffer, release)
	    {
	    }
		
	// Copying
	CompoundUnboundedSeq (const super_t &other):
	    Super (other)
	    {
	    }
	
	// Size requisition
	void _length_set_impl (size_t new_length) {
	    if (new_length > _max)
	    {
		buffer_t buffer_tmp = allocbuf (new_length);
		if (!buffer_tmp)
		    throw CORBA::NO_MEMORY ();
		
		for (index_t i = 0; i < _length; i++)
		    buffer_tmp[i] = _buffer[i];
		
		if (_release)
		    freebuf (_buffer);
		
		_release = true;
		_buffer = buffer_tmp;
		_max = new_length;
	    }
	    
	    _length = new_length;
	}
    };
    
    template<typename CPPElem, CORBA::ULong max, typename SeqTraits=typename CPPElem::SeqTraits>
    class CompoundBoundedSeq: public CompoundSeqBase<CPPElem, SeqTraits>
    {
    public: // public requirement may be a GCC 3.3 bug
	typedef SeqTraits traits_t;
	typedef typename traits_t::value_t  value_t;
	typedef typename traits_t::c_value_t c_value_t;
	typedef typename traits_t::c_seq_t  c_seq_t;
	
	typedef CompoundSeqBase<value_t, SeqTraits> Super;
	typedef Super super_t;
	
	typedef typename Super::size_t   size_t;
	typedef typename Super::index_t  index_t;
	typedef typename Super::buffer_t buffer_t;
	
    public:
	// Empty constructor
	CompoundBoundedSeq ():
	    Super (max)
	    {
	    }
		
	// Create sequence from flat buffer
	CompoundBoundedSeq (size_t length, buffer_t buffer,
			    CORBA::Boolean release = false):
	    Super (max, length, buffer, release)
	    {
	    }
		
	// Copying
	CompoundBoundedSeq (const super_t &other):
	    Super (other)
	    {
	    }
	
	// Size requisition
	void _length_set_impl (size_t new_length) {
	    g_assert (new_length <= _max);
	    
	    _length = new_length;
	}
    };
} // namespace _orbitcpp

#endif // !ORBITCPP_COMPOUND_SEQUENCE_HH

