/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/* 
 * Copyright (C) 2002-2004 Bastien Nocera <hadess@hadess.net>
 *
 * cd-recorder.c
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors: Bastien Nocera <hadess@hadess.net>
 */

#include "config.h"

#include <glib.h>
#include <glib/gi18n.h>

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <math.h>
#include <fcntl.h>
#include <sys/wait.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#ifdef __FreeBSD__
#include <sys/uio.h>
#endif /* __FreeBSD__ */
#include <signal.h>

#ifdef USE_HAL
#include <libhal.h>
#endif

#include "nautilus-burn-recorder.h"
#include "nautilus-burn-recorder-marshal.h"

struct NautilusBurnRecorderPrivate {
	GMainLoop     *loop;
	int            result;
	int            pid;
	int            cdr_stdin;
	GString       *line;
	GString       *cdr_stderr;
	gboolean       changed_text;
	gboolean       expect_cdrecord_to_die;
	gboolean       dangerous;

	char          *last_error;

	GList         *tracks;
	int            track_count;
	gboolean       debug;
	gboolean       can_rewrite;

#ifdef USE_HAL
	LibHalContext *ctx;
#endif
};

/* Signals */
enum {
	PROGRESS_CHANGED,
	ACTION_CHANGED,
	ANIMATION_CHANGED,
	INSERT_MEDIA_REQUEST,
	WARN_DATA_LOSS,
	LAST_SIGNAL
};

static int nautilus_burn_recorder_table_signals [LAST_SIGNAL] = { 0 };

static GObjectClass *parent_class = NULL;

static void nautilus_burn_recorder_class_init      (NautilusBurnRecorderClass     *class);
static void nautilus_burn_recorder_init            (NautilusBurnRecorder          *recorder);

static int  nautilus_burn_recorder_write_cdrecord  (NautilusBurnRecorder          *recorder,
						    NautilusBurnDrive             *drive,
						    GList                         *tracks,
						    int                            speed,
						    NautilusBurnRecorderWriteFlags flags);
static int  nautilus_burn_recorder_write_growisofs (NautilusBurnRecorder          *recorder,
						    NautilusBurnDrive             *drive,
						    GList                         *tracks,
						    int                            speed,
						    NautilusBurnRecorderWriteFlags flags);

G_DEFINE_TYPE(NautilusBurnRecorder, nautilus_burn_recorder, G_TYPE_OBJECT);

/**
 * nautilus_burn_recorder_track_free:
 * @track: #NautilusBurnRecorderTrack
 *
 * Free track data.
 *
 **/
void
nautilus_burn_recorder_track_free (NautilusBurnRecorderTrack *track)
{
	switch (track->type) {
	case NAUTILUS_BURN_RECORDER_TRACK_TYPE_DATA:
		g_free (track->contents.data.filename);
		break;
	case NAUTILUS_BURN_RECORDER_TRACK_TYPE_AUDIO:
		g_free (track->contents.audio.filename);
		g_free (track->contents.audio.cdtext);
		break;
	default:
		g_warning ("Invalid track type %d", track->type);
	}

	g_free (track);
}

static gboolean
can_burn_dvds (NautilusBurnDrive *recorder)
{
	if (!(recorder->type & NAUTILUS_BURN_DRIVE_TYPE_DVD_RW_RECORDER) &&
	    !(recorder->type & NAUTILUS_BURN_DRIVE_TYPE_DVD_PLUS_R_RECORDER) &&
	    !(recorder->type & NAUTILUS_BURN_DRIVE_TYPE_DVD_PLUS_RW_RECORDER)) {
		return FALSE;
	}

	return TRUE;
}

static gboolean
cd_write_needs_growisofs (NautilusBurnDrive    *recorder,
			  NautilusBurnMediaType type,
			  GList                *tracks)
{
	GList *l;

	/* If we cannot burn DVDs, then we don't need growisofs */
	if (can_burn_dvds (recorder) == FALSE) {
		return FALSE;
	}

	if (type == NAUTILUS_BURN_MEDIA_TYPE_DVDR
	    || type == NAUTILUS_BURN_MEDIA_TYPE_DVDRW
	    || type == NAUTILUS_BURN_MEDIA_TYPE_DVD_PLUS_R
	    || type == NAUTILUS_BURN_MEDIA_TYPE_DVD_PLUS_RW) {
		return TRUE;
	}

	/* If we have audio tracks, we're using cdrecord */
	for (l = tracks; l != NULL; l = l->next) {
		NautilusBurnRecorderTrack *track = l->data;
		if (track->type == NAUTILUS_BURN_RECORDER_TRACK_TYPE_AUDIO) {
			return FALSE;
		}
	}

	return FALSE;
}

/**
 * nautilus_burn_recorder_cancel:
 * @recorder: #NautilusBurnRecorder
 * @skip_if_dangerous: 
 *
 * Cancel active writing process.
 *
 * Return value: %TRUE if succesfully cancelled, %FALSE otherwise
 **/
gboolean
nautilus_burn_recorder_cancel (NautilusBurnRecorder *recorder,
			       gboolean              skip_if_dangerous)
{
	if (!recorder->priv->dangerous || !skip_if_dangerous) {
		kill (recorder->priv->pid, SIGINT);

		recorder->priv->result = NAUTILUS_BURN_RECORDER_RESULT_CANCEL;
		g_main_loop_quit (recorder->priv->loop);

		return FALSE;
	}

	return recorder->priv->dangerous;
}

static void
insert_cd_retry (NautilusBurnRecorder *recorder,
		 gboolean              cancel,
		 gboolean              is_reload,
		 gboolean              send_return)
{
	if (cancel) {
		/* Shouldn't be dangerous on reload */
		nautilus_burn_recorder_cancel (recorder, FALSE);
	} else if (is_reload) {
		if (send_return) {
			write (recorder->priv->cdr_stdin, "\n", 1);
		} else {
			kill (recorder->priv->pid, SIGUSR1);
		}
	} else {
		recorder->priv->result = NAUTILUS_BURN_RECORDER_RESULT_RETRY;
		g_main_loop_quit (recorder->priv->loop);
	}
}

static gboolean
growisofs_stdout_read (GIOChannel   *source,
		       GIOCondition  condition,
		       gpointer      data)
{
	NautilusBurnRecorder *recorder = (NautilusBurnRecorder *) data;
	char *line;
	char buf[1];
	GIOStatus status;

	status = g_io_channel_read_line (source,
					 &line, NULL, NULL, NULL);

	if (line && recorder->priv->debug) {
		g_print ("growisofs stdout: %s", line);
	}

	if (status == G_IO_STATUS_NORMAL) {
		double percent;
		int perc_1, perc_2;
		long long tmp;

		if (recorder->priv->line) {
			g_string_append (recorder->priv->line, line);
			g_free (line);
			line = g_string_free (recorder->priv->line, FALSE);
			recorder->priv->line = NULL;
		}

		if (sscanf (line, "%10lld/%lld ( %2d.%1d%%)", &tmp, &tmp, &perc_1, &perc_2) == 4) {
			if (!recorder->priv->changed_text) {
				g_signal_emit (recorder,
					       nautilus_burn_recorder_table_signals [ACTION_CHANGED], 0,
					       NAUTILUS_BURN_RECORDER_ACTION_WRITING, NAUTILUS_BURN_RECORDER_MEDIA_DVD);
			}

			percent = (perc_1 + ((float) perc_2 / 10)) / 100;

			g_signal_emit (recorder,
				       nautilus_burn_recorder_table_signals [PROGRESS_CHANGED], 0,
				       percent);
		} else if (strstr (line, "About to execute") != NULL) {
			recorder->priv->dangerous = TRUE;
		}

		g_free (line);
	} else if (status == G_IO_STATUS_AGAIN) {
		/* A non-terminated line was read, read the data into the buffer. */
		status = g_io_channel_read_chars (source, buf, 1, NULL, NULL);
		if (status == G_IO_STATUS_NORMAL) {
			if (recorder->priv->line == NULL) {
				recorder->priv->line = g_string_new (NULL);
			}
			g_string_append_c (recorder->priv->line, buf[0]);
		}
	} else if (status == G_IO_STATUS_EOF) {
		return FALSE;
	}

	return TRUE;
}

static gboolean
cdrecord_stdout_read (GIOChannel   *source,
		      GIOCondition  condition,
		      gpointer      data)
{
	NautilusBurnRecorder *recorder = (NautilusBurnRecorder *) data;
	char *line;
	char buf[1];
	unsigned int track, mb_written, mb_total;
	GIOStatus status;
	gboolean res;

	status = g_io_channel_read_line (source,
					 &line, NULL, NULL, NULL);
	
	if (line && recorder->priv->debug) {
		g_print ("cdrecord stdout: %s", line);
	}

	if (status == G_IO_STATUS_NORMAL) {
		if (recorder->priv->line) {
			g_string_append (recorder->priv->line, line);
			g_free (line);
			line = g_string_free (recorder->priv->line, FALSE);
			recorder->priv->line = NULL;
		}
		
		if (sscanf (line, "Track %2u: %d of %d MB written",
			    &track, &mb_written, &mb_total) == 3) {
			double percent;
			if (!recorder->priv->changed_text) {
				g_signal_emit (recorder,
					       nautilus_burn_recorder_table_signals [ACTION_CHANGED], 0,
					       NAUTILUS_BURN_RECORDER_ACTION_WRITING, NAUTILUS_BURN_RECORDER_MEDIA_CD);
			}
			percent = 0.98 * ((double)((track-1)/(double)recorder->priv->track_count)
					  + ((double)mb_written/mb_total) / (double)recorder->priv->track_count);
			g_signal_emit (recorder,
				       nautilus_burn_recorder_table_signals [PROGRESS_CHANGED], 0,
				       percent);
		} else if (g_str_has_prefix (line, "Re-load disk and hit <CR>") ||
			   g_str_has_prefix (line, "send SIGUSR1 to continue")) {
			/* This is not supposed to happen since we checked for the cd
			   before starting, but we try to handle it anyway, since mmc
			   profiling can fail. */
			g_signal_emit (recorder,
				       nautilus_burn_recorder_table_signals [INSERT_MEDIA_REQUEST],
				       0, TRUE, recorder->priv->can_rewrite,
				       FALSE, &res);
			recorder->priv->expect_cdrecord_to_die = TRUE;
			insert_cd_retry (recorder, !res, TRUE, (*line == 'R'));
		} else if (g_str_has_prefix (line, "Fixating...")) {
			g_signal_emit (recorder,
				       nautilus_burn_recorder_table_signals [ACTION_CHANGED], 0,
				       NAUTILUS_BURN_RECORDER_ACTION_FIXATING, NAUTILUS_BURN_RECORDER_MEDIA_CD);
		} else if (g_str_has_prefix (line, "Fixating time:")) {
			g_signal_emit (recorder,
				       nautilus_burn_recorder_table_signals [PROGRESS_CHANGED], 0,
				       1.0);
			recorder->priv->result = NAUTILUS_BURN_RECORDER_RESULT_FINISHED;
		} else if (g_str_has_prefix (line, "Last chance to quit, ")) {
			recorder->priv->dangerous = TRUE;
		} else if (g_str_has_prefix (line, "Blanking PMA, TOC, pregap")) {
			g_signal_emit (recorder,
				       nautilus_burn_recorder_table_signals [ACTION_CHANGED], 0,
				       NAUTILUS_BURN_RECORDER_ACTION_BLANKING, NAUTILUS_BURN_RECORDER_MEDIA_CD);
		}

		g_free (line);
		
	} else if (status == G_IO_STATUS_AGAIN) {
		/* A non-terminated line was read, read the data into the buffer. */
		status = g_io_channel_read_chars (source, buf, 1, NULL, NULL);
		if (status == G_IO_STATUS_NORMAL) {
			if (recorder->priv->line == NULL) {
				recorder->priv->line = g_string_new (NULL);
			}
			g_string_append_c (recorder->priv->line, buf[0]);
		}
	} else if (status == G_IO_STATUS_EOF) {
		return FALSE;
	}

	return TRUE;
}

static gboolean
growisofs_stderr_read (GIOChannel   *source,
		       GIOCondition  condition,
		       gpointer      data)
{
	NautilusBurnRecorder *recorder = (NautilusBurnRecorder *) data;
	char *line;
	GIOStatus status;
	gboolean res;

	status = g_io_channel_read_line (source,
					 &line, NULL, NULL, NULL);
	if (line && recorder->priv->debug) {
		g_print ("growisofs stderr: %s", line);
	}

	if (status == G_IO_STATUS_NORMAL && !recorder->priv->expect_cdrecord_to_die) {
		g_string_append (recorder->priv->cdr_stderr, line);
		if (strstr (line, "unsupported MMC profile") != NULL || (strstr (line, "already carries isofs") != NULL && strstr (line, "FATAL:") != NULL)) {
			/* This is not supposed to happen since we checked for the cd
			   type before starting, but we try to handle it anyway, since mmc
			   profiling can fail. */
			g_signal_emit (recorder,
				       nautilus_burn_recorder_table_signals [INSERT_MEDIA_REQUEST],
				       0, TRUE, recorder->priv->can_rewrite,
				       FALSE, &res);
			recorder->priv->expect_cdrecord_to_die = TRUE;
			insert_cd_retry (recorder, !res, FALSE, FALSE);
		} else if (strstr (line, "unable to open") != NULL || strstr (line, "unable to stat") != NULL) {
			/* This fits the "open64" and "open"-like messages */
			recorder->priv->last_error = g_strdup (_("The recorder could not be accessed"));
			recorder->priv->result = NAUTILUS_BURN_RECORDER_RESULT_ERROR;
		} else if (strstr (line, "not enough space available") != NULL) {
			recorder->priv->last_error = g_strdup (_("Not enough space available on the disc"));
			recorder->priv->result = NAUTILUS_BURN_RECORDER_RESULT_ERROR;
		} else if (strstr (line, "end of user area encountered on this track") != NULL) {
			recorder->priv->last_error = g_strdup (_("The files selected did not fit on the CD"));
			recorder->priv->result = NAUTILUS_BURN_RECORDER_RESULT_ERROR;
		} else if (strstr (line, "blocks are free") != NULL) {
			recorder->priv->last_error = g_strdup (_("The files selected did not fit on the CD"));
			recorder->priv->result = NAUTILUS_BURN_RECORDER_RESULT_ERROR;
		} else if (strstr (line, "flushing cache") != NULL) {
			g_signal_emit (recorder,
				       nautilus_burn_recorder_table_signals [ACTION_CHANGED], 0,
				       NAUTILUS_BURN_RECORDER_ACTION_FIXATING, NAUTILUS_BURN_RECORDER_MEDIA_DVD);
			recorder->priv->result = NAUTILUS_BURN_RECORDER_RESULT_FINISHED;
		} else if (strstr (line, ":-(") != NULL || strstr (line, "FATAL") != NULL) {
			recorder->priv->last_error = g_strdup (_("Unhandled error, aborting"));
			recorder->priv->result = NAUTILUS_BURN_RECORDER_RESULT_ERROR;
		}
	} else if (status == G_IO_STATUS_EOF) {
		if (!recorder->priv->expect_cdrecord_to_die) {
			g_main_loop_quit (recorder->priv->loop);
		}
		return FALSE;
	} else {
		g_print ("growisofs stderr read failed, status: %d\n", status);
	}

	return TRUE;
}

static gboolean  
cdrecord_stderr_read (GIOChannel   *source,
		      GIOCondition  condition,
		      gpointer      data)
{
	NautilusBurnRecorder *recorder = (NautilusBurnRecorder *) data;
	char *line;
	GIOStatus status;
	gboolean res;

	status = g_io_channel_read_line (source,
					 &line, NULL, NULL, NULL);
	if (line && recorder->priv->debug) {
		g_print ("cdrecord stderr: %s", line);
	}

	/* TODO: Handle errors */
	if (status == G_IO_STATUS_NORMAL && !recorder->priv->expect_cdrecord_to_die) {
		g_string_prepend (recorder->priv->cdr_stderr, line);
		if (strstr (line, "No disk / Wrong disk!") != NULL) {
			/* This is not supposed to happen since we checked for the cd
			   before starting, but we try to handle it anyway, since mmc
			   profiling can fail. */
			g_signal_emit (recorder,
				       nautilus_burn_recorder_table_signals [INSERT_MEDIA_REQUEST],
				       0, TRUE, recorder->priv->can_rewrite,
				       FALSE, &res);
			recorder->priv->expect_cdrecord_to_die = TRUE;
			insert_cd_retry (recorder, !res, FALSE, FALSE);
		} else if (strstr (line, "This means that we are checking recorded media.") != NULL) {
			recorder->priv->last_error = g_strdup (_("The CD has already been recorded"));
			recorder->priv->result = NAUTILUS_BURN_RECORDER_RESULT_ERROR;
		} else if (strstr (line, "Cannot blank disk, aborting.") != NULL) {
			/* This is not supposed to happen since we checked for the cd
			   type before starting, but we try to handle it anyway, since
			   mmc profiling can fail. */
			g_signal_emit (recorder,
				       nautilus_burn_recorder_table_signals [INSERT_MEDIA_REQUEST],
				       0, TRUE, TRUE,
				       FALSE, &res);
			recorder->priv->expect_cdrecord_to_die = TRUE;
			insert_cd_retry (recorder, !res, FALSE, FALSE);
		} else if (strstr (line, "Data may not fit on current disk") != NULL) {
			recorder->priv->last_error = g_strdup (_("The files selected did not fit on the CD"));
			/* FIXME should we error out in that case?
			   recorder->priv->result = NAUTILUS_BURN_RECORDER_RESULT_ERROR; */
		} else if (strstr (line, "Inappropriate audio coding") != NULL) {
			recorder->priv->last_error = g_strdup (_("All audio files must be stereo, 16-bit digital audio with 44100Hz samples"));
			recorder->priv->result = NAUTILUS_BURN_RECORDER_RESULT_ERROR;
		} else if (strstr (line, "cannot write medium - incompatible format") != NULL) {
			/* This is not supposed to happen since we checked for the cd
			   type before starting, but we try to handle it anyway, since
			   mmc profiling can fail. */
			g_signal_emit (recorder,
				       nautilus_burn_recorder_table_signals [INSERT_MEDIA_REQUEST],
				       0, TRUE, recorder->priv->can_rewrite,
				       FALSE, &res);

			recorder->priv->expect_cdrecord_to_die = TRUE;
			insert_cd_retry (recorder, !res, FALSE, FALSE);
		} else if (strstr (line, "DMA speed too slow") != NULL) {
			recorder->priv->last_error = g_strdup (_("The system is too slow to write the CD at this speed. Try a lower speed."));
		}

		g_free (line);
	} else if (status == G_IO_STATUS_EOF) {
		if (!recorder->priv->expect_cdrecord_to_die) {
			g_main_loop_quit (recorder->priv->loop);
		}
		return FALSE;
	} else {
		g_print ("cdrecord stderr read failed, status: %d\n", status);
	}

	return TRUE;
}

static gboolean
media_type_matches (NautilusBurnMediaType type,
		    gboolean              is_blank,
		    gboolean              was_unmounted)
{
	if (was_unmounted) {
		switch (type) {
		case NAUTILUS_BURN_MEDIA_TYPE_CDRW:
		case NAUTILUS_BURN_MEDIA_TYPE_DVDRW:
		case NAUTILUS_BURN_MEDIA_TYPE_DVD_PLUS_RW:
			return TRUE;
		default:
			return FALSE;
		}
	}

	switch (type) {
	case NAUTILUS_BURN_MEDIA_TYPE_ERROR:
	case NAUTILUS_BURN_MEDIA_TYPE_BUSY:
		return FALSE;
	case NAUTILUS_BURN_MEDIA_TYPE_UNKNOWN:
		return TRUE;
	case NAUTILUS_BURN_MEDIA_TYPE_CD:
	case NAUTILUS_BURN_MEDIA_TYPE_DVD:
	case NAUTILUS_BURN_MEDIA_TYPE_DVD_RAM:
		return FALSE;
	case NAUTILUS_BURN_MEDIA_TYPE_CDR:
	case NAUTILUS_BURN_MEDIA_TYPE_DVDR:
	case NAUTILUS_BURN_MEDIA_TYPE_DVD_PLUS_R:
		if (is_blank)
			return TRUE;
		else
			return FALSE;
	case NAUTILUS_BURN_MEDIA_TYPE_CDRW:
	case NAUTILUS_BURN_MEDIA_TYPE_DVDRW:
	case NAUTILUS_BURN_MEDIA_TYPE_DVD_PLUS_RW:
		return TRUE;
	}

	return FALSE;
}

static NautilusBurnMediaType
nautilus_burn_recorder_wait_for_insertion (NautilusBurnRecorder *recorder,
					   NautilusBurnDrive    *drive,
					   gboolean             *needs_blank)
{
	NautilusBurnMediaType type;
	gboolean              reload;
	gboolean              last_was_unmount;
	gboolean              is_blank;
	gboolean              has_data;
	gboolean              has_audio;

	last_was_unmount = FALSE;
	reload = FALSE;
	type = nautilus_burn_drive_get_media_type_full (drive, needs_blank, &is_blank, &has_data, &has_audio);

	if (type == NAUTILUS_BURN_MEDIA_TYPE_ERROR) {
		reload = TRUE;
	}

	last_was_unmount = FALSE;
	while (!media_type_matches (type, is_blank, last_was_unmount)) {
		gboolean res, busy_cd;

		busy_cd = (type == NAUTILUS_BURN_MEDIA_TYPE_BUSY);

		if (busy_cd && !last_was_unmount) {
			/* umount the disc if we find it, and try it again */
			if (nautilus_burn_drive_unmount (drive)) {
				last_was_unmount = TRUE;
				goto retry;
			}
		}

		g_signal_emit (recorder,
			       nautilus_burn_recorder_table_signals [INSERT_MEDIA_REQUEST],
			       0, reload, recorder->priv->can_rewrite,
			       busy_cd, &res);

		if (res == FALSE) {
			return NAUTILUS_BURN_MEDIA_TYPE_ERROR;
		}

		last_was_unmount = FALSE;

	retry:
		type = nautilus_burn_drive_get_media_type_full (drive, needs_blank, &is_blank, &has_data, &has_audio);

		reload = FALSE;
		if (type == NAUTILUS_BURN_MEDIA_TYPE_UNKNOWN
		    || type == NAUTILUS_BURN_MEDIA_TYPE_ERROR) {
			reload = TRUE;
		}
	}

	if (!is_blank) {
		gboolean res;

		res = TRUE;

		/* We need to warn the user that it's going to erase their stuff */
		g_signal_emit (G_OBJECT (recorder),
			       nautilus_burn_recorder_table_signals [WARN_DATA_LOSS],
			       0, &res);

		if (res != FALSE) {
			return NAUTILUS_BURN_MEDIA_TYPE_ERROR;
		}
	}

	return type;
}

/**
 * nautilus_burn_recorder_write_tracks:
 * @recorder: #NautilusBurnRecorder
 * @drive: #NautilusBurnDrive to write to
 * @tracks: list of tracks to write
 * @speed: speed at which to write data
 * @flags: #NautilusBurnRecorderWriteFlags
 *
 * Write tracks to the specified drive.
 *
 * Return value: #NautilusBurnRecorderResult
 **/
int
nautilus_burn_recorder_write_tracks (NautilusBurnRecorder          *recorder,
				     NautilusBurnDrive             *drive,
				     GList                         *tracks,
				     int                            speed,
				     NautilusBurnRecorderWriteFlags flags)
     /* TODO: GError **error */
{
	NautilusBurnMediaType type;
	gboolean needs_blank;
	gboolean is_locked;
	int ret;

	g_return_val_if_fail (tracks != NULL, NAUTILUS_BURN_RECORDER_RESULT_ERROR);
	recorder->priv->tracks = tracks;
	recorder->priv->track_count = g_list_length (tracks);
	recorder->priv->debug = (flags & NAUTILUS_BURN_RECORDER_WRITE_DEBUG);

	/* Let's hope that DVD+RW and DVD-RW drives can also do CD-RWs */
	recorder->priv->can_rewrite =
		(drive->type & NAUTILUS_BURN_DRIVE_TYPE_CDRW_RECORDER);

	if (recorder->priv->track_count > 99) {
		recorder->priv->last_error = g_strdup (_("You can only burn 99 tracks on one disc"));
		recorder->priv->result = NAUTILUS_BURN_RECORDER_RESULT_ERROR;
		return recorder->priv->result;
	}

	is_locked = nautilus_burn_drive_lock (drive, _("Burning CD"), NULL);

	type = nautilus_burn_recorder_wait_for_insertion (recorder,
							  drive,
							  &needs_blank);

	if (type == NAUTILUS_BURN_MEDIA_TYPE_ERROR) {
		if (is_locked)
			nautilus_burn_drive_unlock (drive);

		recorder->priv->result = NAUTILUS_BURN_RECORDER_RESULT_CANCEL;
		return recorder->priv->result;
	}

	if (needs_blank != FALSE) {
		flags |= NAUTILUS_BURN_RECORDER_WRITE_BLANK;
	}

	if (cd_write_needs_growisofs (drive, type, tracks)) {
		ret = nautilus_burn_recorder_write_growisofs (recorder,
							      drive, tracks, speed,
							      flags);
	} else {
		ret = nautilus_burn_recorder_write_cdrecord (recorder,
							     drive, tracks, speed,
							     flags);
	}

	if (is_locked) {
		nautilus_burn_drive_unlock (drive);
	}

	return ret;
}

static int
nautilus_burn_recorder_write_growisofs (NautilusBurnRecorder          *recorder,
					NautilusBurnDrive             *drive,
					GList                         *tracks,
					int                            speed,
					NautilusBurnRecorderWriteFlags flags)
{
	GPtrArray *argv;
	char *speed_str, *dev_str;
	int stdout_pipe, stderr_pipe;
	guint stdout_tag, stderr_tag;
	GIOChannel *channel;
	GError *error;
	NautilusBurnRecorderTrack *t;

	if (g_list_length (tracks) != 1) {
		g_warning ("Can only use growisofs on a single track");
		return NAUTILUS_BURN_RECORDER_RESULT_ERROR;
	}
	t = (NautilusBurnRecorderTrack*)tracks->data;
	if (t->type != NAUTILUS_BURN_RECORDER_TRACK_TYPE_DATA) {
		g_warning ("Can only use growisofs on a data track");
		return NAUTILUS_BURN_RECORDER_RESULT_ERROR;
	}

	argv = g_ptr_array_new ();

	g_ptr_array_add (argv, "growisofs");
	speed_str = g_strdup_printf ("-speed=%d", speed);
	if (speed != 0) {
		g_ptr_array_add (argv, speed_str);
	}

	g_ptr_array_add (argv, "-dvd-compat");

	/* Weird, innit? We tell growisofs we have a tty so it ignores
	 * the fact that the DVD+ has an ISO fs already */
	if (flags & NAUTILUS_BURN_RECORDER_WRITE_BLANK) {
		g_ptr_array_add (argv, "-use-the-force-luke=tty");
	}

	g_ptr_array_add (argv, "-Z");

	dev_str = g_strdup_printf ("%s=%s", drive->device, t->contents.data.filename);
	g_ptr_array_add (argv, dev_str);

	g_ptr_array_add (argv, NULL);

	recorder->priv->cdr_stderr = NULL;
 retry_growisofs:
	recorder->priv->result = NAUTILUS_BURN_RECORDER_RESULT_ERROR;
	recorder->priv->expect_cdrecord_to_die = FALSE;
	recorder->priv->line = NULL;
	if (recorder->priv->cdr_stderr != NULL) {
		g_string_truncate (recorder->priv->cdr_stderr, 0);
	} else {
		recorder->priv->cdr_stderr = g_string_new (NULL);
	}

	g_signal_emit (recorder,
		       nautilus_burn_recorder_table_signals [ACTION_CHANGED], 0,
		       NAUTILUS_BURN_RECORDER_ACTION_PREPARING_WRITE, NAUTILUS_BURN_RECORDER_MEDIA_DVD);
	recorder->priv->changed_text = FALSE;
	g_signal_emit (recorder,
		       nautilus_burn_recorder_table_signals [PROGRESS_CHANGED], 0,
		       0.0);
	g_signal_emit (recorder,
		       nautilus_burn_recorder_table_signals [ANIMATION_CHANGED], 0,
		       TRUE);
	recorder->priv->dangerous = FALSE;

	if (recorder->priv->debug)
		{
			guint i;
			g_print ("launching command: ");
			for (i = 0; i < argv->len - 1; i++) {
				g_print ("%s ", (char*)g_ptr_array_index (argv, i));
			}
			g_print ("\n");
		}

	error = NULL;
	if (!g_spawn_async_with_pipes  (NULL,
					(char**)argv->pdata,
					NULL,
					G_SPAWN_SEARCH_PATH,
					NULL, NULL,
					&recorder->priv->pid,
					&recorder->priv->cdr_stdin,
					&stdout_pipe,
					&stderr_pipe,
					&error)) {
		guint i;
		g_warning ("growisofs command failed: %s\n", error->message);
		for (i = 0; i < argv->len - 1; i++) {
			g_print ("%s ", (char*)g_ptr_array_index (argv, i));
		}
		g_print ("\n");
		g_error_free (error);
		/* TODO: Better error handling */
	} else {
		/* Make sure we don't block on a read. */
		fcntl (stdout_pipe, F_SETFL, O_NONBLOCK);
		fcntl (stderr_pipe, F_SETFL, O_NONBLOCK);

		recorder->priv->loop = g_main_loop_new (NULL, FALSE);

		channel = g_io_channel_unix_new (stdout_pipe);
		g_io_channel_set_encoding (channel, NULL, NULL);
		stdout_tag = g_io_add_watch (channel,
					     (G_IO_IN | G_IO_HUP | G_IO_ERR),
					     growisofs_stdout_read,
					     recorder);
		g_io_channel_unref (channel);
		channel = g_io_channel_unix_new (stderr_pipe);
		g_io_channel_set_encoding (channel, NULL, NULL);
		stderr_tag = g_io_add_watch (channel,
					     (G_IO_IN | G_IO_HUP | G_IO_ERR),
					     growisofs_stderr_read,
					     recorder);
		g_io_channel_unref (channel);

		recorder->priv->dangerous = FALSE;

		g_main_loop_run (recorder->priv->loop);
		g_main_loop_unref (recorder->priv->loop);

		g_source_remove (stdout_tag);
		g_source_remove (stderr_tag);

		if (recorder->priv->result == NAUTILUS_BURN_RECORDER_RESULT_RETRY) {
			goto retry_growisofs;
		}
	}

	g_free (speed_str);
	g_free (dev_str);
	g_ptr_array_free (argv, TRUE);
	
	g_signal_emit (recorder,
		       nautilus_burn_recorder_table_signals [ANIMATION_CHANGED], 0,
		       FALSE);

	if (flags & NAUTILUS_BURN_RECORDER_WRITE_EJECT && recorder->priv->result == NAUTILUS_BURN_RECORDER_RESULT_FINISHED) {
		char *cmd;

		cmd = g_strdup_printf ("eject %s", drive->device);
		g_spawn_command_line_sync (cmd, NULL, NULL, NULL, NULL);
		g_free (cmd);
	}

	return recorder->priv->result;
}

static void
nautilus_burn_recorder_run_cdrecord (NautilusBurnRecorder *recorder,
				     NautilusBurnDrive    *drive,
				     GPtrArray            *argv)
{
	int         stdout_pipe, stderr_pipe;
	guint       stdout_tag, stderr_tag;
	GIOChannel *channel;
	GError     *error;

	recorder->priv->cdr_stderr = NULL;
 retry:
	recorder->priv->result = NAUTILUS_BURN_RECORDER_RESULT_ERROR;
	recorder->priv->expect_cdrecord_to_die = FALSE;
	recorder->priv->line = NULL;
	if (recorder->priv->cdr_stderr != NULL) {
		g_string_truncate (recorder->priv->cdr_stderr, 0);
	} else {
		recorder->priv->cdr_stderr = g_string_new (NULL);
	}

	g_signal_emit (G_OBJECT (recorder),
		       nautilus_burn_recorder_table_signals [ACTION_CHANGED], 0,
		       NAUTILUS_BURN_RECORDER_ACTION_PREPARING_WRITE, NAUTILUS_BURN_RECORDER_MEDIA_CD);
	recorder->priv->changed_text = FALSE;
	g_signal_emit (G_OBJECT (recorder),
		       nautilus_burn_recorder_table_signals [PROGRESS_CHANGED], 0,
		       0.0);
	g_signal_emit (G_OBJECT (recorder),
		       nautilus_burn_recorder_table_signals [ANIMATION_CHANGED], 0,
		       TRUE);
	recorder->priv->dangerous = FALSE;

	if (recorder->priv->debug) {
		guint i;
		g_print ("launching command: ");
		for (i = 0; i < argv->len - 1; i++) {
			g_print ("%s ", (char*)g_ptr_array_index (argv, i));
		}
		g_print ("\n");
	}

	error = NULL;
	if (!g_spawn_async_with_pipes (NULL,
				       (char**)argv->pdata,
				       NULL,
				       G_SPAWN_SEARCH_PATH,
				       NULL, NULL,
				       &recorder->priv->pid,
				       &recorder->priv->cdr_stdin,
				       &stdout_pipe,
				       &stderr_pipe,
				       &error)) {
		g_warning ("cdrecord command failed: %s\n", error->message);
		g_error_free (error);
		/* TODO: Better error handling */
	} else {
		/* Make sure we don't block on a read. */
		fcntl (stdout_pipe, F_SETFL, O_NONBLOCK);
		fcntl (stderr_pipe, F_SETFL, O_NONBLOCK);

		recorder->priv->loop = g_main_loop_new (NULL, FALSE);

		channel = g_io_channel_unix_new (stdout_pipe);
		g_io_channel_set_encoding (channel, NULL, NULL);
		stdout_tag = g_io_add_watch (channel, 
					     (G_IO_IN | G_IO_HUP | G_IO_ERR), 
					     cdrecord_stdout_read,
					     recorder);
		g_io_channel_unref (channel);
		channel = g_io_channel_unix_new (stderr_pipe);
		g_io_channel_set_encoding (channel, NULL, NULL);
		stderr_tag = g_io_add_watch (channel, 
					     (G_IO_IN | G_IO_HUP | G_IO_ERR), 
					     cdrecord_stderr_read,
					     recorder);
		g_io_channel_unref (channel);

		recorder->priv->dangerous = FALSE;

		g_main_loop_run (recorder->priv->loop);
		g_main_loop_unref (recorder->priv->loop);
		
		g_source_remove (stdout_tag);
		g_source_remove (stderr_tag);

		if (recorder->priv->result == NAUTILUS_BURN_RECORDER_RESULT_RETRY) {
			goto retry;
		}
	}

	g_signal_emit (G_OBJECT (recorder),
		       nautilus_burn_recorder_table_signals [ANIMATION_CHANGED], 0,
		       FALSE);
}

static int
nautilus_burn_recorder_write_cdrecord (NautilusBurnRecorder          *recorder,
				       NautilusBurnDrive             *drive,
				       GList                         *tracks,
				       int                            speed,
				       NautilusBurnRecorderWriteFlags flags)
{
	GPtrArray *argv;
	char      *speed_str, *dev_str;
	GList     *l;

	g_return_val_if_fail (tracks != NULL, NAUTILUS_BURN_RECORDER_RESULT_ERROR);

	argv = g_ptr_array_new ();
	g_ptr_array_add (argv, "cdrecord");

	speed_str = g_strdup_printf ("speed=%d", speed);
	if (speed != 0) {
		g_ptr_array_add (argv, speed_str);
	}
	dev_str = g_strdup_printf ("dev=%s", drive->cdrecord_id);
	g_ptr_array_add (argv, dev_str);
	if (flags & NAUTILUS_BURN_RECORDER_WRITE_DUMMY_WRITE) {
		g_ptr_array_add (argv, "-dummy");
	}
	if (flags & NAUTILUS_BURN_RECORDER_WRITE_EJECT) {
		g_ptr_array_add (argv, "-eject");
	}
	if (flags & NAUTILUS_BURN_RECORDER_WRITE_BLANK) {
		g_ptr_array_add (argv, "blank=fast");
	}
	if (flags & NAUTILUS_BURN_RECORDER_WRITE_DISC_AT_ONCE) {
		g_ptr_array_add (argv, "-dao");
	}
	if (flags & NAUTILUS_BURN_RECORDER_WRITE_OVERBURN) {
		g_ptr_array_add (argv, "-overburn");
	}
	if (flags & NAUTILUS_BURN_RECORDER_WRITE_BURNPROOF) {
		g_ptr_array_add (argv, "driveropts=burnfree");
	}
	g_ptr_array_add (argv, "-v");

	l = tracks;
	while (l != NULL && l->data != NULL) {
		NautilusBurnRecorderTrack *track = l->data;
		switch (track->type) {
		case NAUTILUS_BURN_RECORDER_TRACK_TYPE_DATA:
			g_ptr_array_add (argv, "-data");
			g_ptr_array_add (argv, "-nopad");
			g_ptr_array_add (argv, track->contents.data.filename);
			break;
		case NAUTILUS_BURN_RECORDER_TRACK_TYPE_AUDIO:
			g_ptr_array_add (argv, "-copy");
			g_ptr_array_add (argv, "-audio");
			g_ptr_array_add (argv, "-pad");
			g_ptr_array_add (argv, track->contents.audio.filename);
			/* TODO: handle CD-TEXT somehow */
			break;
		default:
			g_warning ("Unknown track type %d", track->type);
		}
		l = g_list_next (l);
	}
	g_ptr_array_add (argv, NULL);

	nautilus_burn_recorder_run_cdrecord (recorder, drive, argv);

	g_free (speed_str);
	g_free (dev_str);
	g_ptr_array_free (argv, TRUE);

	return recorder->priv->result;
}

static int
nautilus_burn_recorder_blank_disc_cdrecord (NautilusBurnRecorder         *recorder,
					    NautilusBurnDrive            *drive,
					    NautilusBurnRecorderBlankType type,
					    gboolean                      debug)
{
	GPtrArray            *argv;
	NautilusBurnMediaType media_type;
	char                 *blank_str, *dev_str;
	gboolean              is_locked;
	gboolean              can_blank;

	recorder->priv->debug = debug;

	recorder->priv->can_rewrite = (drive->type & NAUTILUS_BURN_DRIVE_TYPE_CDRW_RECORDER);

	if (!recorder->priv->can_rewrite) {
		recorder->priv->result = NAUTILUS_BURN_RECORDER_RESULT_CANCEL;
		return recorder->priv->result;
	}

	is_locked = nautilus_burn_drive_lock (drive, _("Blanking CD"), NULL);

	media_type = nautilus_burn_recorder_wait_for_insertion (recorder, drive, &can_blank);

	if (media_type == NAUTILUS_BURN_MEDIA_TYPE_ERROR || can_blank == FALSE) {
		if (is_locked)
			nautilus_burn_drive_unlock (drive);

		recorder->priv->result = NAUTILUS_BURN_RECORDER_RESULT_CANCEL;
		return recorder->priv->result;
	}

	argv = g_ptr_array_new ();
	g_ptr_array_add (argv, "cdrecord");

	dev_str = g_strdup_printf ("dev=%s", drive->cdrecord_id);
	g_ptr_array_add (argv, dev_str);

	blank_str = g_strdup_printf ("blank=%s", (type == NAUTILUS_BURN_RECORDER_BLANK_FAST) ? "fast" : "full");
	g_ptr_array_add (argv, blank_str);

	g_ptr_array_add (argv, NULL);

	nautilus_burn_recorder_run_cdrecord (recorder, drive, argv);

	if (is_locked) {
		nautilus_burn_drive_unlock (drive);
	}

	g_free (dev_str);
	g_free (blank_str);
	g_ptr_array_free (argv, TRUE);
	
	return recorder->priv->result;
}

static int
nautilus_burn_recorder_blank_disc_dvdrw_format (NautilusBurnRecorder         *recorder,
						NautilusBurnDrive            *drive,
						NautilusBurnRecorderBlankType type,
						gboolean                      debug)
{
	g_warning ("Implement me!");
	return NAUTILUS_BURN_RECORDER_RESULT_ERROR;
}


static gboolean
nautilus_burn_drive_format_needs_growisofs (NautilusBurnDrive    *drive,
					    NautilusBurnMediaType type)
{
	if (can_burn_dvds (drive) == FALSE) {
		return FALSE;
	}

	if (type == NAUTILUS_BURN_MEDIA_TYPE_DVDRW
	    || type == NAUTILUS_BURN_MEDIA_TYPE_DVD_PLUS_RW) {
		return TRUE;
	}

	return FALSE;
}

int
nautilus_burn_recorder_blank_disc (NautilusBurnRecorder         *recorder,
				   NautilusBurnDrive            *drive,
				   NautilusBurnRecorderBlankType type,
				   gboolean                      debug)
{
	NautilusBurnMediaType media_type;
	gboolean              needs_blank;

	media_type = nautilus_burn_recorder_wait_for_insertion (recorder, drive, &needs_blank);

	if (media_type == NAUTILUS_BURN_MEDIA_TYPE_ERROR || media_type == NAUTILUS_BURN_MEDIA_TYPE_BUSY) {
		return NAUTILUS_BURN_RECORDER_RESULT_ERROR;
	} else if (needs_blank == FALSE) {
		return NAUTILUS_BURN_RECORDER_RESULT_ERROR;
	}

	if (nautilus_burn_drive_format_needs_growisofs (drive, media_type)) {
		return nautilus_burn_recorder_blank_disc_cdrecord (recorder, drive,
								   type, debug);
	} else {
		return nautilus_burn_recorder_blank_disc_dvdrw_format (recorder, drive,
								       type, debug);
	}
}


/**
 * nautilus_burn_recorder_get_error_message_details:
 * @recorder: #NautilusBurnRecorder
 *
 * Get a detailed string description of the most recently occuring error.
 *
 * Return value: Detailed string description of error condition.  The return
 * value is owned by NautilusBurnRecorder and must not be modified or freed.
 **/
const char *
nautilus_burn_recorder_get_error_message_details (NautilusBurnRecorder *recorder)
{
	g_return_val_if_fail (recorder->priv->result == NAUTILUS_BURN_RECORDER_RESULT_ERROR, NULL);

	return (const char *)recorder->priv->cdr_stderr->str;
}

/**
 * nautilus_burn_recorder_get_error_message:
 * @recorder: #NautilusBurnRecorder
 *
 * Get a string description of the most recently occuring error.
 *
 * Return value: String description of error condition.  The return
 * value is owned by NautilusBurnRecorder and must not be modified or freed.
 **/
const char *
nautilus_burn_recorder_get_error_message (NautilusBurnRecorder *recorder)
{
	g_return_val_if_fail (recorder->priv->result == NAUTILUS_BURN_RECORDER_RESULT_ERROR, NULL);

	return (const char *)recorder->priv->last_error;
}

static void
nautilus_burn_recorder_finalize (GObject *object)
{
	NautilusBurnRecorder *recorder = NAUTILUS_BURN_RECORDER (object);

	g_return_if_fail (object != NULL);

	if (recorder->priv->cdr_stderr != NULL) {
		g_string_free (recorder->priv->cdr_stderr, TRUE);
	}

	if (recorder->priv->line != NULL) {
		g_string_free (recorder->priv->line, TRUE);
	}

	g_free (recorder->priv->last_error);

	if (G_OBJECT_CLASS (parent_class)->finalize != NULL) {
		(* G_OBJECT_CLASS (parent_class)->finalize) (object);
	}
}

static void
nautilus_burn_recorder_init (NautilusBurnRecorder *recorder)
{
	recorder->priv = g_new0 (NautilusBurnRecorderPrivate, 1);
}

/**
 * nautilus_burn_recorder_new:
 *
 * Create a new #NautilusBurnRecorder.
 *
 * Return value: The new recorder.
 **/
NautilusBurnRecorder *
nautilus_burn_recorder_new (void)
{
	return g_object_new (NAUTILUS_BURN_TYPE_RECORDER, NULL);
}

static void
nautilus_burn_recorder_class_init (NautilusBurnRecorderClass *klass)
{
	GObjectClass *object_class;

	object_class = (GObjectClass *) klass;

	parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = nautilus_burn_recorder_finalize;

	/* Signals */
	nautilus_burn_recorder_table_signals [PROGRESS_CHANGED] =
		g_signal_new ("progress-changed",
			      G_TYPE_FROM_CLASS (klass),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (NautilusBurnRecorderClass,
					       progress_changed),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__DOUBLE,
			      G_TYPE_NONE, 1, G_TYPE_DOUBLE);
	nautilus_burn_recorder_table_signals [ACTION_CHANGED] =
		g_signal_new ("action-changed",
			      G_TYPE_FROM_CLASS (klass),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (NautilusBurnRecorderClass,
					       action_changed),
			      NULL, NULL,
			      nautilus_burn_recorder_marshal_VOID__INT_INT,
			      G_TYPE_NONE, 2, G_TYPE_INT, G_TYPE_INT);
	nautilus_burn_recorder_table_signals [ANIMATION_CHANGED] =
		g_signal_new ("animation-changed",
			      G_TYPE_FROM_CLASS (klass),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (NautilusBurnRecorderClass,
					       animation_changed),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__BOOLEAN,
			      G_TYPE_NONE, 1, G_TYPE_BOOLEAN);
	nautilus_burn_recorder_table_signals [INSERT_MEDIA_REQUEST] =
		g_signal_new ("insert-media-request",
			      G_TYPE_FROM_CLASS (klass),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (NautilusBurnRecorderClass,
					       insert_cd_request),
			      NULL, NULL,
			      nautilus_burn_recorder_marshal_BOOLEAN__BOOLEAN_BOOLEAN_BOOLEAN,
			      G_TYPE_BOOLEAN, 3, G_TYPE_BOOLEAN,
			      G_TYPE_BOOLEAN, G_TYPE_BOOLEAN);
	nautilus_burn_recorder_table_signals [WARN_DATA_LOSS] =
		g_signal_new ("warn-data-loss",
			      G_TYPE_FROM_CLASS (klass),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (NautilusBurnRecorderClass,
					       warn_data_loss),
			      NULL, NULL,
			      nautilus_burn_recorder_marshal_BOOLEAN__VOID,
			      G_TYPE_BOOLEAN, 0);
}
