/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */

package org.gnu.gtk;

import java.util.Vector;

import org.gnu.glib.EventMap;
import org.gnu.glib.EventType;
import org.gnu.glib.GObject;
import org.gnu.glib.Handle;
import org.gnu.gtk.event.ToolButtonEvent;
import org.gnu.gtk.event.ToolButtonListener;

/**
 * A {@link ToolItem} subclass that displays buttons.
 */
public class ToolButton extends ToolItem {
	
	public ToolButton(Widget iconWidget, String label) {
		super(init(iconWidget, label));
	}
	
	private static Handle init(Widget iconWidget, String label) {
	    Handle iconHndl = null;
		if (null != iconWidget)
			iconHndl = iconWidget.getHandle();
		return gtk_tool_button_new(iconHndl, label);
	}
	
	public ToolButton(String stockId) {
		super(gtk_tool_button_new_from_stock(stockId));
	}
	
	public ToolButton(Handle hndl) {
		super(hndl);
	}
	
	public void setLabel(String label) {
		gtk_tool_button_set_label(getHandle(), label);
	}
	
	public String getLabel() {
		return gtk_tool_button_get_label(getHandle());
	}
	
	public void setUseUnderline(boolean useUnderline) {
		gtk_tool_button_set_use_underline(getHandle(), useUnderline);
	}
	
	public boolean getUseUnderline() {
		return gtk_tool_button_get_use_underline(getHandle());
	}
	
	public void setStockId(String stockId) {
		gtk_tool_button_set_stock_id(getHandle(), stockId);
	}
	
	public String getStockId() {
		return gtk_tool_button_get_stock_id(getHandle());
	}
	
	public void setIconWidget(Widget iconWidget) {
		gtk_tool_button_set_icon_widget(getHandle(), iconWidget.getHandle());
	}
	
	public Widget getIconWidget() {
	    Handle hndl = gtk_tool_button_get_icon_widget(getHandle());
		if (null == hndl)
			return null;
		GObject obj = getGObjectFromHandle(hndl);
		if (null != obj)
			return (Widget)obj;
		return new Widget(hndl);
	}
	
	public void setLabelWidget(Label labelWidget) {
		gtk_tool_button_set_label_widget(getHandle(), labelWidget.getHandle());
	}
	
	public Label getLabelWidget() {
	    Handle hndl = gtk_tool_button_get_label_widget(getHandle());
		GObject obj = getGObjectFromHandle(hndl);
		if (null != obj)
			return (Label)obj;
		return new Label(hndl);
	}

	/***************************************
	 * EVENT HANDLING
	 ****************************************/
	/**
	 * Listeners for handling events
	 */
	private Vector tbListeners = null;

	/**
	 * Register an object to handle dialog events.
	 * @see ToolButtonListener
	 */
	public void addListener(ToolButtonListener listener) {
		// Don't add the listener a second time if it is in the Vector.
		int i = findListener(tbListeners, listener);
		if (i == -1) {
			if (null == tbListeners) {
				evtMap.initialize(this, ToolButtonEvent.Type.CLICKED);
				tbListeners = new Vector();
			}
			tbListeners.addElement(listener);
		}
	}
	
	/**
	 * Removes a listener
	 * @see #addListener(ToolButtonListener)
	 */
	public void removeListener(ToolButtonListener listener) {
		int i = findListener(tbListeners, listener);
		if (i > -1) {
			tbListeners.remove(i);
		}
		if (0 == tbListeners.size()) {
			evtMap.uninitialize(this, ToolButtonEvent.Type.CLICKED);
			tbListeners = null;
		}
	}

	protected void fireToolButtonEvent(ToolButtonEvent event) {
		if (null == tbListeners) {
			return;
		}
		int size = tbListeners.size();
		int i = 0;
		while (i < size) {
			ToolButtonListener tbl = (ToolButtonListener)tbListeners.elementAt(i);
			tbl.toolButtonEvent(event);
			i++;
		}
	}

	private void handleClicked() {
		ToolButtonEvent evt = new ToolButtonEvent(this, ToolButtonEvent.Type.CLICKED);
		fireToolButtonEvent(evt);
	}

	public Class getEventListenerClass(String signal) {
		Class cls = evtMap.getEventListenerClass(signal);
		if (cls == null) cls = super.getEventListenerClass(signal);
		return cls;
	}

	public EventType getEventType(String signal) {
		EventType et = evtMap.getEventType(signal);
		if (et == null) et = super.getEventType(signal);
		return et;
	}

	private static EventMap evtMap = new EventMap();
	static {
		addEvents(evtMap);
	}

	/**
	 * Implementation method to build an EventMap for this widget class.
	 * Not useful (or supported) for application use.
	 */
	private static void addEvents(EventMap anEvtMap) {
		anEvtMap.addEvent("clicked", "handleClicked", ToolButtonEvent.Type.CLICKED, ToolButtonListener.class);
	}

	native static final protected int gtk_tool_button_get_type();
	native static final protected Handle gtk_tool_button_new(Handle icon, String label);
	native static final protected Handle gtk_tool_button_new_from_stock(String stockId);
	native static final protected void gtk_tool_button_set_label(Handle button, String label);
	native static final protected String gtk_tool_button_get_label(Handle button);
	native static final protected void gtk_tool_button_set_use_underline(Handle button, boolean useUnderline);
	native static final protected boolean gtk_tool_button_get_use_underline(Handle button);
	native static final protected void gtk_tool_button_set_stock_id(Handle button, String stockId);
	native static final protected String gtk_tool_button_get_stock_id(Handle button);
	native static final protected void gtk_tool_button_set_icon_widget(Handle button, Handle widget);
	native static final protected Handle gtk_tool_button_get_icon_widget(Handle button);
	native static final protected void gtk_tool_button_set_label_widget(Handle button, Handle widget);
	native static final protected Handle gtk_tool_button_get_label_widget(Handle button);
}
