/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */

package org.gnu.gtk;

import org.gnu.glib.GObject;
import org.gnu.glib.Type;

/**
 * The FileSelection widget lists all the files and directories in
 * the current working directory and enables the user to select one of
 * the file names. There are a number of navigation controls that
 * enable the user to browse through the entire file system. There also
 * are buttons that create directories, delete files, and rename files.
 */
public class FileSelection extends Dialog {
	
	/**
	 * construct a new FileSelection dialog.
	 * 
	 * @param title The string that is displayed in the title bar of the dialog.
	 */
	public FileSelection(String title) {
		setHandle(FileSelection.gtk_file_selection_new(title));
		initializeEventHandlers();
	}
	
	/**
	 * Construct a FileSelection using a handle to a native resource.
	 */
	public FileSelection(int handle) {
	    setHandle(handle);
	    initializeEventHandlers();
	}

	protected void noopInit() {}
	
	/**
	 * Sets the default path for the file requestor.  If filename includes a
	 * directory path the requestor will open with that path set as its 
	 * current working directory.
	 * 
	 * @param filename The default path for the widget.
	 */
	public void setFilename(String filename) {
		FileSelection.gtk_file_selection_set_filename(getHandle(), filename);
	}
	
	/**
	 * Returns the selected filename.
	 * 
	 * @return The file that is selected in the dialog.
	 */
	public String getFilename() {
		return FileSelection.gtk_file_selection_get_filename(getHandle());
	}
	
	/**
	 * Will attempt to match <code>pattern</code> to a valid filename or subdirectory
	 * in the current directory.  If a match can be made the matched filename will appear
	 * in the text entry field in the file selection dialog.  If a partial match can be made the
	 * "Files" list will contain those file names which have been partially matched and the
	 * "Directories" list will list those directories which have been partially matched.
	 * 
	 * @param pattern The pattern to use for matching.
	 */
	public void complete(String pattern) {
		FileSelection.gtk_file_selection_complete(getHandle(), pattern);
	}
	
	/**
	 * Shows the file operation buttons, if they have previously been hidden.  The rest
	 * of the widgets in the dialog will be resized accordingly.
	 */
	public void showFileopButtons() {
		FileSelection.gtk_file_selection_show_fileop_buttons(getHandle());
	}
	
	/**
	 * Hides the file operation buttons that normally appear at the top of the dialog.
	 */
	public void hideFileopButtons() {
		FileSelection.gtk_file_selection_hide_fileop_buttons(getHandle());
	}
	
	/**
	 * Return the OK Button widget for this Dialog.
	 * 
	 * @return The OK Button.
	 */
	public Button getOKButton() {
		int hndl = FileSelection.getOkButton(getHandle());
		GObject obj = retrieveGObject(hndl);
		if (null != obj)
			return (Button)obj;
		return new Button(hndl);
	}
	
	/**
	 * Return the Cancel Button widget for this dialog.
	 * 
	 * @return The Cancel Button.
	 */
	public Button getCancelButton() {
		int hndl = FileSelection.getCancelButton(getHandle());
		GObject obj = retrieveGObject(hndl);
		if (null != obj)
			return (Button)obj;
		return new Button(hndl);
	}
	
	/**
	 * Return the Help Button widget for this dialog.
	 * 
	 * @return The Help Button.
	 */
	public Button getHelpButton() {
		int hndl = FileSelection.getHelpButton(getHandle());
		GObject obj = retrieveGObject(hndl);
		if (null != obj)
			return (Button)obj;
		return new Button(hndl);
	}
	

	/**
	 * Return the directory list for this widget
	 *
	 * @return The Directory List
	 */
	public Widget getDirList(){
		int hndl = getDirList(getHandle());
		if (0 == hndl)
			return null;
		GObject obj = retrieveGObject(hndl);
		if (null != obj)
			return (Widget)obj;
		return new Widget(hndl);
	}

	/**
	 * Returns the the file list for this widget
	 *
	 * @return The File List
	 */
	public Widget getFileList(){
		int hndl = getFileList(getHandle());
		if (0 == hndl)
			return null;
		GObject obj = retrieveGObject(hndl);
		if (null != obj)
			return (Widget)obj;
		return new Widget(hndl);
	}

	/**
	 * Returns the selection entry
	 *
	 * @return selection entry
	 */
	public Widget getSelectionEntry(){
		int hndl = getSelectionEntry(getHandle());
		if (0 == hndl)
			return null;
		GObject obj = retrieveGObject(hndl);
		if (null != obj)
			return (Widget)obj;
		return new Widget(hndl);
	}

	/**
	 * Retrieve the runtime type used by the GLib library.
	 */
	public static Type getType() {
		return new Type(gtk_file_selection_get_type());
	}

    /****************************************
     * BEGINNING OF GENERATED CODE
     ****************************************/
    native static final protected int getDirList (int cptr);
    native static final protected int getFileList (int cptr);
    native static final protected int getSelectionEntry (int cptr);
    native static final protected int getSelectionText (int cptr);
    native static final protected int getMainVbox (int cptr);
    native static final protected int getOkButton (int cptr);
    native static final protected int getCancelButton (int cptr);
    native static final protected int getHelpButton (int cptr);
    native static final protected int getHistoryPulldown (int cptr);
    native static final protected int getHistoryMenu (int cptr);
    native static final protected int getButtonArea (int cptr);
    native static final protected int gtk_file_selection_get_type ();
    native static final protected int gtk_file_selection_new (String title);
    native static final protected void gtk_file_selection_set_filename (int filesel, String filename);
    native static final protected String gtk_file_selection_get_filename (int filesel);
    native static final protected void gtk_file_selection_complete (int filesel, String pattern);
    native static final protected void gtk_file_selection_show_fileop_buttons (int filesel);
    native static final protected void gtk_file_selection_hide_fileop_buttons (int filesel);
    /****************************************
     * END OF GENERATED CODE
     ****************************************/
}

