/* sel-graphs.c
 *
 * Copyright (C) 2004 - 2006 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "sel-graphs.h"
#include <libgda/libgda.h>
#include <glib/gi18n-lib.h>
#include "gnome-db-stock.h"

/*
 *
 * Module for all graphs managed by GdaDict
 *
 */
static void         module_graphs_fill_model (Module *module);
static void         module_graphs_free (Module *module);
static const gchar *module_graphs_col_name (Module *module, guint colno);
static void         module_graphs_model_store_data (Module *module, GtkTreeIter *iter);
static const gchar *module_graphs_render_graph_type (GdaGraph *graph);

Module *
sel_module_graphs_new (GnomeDbSelector *mgsel, gboolean insert_header, 
		       GtkTreeIter *iter, gpointer data)
{
	Module *module;

	module = g_new0 (Module, 1);
	module->selector = mgsel;
	module->fill_model = module_graphs_fill_model;
	module->free = module_graphs_free;
	module->col_name = module_graphs_col_name;
	module->obj_manager = NULL;
	module->model_store_data = module_graphs_model_store_data;
	module->mod_data = NULL;
	module->iter = NULL;
	module->parent_module = NULL;
	module->sub_modules = NULL;

	if (insert_header) {
		GdkPixbuf *pixbuf = NULL;
		GtkTreeModel *model = mgsel->priv->model;

		pixbuf = gnome_db_stock_get_icon_pixbuf (GNOME_DB_STOCK_RELATIONS);
		module->iter = g_new0 (GtkTreeIter, 1);
		gtk_tree_store_append (GTK_TREE_STORE (model), module->iter, iter);
		gtk_tree_store_set (GTK_TREE_STORE (model), module->iter, NAME_COLUMN, 
				    _("Graphs"), PIXBUF_COLUMN, pixbuf, 
				    CONTENTS_COLUMN, CONTENTS_TOP_CATEGORY, 
				    SUB_MODULE_COLUMN, NULL, -1);		
	}
	else {
		if (iter)
			module->iter = gtk_tree_iter_copy (iter);
	}

	return module;
}

static void local_flat_obj_added_cb (GdaDict *dict, GObject *obj, Module *module);
static void local_flat_obj_removed_cb (GdaDict *dict, GObject *obj, Module *module);
static void local_flat_obj_updated_cb (GdaDict *dict, GObject *obj, Module *module);

static GSList *module_graphs_get_objects_list (Module *module);
static void
module_graphs_fill_model (Module *module)
{
	GtkTreeModel *model;
	GdkPixbuf *pixbuf_graph = NULL;

	pixbuf_graph = gnome_db_stock_get_icon_pixbuf_file ("gnome-db-relations_16x16.png");

	/* Module's private data */
	module->mod_data = g_new0 (ModFlatData, 1);
	FLAT_DATA (module)->manager = NULL;
	FLAT_DATA (module)->manager_weak_refed = FALSE;
	FLAT_DATA (module)->fallback_obj_pixbuf = pixbuf_graph;
	FLAT_DATA (module)->get_objects_list = module_graphs_get_objects_list;

	/* Initial model filling */
	model = module->selector->priv->model;
	flat_init_model_fill (module, model);

	/* Signals handlers */
	g_signal_connect (G_OBJECT (module->selector->priv->dict), "object_added",
			  G_CALLBACK (local_flat_obj_added_cb), module);
	g_signal_connect (G_OBJECT (module->selector->priv->dict), "object_removed",
			  G_CALLBACK (local_flat_obj_removed_cb), module);
	g_signal_connect (G_OBJECT (module->selector->priv->dict), "object_updated",
			  G_CALLBACK (local_flat_obj_updated_cb), module);
}

static void
local_flat_obj_added_cb (GdaDict *dict, GObject *obj, Module *module)
{
	if (G_OBJECT_TYPE (obj) == GDA_TYPE_GRAPH)
		flat_obj_added_cb ((GObject *) dict, obj, module);
}

static void
local_flat_obj_removed_cb (GdaDict *dict, GObject *obj, Module *module)
{
	if (G_OBJECT_TYPE (obj) == GDA_TYPE_GRAPH)
		flat_obj_removed_cb ((GObject *) dict, obj, module);
}

static void
local_flat_obj_updated_cb (GdaDict *dict, GObject *obj, Module *module)
{
	if (G_OBJECT_TYPE (obj) == GDA_TYPE_GRAPH)
		flat_obj_updated_cb ((GObject *) dict, obj, module);
}

static GSList *
module_graphs_get_objects_list (Module *module)
{
	return gda_graphs_get_with_type (module->selector->priv->dict, GDA_GRAPH_DB_RELATIONS);
}

static void
module_graphs_free (Module *module)
{
	GSList *list = module->sub_modules;

	/* Free the sub modules */
	while (list) {
		(MODULE (list->data)->free) (MODULE (list->data));
		g_free (list->data);
		list = g_slist_next (list);
	}
	if (module->sub_modules) {
		g_slist_free (module->sub_modules);
		module->sub_modules = NULL;
	}

	/* free this module */
	g_signal_handlers_disconnect_by_func (G_OBJECT (module->selector->priv->dict),
					      G_CALLBACK (local_flat_obj_added_cb), module);
	g_signal_handlers_disconnect_by_func (G_OBJECT (module->selector->priv->dict),
					      G_CALLBACK (local_flat_obj_removed_cb), module);
	g_signal_handlers_disconnect_by_func (G_OBJECT (module->selector->priv->dict),
					      G_CALLBACK (local_flat_obj_updated_cb), module);

	if (module->iter)
		gtk_tree_iter_free (module->iter);

	flat_free_mod_data (module);
	g_free (module->mod_data);
	module->mod_data = NULL;
}


static const gchar *
module_graphs_col_name (Module *module, guint colno)
{
	switch (colno) {
	case 0:
		return _("Graph");
		break;
	case EXTRA1_COLUMN:
		return _("Type");
		break;
	default:
		return NULL;
		break;
	}
}

static void
module_graphs_model_store_data (Module *module, GtkTreeIter *iter)
{
	GObject *obj;
	GtkTreeModel *model;

	model = module->selector->priv->model;
	gtk_tree_model_get (model, iter, OBJ_COLUMN, &obj, -1);

	if (obj && GDA_IS_GRAPH (obj)) {
		const gchar *str1;
		gtk_tree_store_set (GTK_TREE_STORE (model), iter, 
				    EXTRA1_COLUMN, module_graphs_render_graph_type (GDA_GRAPH (obj)),
				    -1);
		str1 = gda_object_get_name (GDA_OBJECT (obj));
		if (!str1 || !(*str1)) {
			gtk_tree_store_set (GTK_TREE_STORE (model), iter, 
				    NAME_COLUMN, _("Graph <no name>"),
				    -1);
		}
	}
}

static const gchar *
module_graphs_render_graph_type (GdaGraph *graph)
{
	switch (gda_graph_get_graph_type (graph)) {
	case GDA_GRAPH_DB_RELATIONS:
		return _("Database relations");
	case GDA_GRAPH_QUERY_JOINS:
		return _("Query joins");
	case GDA_GRAPH_MODELLING:
		return _("Model");
	default:
		g_assert_not_reached ();
	}

	return NULL;
}

