/* $Id: guppi-paths.c,v 1.6 2000/04/25 13:57:11 trow Exp $ */

/*
 * guppi-paths.c
 *
 * Copyright (C) 1999, 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org>
 * and Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include "guppi-paths.h"

const gchar*
guppi_logo_graphic_path(void)
{
  static gchar* path = NULL;

#ifdef GUPPI_PIXMAPS
  if (path == NULL) {
    path = g_strdup_printf("%s/guppi-splash.png", GUPPI_PIXMAPS);
    if (!g_file_exists(path)) {
      g_free(path);
      path = NULL;
    }
  }
#endif

  /* This will work iff you are running an uninstalled guppi out of
     the src/guppi directory. */
  if (path == NULL) {
    path = "../../pixmaps/guppi-splash.png";
    if (!g_file_exists(path))
      path = NULL;
  }

  if (path == NULL)
    g_warning("Couldn't find guppi-splash.png");

  return path;
}

/*****************************************************************************/

/* This code can leak a little memory, but the amount of leakage
   is strictly bounded... */

static gboolean glade_path_initialized = FALSE;
static GHashTable* glade_path_hash = NULL;

static void
guppi_glade_path_add(const gchar* file, const gchar* path)
{
  gpointer data;

  if (glade_path_hash == NULL) {
    glade_path_hash = g_hash_table_new(g_str_hash, g_str_equal);
  }

  data = g_hash_table_lookup(glade_path_hash, file);
  /* Silently ignore collisions... */
  if (data != NULL)
    return;

  g_hash_table_insert(glade_path_hash, (gchar*)file, (gpointer)path);
}

static const gchar*
guppi_glade_path_lookup(const gchar* file)
{
  if (glade_path_hash == NULL)
    return NULL;

  return (const gchar*)g_hash_table_lookup(glade_path_hash, file);
}

static void
guppi_glade_path_scan_dir(const gchar* path_orig, gboolean recursive)
{
  DIR* dir;
  struct dirent* dirent;
  struct stat buf;
  gchar path[1024];
  gchar* str;
  gint i=0, j=0;

  /*
    If we are handed something like
    /home/foo/plug-ins:/bar/plug-ins:/usr/local/plug-ins
    we should do the right thing and load from all three.
  */
  while (path_orig[i]) {
    if (path_orig[i] == ':') {
      path[j] = '\0';
      guppi_glade_path_scan_dir(path, recursive);
      j=0;
    } else {
      path[j] = path_orig[i];
      ++j;
    }
    ++i;
  }
  path[j]='\0';

  dir = opendir(path);
  if (dir == NULL)
    return;

  errno = 0;
  
  while ((dirent = readdir(dir)) != NULL) {

    if (recursive &&
        strcmp(dirent->d_name, ".") && 
        strcmp(dirent->d_name, "..")) {
      str = g_strdup_printf("%s/%s", path, dirent->d_name);
      if (stat(str, &buf) < 0) 
        g_message("stat() on %s failed", str);
      else if (S_ISDIR(buf.st_mode) && recursive) 
        guppi_glade_path_scan_dir(str, recursive);
      g_free(str);
    }

    if (!strcmp(g_extension_pointer(dirent->d_name), "glade")) {
      str = g_strdup_printf("%s/%s", path, dirent->d_name);
      guppi_glade_path_add(g_filename_pointer(str), str);
    }

    errno = 0;
  }
  if (errno) {
    g_message("An error occured while reading %s", path);
  }

  closedir(dir);


}

const gchar*
guppi_glade_path(const gchar* filename)
{
  const gchar* path;

  if (!glade_path_initialized) {
#ifdef GUPPI_GLADE_PATH
    guppi_glade_path_scan_dir(GUPPI_GLADE_PATH, TRUE);
#endif
    guppi_glade_path_scan_dir("../../plug-ins", TRUE);
    glade_path_initialized = TRUE;
  }

  path = guppi_glade_path_lookup(filename);
  
  if (path == NULL)
    g_warning("Couldn't file \"%s\"", filename);

  return path;
}

/*****************************************************************************/

const gchar*
gnome_guile_repl_path(void)
{
  static gchar* path = NULL;

  if (path == NULL) {
    path = gnome_is_program_in_path("gnome-guile-repl");
  }

  /* Maybe we built guppi with some funny prefix and $(prefix)/bin isn't
     in our path. */
  if (path == NULL) {
    path = g_strdup_printf("%s/gnome-guile-repl", GUPPI_PREFIX);
    if (!g_file_exists(path)) {
      g_free(path);
      path = NULL;
    }
  }

  /* Try the hard-wired path, in case we haven't installed yet and
     are running out of src/guppi. */
  if (path == NULL) {
    path = "../guile-server/gnome-guile-repl";
    if (!g_file_exists(path))
      path = NULL;
  }
  
  if (path == NULL) 
    gtk_widget_show_all(gnome_error_dialog(_("The program \"gnome-guile-repl\" is not in your $PATH.\nWithout it, you cannot access the guile terminal.")));
  
  return path;
}




/* $Id: guppi-paths.c,v 1.6 2000/04/25 13:57:11 trow Exp $ */
