/* $Id: guppi-layout-grid-allocate.c,v 1.2 2000/05/03 17:06:51 trow Exp $ */

/*
 * guppi-layout-grid-allocate.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org>.

 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <math.h>
#include "guppi-layout-grid-allocate.h"

static double
allocation_size(double t,
		GuppiLayoutGridAllocation* al,
		gint N)
{
  gint i;
  double x, sum=0;

  g_return_val_if_fail(al != NULL, 0);
  g_return_val_if_fail(N > 0, 0);

  for (i=0; i<N; ++i) {

    x = MAX(0, al[i].request * (1 + t*al[i].hunger));

    if (al[i].min_size > 0)
      x = MAX(x, al[i].min_size);

    if (al[i].max_size > 0)
      x = MIN(x, al[i].max_size);

    al[i].allocation = x;
    sum += x;
  }
  return sum;
}

void
guppi_layout_grid_calculate_allocation(double size,
				       GuppiLayoutGridAllocation* al,
				       gint N)
{
  const double epsilon = 0.01;
  gint i, count, nonzero=0;
  double a, b, mid, curr_size;
  double total_req=0, max_req=0;
  gboolean max_problem=FALSE;

  g_return_if_fail(size > 0);
  g_return_if_fail(al != NULL);
  g_return_if_fail(N > 0);

  /* First, we need to replace any requests of 0 with a sensible default. */
  for (i=0; i<N; ++i) {
    total_req += al[i].request;
    if (al[i].request > 0)
      ++nonzero;
  }
  for (i=0; i<N; ++i) {

    if (al[i].request == 0) {

      if (nonzero > 0)
	curr_size = total_req / (N-nonzero);
      else
	curr_size = size / N;
      
      if (al[i].min_size > 0)
	curr_size = MAX(curr_size,  al[i].min_size);
      if (al[i].max_size > 0)
	curr_size = MIN(curr_size, al[i].max_size);

      al[i].request = curr_size;
    }
  }

  /* Now we need to check that we don't have a problem with maximums */
  max_problem = TRUE;
  for (i=0; i<N && max_problem; ++i) {
    if (al[i].max_size == 0 && al[i].hunger > 0)
      max_problem = FALSE;
    else
      max_req += al[i].hunger > 0 ? al[i].max_size : al[i].request;
  }
  if (max_problem && max_req < size) {
    for (i=0; i<N; ++i)
      if (al[i].hunger > 0)
	al[i].max_size *= 1.1 * size / max_req;
      else
	al[i].request *= size / max_req;
  }


  /* Find initial bounding values */
  a = -1;
  count = 100;
  while (allocation_size(a, al, N) > size && count > 0) {
    a *= 2;
    --count;
  }
  g_assert(count != 0);

  b = 1;
  count = 100;
  while (allocation_size(b, al, N) < size && count > 0) {
    b *= 2;
    --count;
  }
  g_assert(count != 0);

  /* Repeatedly bisect the interval to find the optimal allocation.
     If allocation_size() isn't monotone increasing, there is going
     to be trouble. */
  count=0;
  while (count < 100) {
    mid = (a+b)/2;
    curr_size = allocation_size(mid, al, N);

    if (fabs(curr_size - size) < epsilon)
      return;
    
    if (curr_size < size)
      a = mid;
    else
      b = mid;

    ++count;
  }

  g_assert_not_reached();
}


/* $Id: guppi-layout-grid-allocate.c,v 1.2 2000/05/03 17:06:51 trow Exp $ */
