/* $Id: guppi.c,v 1.23 2000/05/01 04:41:37 trow Exp $ */

/*
 * guppi.c
 *
 * Copyright (C) 1999 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org>
 * and Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <gnome.h>
#include <glade/glade.h>
#include <signal.h>
#include <time.h>
#include <guppi-data-tree-view.h>
#include "corba_guppi.h"
#include <guppi-useful.h>
#include <guppi-data-init.h>
#include <guppi-stat-init.h>
#include <guppi-plot-init.h>
#include <guppi-plot-plug-in.h>
#ifdef USING_BONOBO
#include <guppi-bonobo.h>
#endif
#include "app_scm.h"
#include "term.h"
#include "file-open.h"
#include "plug-ins.h"
#include "guile-load.h"
#include "guppi-app.h"
#include "guppi.h"

static GnomeApp* guppi_app = NULL;

static void
dummy_uib_signal_connector(GnomeUIInfo* foo, gchar* bar, 
			   GnomeUIBuilderData* foobar)
{ }

void
guppi_add_menubar_item(const gchar* path, const gchar* name)
{
  GnomeUIInfo item[] = {
    GNOMEUIINFO_SUBTREE(NULL, NULL), 
    GNOMEUIINFO_END
  };

  GnomeUIBuilderData uibdata = { dummy_uib_signal_connector,
				 NULL, FALSE, NULL, NULL };

  GtkWidget* parent;
  gint pos;

  g_return_if_fail(name != NULL);

  item[0].label = (gchar*)name;
  /* We need a pointer to a GNOMEUIINFO_END to put in as the contents
     of the empty item we just built, and item[1] is just sitting
     there looking very tempting... */
  item[0].moreinfo = &item[1];

  if (path == NULL || *path == '\0')
    parent = gnome_app_find_menu_pos(guppi_app->menubar, "Help", &pos);
  else
    parent = gnome_app_find_menu_pos(guppi_app->menubar, path, &pos);

  g_return_if_fail(parent != NULL);

  gnome_app_fill_menu_custom(GTK_MENU_SHELL(parent), item, &uibdata,
			     guppi_app->accel_group, TRUE, pos-1);
}

void
guppi_add_menu(const gchar* path,
		    GnomeUIInfo* menuinfo)
{
  gnome_app_insert_menus(guppi_app, path, menuinfo);
}


/****************************************************************************/

static void
load_rc_files(void)
{
  gchar buffer[1024];
  const gint buffer_len = 1024;

  g_snprintf(buffer, buffer_len, "%s/.guppirc", gnome_util_user_home());
  if (g_file_exists(buffer)) {
    guppi_splash_message(_("Loading .guppirc"));
    guppi_safe_load(buffer);
  }

  strncpy(buffer, "./.guppilocalrc", buffer_len);
  if (g_file_exists(buffer)) {
    guppi_splash_message(_("Loading .guppilocalrc"));
    guppi_safe_load(buffer);
  }
}

/*
  On the first try, we attempt a graceful exit.
  If a second attempt comes in soon enough, we give up and abort.
*/
static void
exit_signal_handler(gint x)
{
  static time_t then = 0;
  time_t now;
  
  time(&now);
  if (now - then > 10) {
    guppi_exit();
    then = now;
  } else {
    guppi_abort();
  }
}

static void
abort_signal_handler(gint x)
{
  guppi_abort();
}

static void
real_main(void* closure, int argc, char* argv[])
{
  gchar* opt_script_filename = NULL;
  gint opt_batch_flag = 0;
  struct poptOption options[] = {
    { "script", 's', POPT_ARG_STRING, &opt_script_filename, 0, 
      N_("Load and evaluate code from FILENAME"), N_("FILENAME") },
    { "batch", 'b', POPT_ARG_NONE, &opt_batch_flag, 0,
      N_("Run in batch mode"), NULL },
    { NULL, '\0', 0, NULL, 0, NULL, NULL }
  };
  poptContext pctx;
  
  struct sigaction xsh, ash;

  /* Trap ctrl-c from the starting console */
  xsh.sa_handler = exit_signal_handler;
  xsh.sa_flags = 0;

  ash.sa_handler = abort_signal_handler;
  ash.sa_flags = 0;

  sigaction(SIGTERM, &ash, NULL);
  sigaction(SIGINT, &xsh, NULL);
  sigaction(SIGHUP, &ash, NULL);

  if (!opt_batch_flag)
    guppi_splash_create();

  /*
    guppi_corba_init() calls gnome_CORBA_init(), which in turn calls
    gnome_init().
  */
  guppi_splash_message(_("Initializing CORBA"));
  guppi_corba_init(&argc, argv, options, 0, &pctx);
  guppi_exit_connect_shutdown_func(guppi_corba_shutdown, NULL);

  /* We have many, many libraries and misc. things to initialize */
  guppi_splash_message(_("Initializing libglade"));
  glade_gnome_init();
  
  guppi_splash_message(_("Initializing Utility Library"));
  guppi_useful_init();

#ifdef USING_BONOBO
  guppi_splash_message(_("Initializing Bonobo"));
  guppi_bonobo_init();
#endif

  guppi_splash_message(_("Initializing Statistics Library"));
  guppi_stat_init();

  guppi_splash_message(_("Initializing Data Library"));
  guppi_data_init();

  guppi_splash_message(_("Initializing Plot Library"));
  guppi_plot_init();

  guppi_splash_message(_("Initializing GuppiAppGuile"));
  guppi_app_scm_init();

  guppi_splash_message(_("Loading Plug-Ins"));
  guppi_plug_in_load_all();

  if (!opt_batch_flag) {
    guppi_splash_message(_("Building Main Guppi Window"));
    guppi_app = GNOME_APP(guppi_app_new());
  }

  /* We load our rc files after building our app window so that we can
     freely monkey with the app's UI. */
  load_rc_files();

  guppi_splash_destroy();

  if (opt_script_filename != NULL) {
    g_message("script filename: %s", opt_script_filename);
  }

  if (opt_batch_flag) {
    guppi_exit();
  } else {
    gtk_widget_show_all(GTK_WIDGET(guppi_app));
    gtk_main();
  }
}

int
main(int argc, char* argv[])
{
  /*
    We need to start up enough of the toolkit to let us put up our
     spash screen.  We'll init the rest of gnome in just a little while.
  */
  gtk_init(&argc, &argv);
  gdk_imlib_init();

  bindtextdomain(PACKAGE, GNOMELOCALEDIR);
  textdomain(PACKAGE);

  g_message(_("Booting Guile"));

  scm_boot_guile(argc, argv, real_main, NULL);
  g_assert_not_reached();
  return 1;
}


/* $Id: guppi.c,v 1.23 2000/05/01 04:41:37 trow Exp $ */
