/* $Id: file-open.c,v 1.8 2000/04/25 13:57:06 trow Exp $ */

/*
 * file-open.c
 *
 * Copyright (C) 1999 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org>
 * and Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <gnome.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include <guppi-data-tree.h>
#include <guppi-file.h>
#include <guppi-data-importer.h>
#include <guppi-data-importer-plug-in.h>

#include "file-open.h"
#include "guppi-app.h"

#define CANNOT_OPEN_FILE _("Can't open file \"%s\"")
#define CANNOT_OPEN_FILE_WITH_REASON _("Can't open file \"%s\": %s")

static void
do_import(GuppiDataImporter* imp)
{
  GuppiData** dv;
  gint i;
  GuppiApp* app;
  const gchar* filename = NULL;
  gboolean activity=FALSE;

  dv = guppi_data_importer_import(imp);
  if (dv == NULL) {
    gtk_widget_show_all(gnome_warning_dialog(_("No data imported.")));
    return;
  }

  /* Add the imported data to the tree, handling ref counting */
  for (i=0; dv[i] != NULL; ++i) {
    guppi_data_tree_add(guppi_data_tree_main(), dv[i]);
    gtk_object_unref(GTK_OBJECT(dv[i]));
    activity = TRUE;
  }
  if (activity) {
    app = GUPPI_APP(gtk_object_get_data(GTK_OBJECT(imp), "app"));
    filename = guppi_data_importer_filename(imp);
    if (filename)
      guppi_app_add_recently_visited_file(app, filename);
  }
    
  g_free(dv);
  gtk_object_unref(GTK_OBJECT(imp));
}

/****************************************************************************/

static void
on_import_config_clicked(GnomeDialog* dialog, gint n, gpointer user_data)
{
  gnome_dialog_close(dialog);
  if (n == 0)
    do_import(GUPPI_DATA_IMPORTER(user_data));
  else
    gtk_object_unref(GTK_OBJECT(user_data));
}

static void
do_import_config(GuppiDataImporterPlugIn* dipi, GuppiFile* file, GuppiApp* app)
{
  GuppiDataImporter* imp;
  GtkWidget* imp_config;
  GtkWidget* dialog;
  gchar* filename;

  g_return_if_fail(dipi != NULL);
  g_return_if_fail(file != NULL);

  g_return_if_fail(dipi->construct != NULL);
  imp = (dipi->construct)();
  g_return_if_fail(imp != NULL);

  gtk_object_set_data(GTK_OBJECT(imp), "app", app);

  filename = g_strdup(guppi_file_filename(file));
  if (filename) {
    g_free(app->most_recent_open_directory);
    *(g_filename_pointer(filename)) = '\0';
    app->most_recent_open_directory = filename;
  }

  guppi_data_importer_set_source(imp, file);

  imp_config = NULL;
  if (dipi->state_edit_widget)
    imp_config = (dipi->state_edit_widget)(imp);
  if (imp_config) {
    dialog = gnome_dialog_new(_("Configure Import"),
			      GNOME_STOCK_BUTTON_OK,
			      GNOME_STOCK_BUTTON_CANCEL,
			      NULL);
    gtk_window_set_policy(GTK_WINDOW(dialog), FALSE, TRUE, FALSE);
    gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox),
		       imp_config, TRUE, TRUE, 2);
    gtk_signal_connect(GTK_OBJECT(dialog),
		       "clicked",
		       GTK_SIGNAL_FUNC(on_import_config_clicked),
		       imp);
    gtk_widget_show_all(dialog);
  } else {
    do_import(imp);
  }
}

/****************************************************************************/

struct import_gadget {
  GuppiFile* file;
  GuppiDataImporterPlugIn* selected_plug_in;
  GuppiApp* app;
};

static void
on_radio_toggled(GtkToggleButton* b, struct import_gadget* gadget)
{
  if (gtk_toggle_button_get_active(b)) {
    gadget->selected_plug_in =
      GUPPI_DATA_IMPORTER_PLUG_IN(gtk_object_get_user_data(GTK_OBJECT(b)));
  }
}

static void
on_plugin_select_dialog_clicked(GnomeDialog* dialog, gint n, 
				struct import_gadget* gadget)
{
  GuppiDataImporterPlugIn* dipi = gadget->selected_plug_in;
  GuppiFile* file = gadget->file;
  gtk_widget_destroy(GTK_WIDGET(dialog));
  if (n == 0 && dipi) 
    do_import_config(dipi, file, gadget->app);
  
}

static void
on_plugin_select_dialog_close(GnomeDialog* dialog, gpointer gadget)
{
  g_free(gadget);
}

void
file_open_ask_about_import_type(const gchar* filename,
				gboolean force_ask,
				double threshold,
				GuppiApp* app)
{
  GuppiFile* file;
  const gint buffer_size = 256;
  gchar buffer[256];
  gint peek_size;
  GList* assessment;
  GList* iter;
  GtkWidget* dialog;
  GtkWidget* rb;
  GtkWidget* first_rb;
  GuppiDataImporterAssessment* impass;
  GuppiPlugIn* pi;
  GuppiDataImporterPlugIn* dipi;
  struct import_gadget* gadget = NULL;

  g_return_if_fail(filename != NULL);

  file = guppi_file_open(filename);
  if (file == NULL) {
    g_snprintf(buffer, buffer_size, CANNOT_OPEN_FILE, filename);
    gtk_widget_show_all(gnome_error_dialog(buffer));
    return;
  }

  peek_size = guppi_file_peek(file, buffer, buffer_size);
  assessment = guppi_data_importer_plug_in_assess(filename,
						  peek_size ? buffer : NULL,
						  peek_size);

  if (assessment == NULL) {
    g_snprintf(buffer, buffer_size,
	       CANNOT_OPEN_FILE_WITH_REASON,
	       filename, _("Unknown file type"));
    gtk_widget_show_all(gnome_error_dialog(buffer));
    return;
  }

  /* If only one plug-in seems possible, don't ask --- just use it. */
  if (g_list_next(assessment) == NULL) {
    impass = (GuppiDataImporterAssessment*)assessment->data;
    do_import_config(impass->plug_in, file, app);
    g_free(assessment->data);
    g_list_free(assessment);
    return;
  }

  dialog = gnome_dialog_new(_("Select Import Method"),
			    GNOME_STOCK_BUTTON_OK,
			    GNOME_STOCK_BUTTON_CANCEL,
			    NULL);

  gadget = g_new(struct import_gadget, 1);
  gadget->file = file;
  gadget->selected_plug_in = NULL;
  gadget->app = app;

  first_rb = rb = NULL;
  for (iter = assessment; iter != NULL; iter = g_list_next(iter)) {
    impass = (GuppiDataImporterAssessment*)iter->data;
    if (impass->confidence >= threshold) {
      dipi = impass->plug_in;
      pi = GUPPI_PLUG_IN(dipi);

      if (rb == NULL) {
	first_rb = rb = gtk_radio_button_new_with_label(NULL,
							pi->external_name);
	gadget->selected_plug_in = dipi;
      } else {
	rb = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(rb),
							 pi->external_name);
      }
      gtk_object_set_user_data(GTK_OBJECT(rb), dipi);
      gtk_signal_connect(GTK_OBJECT(rb),
			 "toggled",
			 GTK_SIGNAL_FUNC(on_radio_toggled),
			 gadget);
      gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), 
			 rb, TRUE, TRUE, 0);
    }
  }

  if (first_rb)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(first_rb), TRUE);

  gtk_signal_connect(GTK_OBJECT(dialog),
		     "clicked",
		     GTK_SIGNAL_FUNC(on_plugin_select_dialog_clicked),
		     gadget);
  gtk_signal_connect(GTK_OBJECT(dialog),
		     "close",
		     GTK_SIGNAL_FUNC(on_plugin_select_dialog_close),
		     gadget);
  gtk_widget_show_all(dialog);
  
  
  /* Deallocate list */
  for(iter = assessment; iter != NULL; iter = g_list_next(iter))
    g_free(iter->data);
  g_list_free(assessment);

  
}

/****************************************************************************/


static void
on_file_ok_clicked(GtkButton* b, GtkFileSelection* fs)
{
  gchar* filename;
  struct stat buf;
  GuppiApp* app;

  filename = gtk_file_selection_get_filename(fs);
  
  if (stat(filename, &buf)) {
    filename = g_strdup_printf(CANNOT_OPEN_FILE, filename);
    gtk_widget_show_all(gnome_warning_dialog(filename));
    g_free(filename);
    gtk_widget_destroy(GTK_WIDGET(fs));
  } else if (S_ISDIR(buf.st_mode)) {
    filename = g_strdup_printf("%s/", filename);
    gtk_file_selection_set_filename(fs, filename);
    g_free(filename);
  } else {
    app = GUPPI_APP(gtk_object_get_data(GTK_OBJECT(fs), "app"));
    filename = g_strdup(filename);
    gtk_widget_destroy(GTK_WIDGET(fs));
    file_open_ask_about_import_type(filename, FALSE,
				    DEFAULT_FILE_OPEN_THRESHOLD,
				    app);

    g_free(filename);
  }
}

void
file_open(GtkWidget* w, GuppiApp* app)
{
  GtkWidget* file_selector;

  file_selector = gtk_file_selection_new(_("Select File To Open"));
  gtk_object_set_data(GTK_OBJECT(file_selector), "app", app);

  gtk_file_selection_hide_fileop_buttons(GTK_FILE_SELECTION(file_selector));

  if (app->most_recent_open_directory)
    gtk_file_selection_set_filename(GTK_FILE_SELECTION(file_selector),
				    app->most_recent_open_directory);

  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(file_selector)->ok_button),
		     "clicked",
                     GTK_SIGNAL_FUNC(on_file_ok_clicked),
		     file_selector);

  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(file_selector)->cancel_button),
			    "clicked",
			    GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    GTK_OBJECT(file_selector));
  
  gtk_widget_show_all(GTK_WIDGET(file_selector));
}

/* $Id: file-open.c,v 1.8 2000/04/25 13:57:06 trow Exp $ */
