/* This is -*- C -*- */
/* $Id: guppi-seq-scalar.c,v 1.4 2000/03/04 18:51:36 trow Exp $ */

/*
 * guppi-seq-scalar.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <math.h>
#include "guppi-seq-scalar.h"
#include "guppi-seq-scalar-impl.h"
#include "guppi-seq-boolean-impl.h"
#include "guppi-seq-scalar-core-impl.h"

static GtkObjectClass* parent_class = NULL;

enum {
  ARG_0
};

static void
guppi_seq_scalar_get_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_seq_scalar_set_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_seq_scalar_destroy(GtkObject* obj)
{
  if (parent_class->destroy)
    parent_class->destroy(obj);
}

static void
guppi_seq_scalar_finalize(GtkObject* obj)
{
  if (parent_class->finalize)
    parent_class->finalize(obj);
}

static void
guppi_seq_scalar_class_init(GuppiSeqScalarClass* klass)
{
  GtkObjectClass* object_class = (GtkObjectClass*)klass;
  GuppiDataClass* data_class = GUPPI_DATA_CLASS(klass);

  parent_class = gtk_type_class(GUPPI_TYPE_SEQ);

  data_class->default_impl = GUPPI_TYPE_SEQ_SCALAR_CORE_IMPL;
  data_class->type_name = _("Scalar Sequence");

  object_class->get_arg = guppi_seq_scalar_get_arg;
  object_class->set_arg = guppi_seq_scalar_set_arg;
  object_class->destroy = guppi_seq_scalar_destroy;
  object_class->finalize = guppi_seq_scalar_finalize;

}

static void
guppi_seq_scalar_init(GuppiSeqScalar* obj)
{

}

GtkType
guppi_seq_scalar_get_type(void)
{
  static GtkType guppi_seq_scalar_type = 0;
  if (!guppi_seq_scalar_type) {
    static const GtkTypeInfo guppi_seq_scalar_info = {
      "GuppiSeqScalar",
      sizeof(GuppiSeqScalar),
      sizeof(GuppiSeqScalarClass),
      (GtkClassInitFunc)guppi_seq_scalar_class_init,
      (GtkObjectInitFunc)guppi_seq_scalar_init,
      NULL, NULL, (GtkClassInitFunc)NULL
    };
    guppi_seq_scalar_type = gtk_type_unique(GUPPI_TYPE_SEQ, &guppi_seq_scalar_info);
  }
  return guppi_seq_scalar_type;
}

GuppiData*
guppi_seq_scalar_new(void)
{
  return GUPPI_DATA(gtk_type_new(guppi_seq_scalar_get_type()));
}

/*****************************************************************************/

double
guppi_seq_scalar_get(const GuppiSeqScalar* seq, gint i)
{
  const GuppiSeqScalarImpl* impl;
  GuppiSeqScalarImplClass* impl_class;

  g_return_val_if_fail(seq != NULL, 0);
  g_return_val_if_fail(guppi_seq_in_bounds(GUPPI_SEQ(seq), i), 0);

  impl = GUPPI_SEQ_SCALAR_IMPL(guppi_data_impl(GUPPI_DATA(seq)));
  impl_class = GUPPI_SEQ_SCALAR_IMPL_CLASS(GTK_OBJECT(impl)->klass);

  g_assert(impl_class->get);
  return (impl_class->get)(impl, i);
}

void
guppi_seq_scalar_set(GuppiSeqScalar* seq, gint i, double x)
{
  GuppiSeqScalarImpl* impl;
  GuppiSeqScalarImplClass* impl_class;

  g_return_if_fail(seq != NULL);
  g_return_if_fail(guppi_data_can_change(GUPPI_DATA(seq)));
  g_return_if_fail(guppi_seq_in_bounds(GUPPI_SEQ(seq), i));

  impl = GUPPI_SEQ_SCALAR_IMPL(guppi_data_impl(GUPPI_DATA(seq)));
  impl_class = GUPPI_SEQ_SCALAR_IMPL_CLASS(GTK_OBJECT(impl)->klass);

  g_assert(impl_class->set);
  (impl_class->set)(impl, i, x);

  guppi_data_changed(GUPPI_DATA(seq));
}

void
guppi_seq_scalar_prepend(GuppiSeqScalar* seq, double x)
{
  gint first;
  first = guppi_seq_min_index(GUPPI_SEQ(seq));
  guppi_seq_scalar_insert(seq, first, x);
}

void
guppi_seq_scalar_append(GuppiSeqScalar* seq, double x)
{
  gint last;
  last = guppi_seq_max_index(GUPPI_SEQ(seq));
  guppi_seq_scalar_insert(seq, last+1, x);
}

void
guppi_seq_scalar_append_many(GuppiSeqScalar* seq, const double* x, gsize n)
{
  gint last;
  g_return_if_fail(seq != NULL);
  g_return_if_fail(x != NULL);
  if (n == 0)
    return;
  last = guppi_seq_max_index(GUPPI_SEQ(seq));
  guppi_seq_scalar_insert_many(seq, last+1, x, n);
}

void
guppi_seq_scalar_insert(GuppiSeqScalar* seq, gint i, double x)
{
  GuppiSeqScalarImpl* impl;
  GuppiSeqScalarImplClass* impl_class;

  g_return_if_fail(seq != NULL);
  g_return_if_fail(guppi_data_can_change(GUPPI_DATA(seq)));

  impl = GUPPI_SEQ_SCALAR_IMPL(guppi_data_impl(GUPPI_DATA(seq)));
  impl_class = GUPPI_SEQ_SCALAR_IMPL_CLASS(GTK_OBJECT(impl)->klass);

  g_assert(impl_class->insert);
  (impl_class->insert)(impl, i, x);
  guppi_data_changed(GUPPI_DATA(seq));
}

void
guppi_seq_scalar_insert_many(GuppiSeqScalar* seq, gint i,
			     const double* x, gsize n)
{
  GuppiSeqScalarImpl* impl;
  GuppiSeqScalarImplClass* impl_class;
  gint j;

  g_return_if_fail(seq != NULL);
  g_return_if_fail(guppi_data_can_change(GUPPI_DATA(seq)));
  g_return_if_fail(x != NULL);

  if (n == 0)
    return;

  impl = GUPPI_SEQ_SCALAR_IMPL(guppi_data_impl(GUPPI_DATA(seq)));
  impl_class = GUPPI_SEQ_SCALAR_IMPL_CLASS(GTK_OBJECT(impl)->klass);

  guppi_seq_size_hint(GUPPI_SEQ(seq), guppi_seq_size(GUPPI_SEQ(seq)) + n);

  if (impl_class->insert_many) {
    (impl_class->insert_many)(impl, i, x, n);
  } else {
    g_assert(impl_class->insert);
    for (j=0; j<n; ++j)
      (impl_class->insert)(impl, i+j, x[j]);
  }

  guppi_data_changed(GUPPI_DATA(seq));
}

static gint
do_range_query(const GuppiSeqScalar* seq,
	       GuppiSeqBoolean* bseq,
	       double min, double max,
	       gboolean do_and)
{
  const GuppiSeqScalarImpl* impl;
  const GuppiSeqScalarImplClass* impl_class;
  GuppiSeqBooleanImpl* b_impl;
  GuppiSeqBooleanImplClass* b_impl_class;

  gint hits;

  gint i, i0, i1;
  double x;
  gint count=0;
  gboolean st;

  if (min > max) {
    x = min;
    min = max;
    max = x;
  }

  g_return_val_if_fail(seq != NULL, 0);
  g_return_val_if_fail(bseq != NULL, 0);

  impl = GUPPI_SEQ_SCALAR_IMPL(guppi_data_impl(GUPPI_DATA(seq)));
  impl_class = GUPPI_SEQ_SCALAR_IMPL_CLASS(GTK_OBJECT(impl)->klass);

  if (impl_class->range_query) {
    hits = (impl_class->range_query)(impl, bseq, min, max, do_and);
    if (hits >= 0)
      return hits;
  }

  b_impl = GUPPI_SEQ_BOOLEAN_IMPL(guppi_data_impl(GUPPI_DATA(bseq)));
  b_impl_class = GUPPI_SEQ_BOOLEAN_IMPL_CLASS(GTK_OBJECT(b_impl)->klass);

  if (b_impl_class->range_query) {
    hits = (b_impl_class->range_query)(seq, b_impl, min, max, do_and);
    if (hits >= 0)
      return hits;
  }

  guppi_seq_common_bounds(GUPPI_SEQ(seq), GUPPI_SEQ(bseq), &i0, &i1);

  if (do_and) {
    for (i=i0; i<=i1; ++i) {
      st = guppi_seq_boolean_get(bseq, i);
      if (st) {
	x = guppi_seq_scalar_get(seq, i);
	if (x < min || max < x)
	  guppi_seq_boolean_set(bseq, i, FALSE);
	else
	  ++count;
      }
    }
  } else {
    for (i=i0; i<=i1; ++i) {
      x = guppi_seq_scalar_get(seq, i);
      st = min <= x && x <= max;
      guppi_seq_boolean_set(bseq, i, st);
      if (st) ++count;
    }
  }

  return count;
}

GuppiSeqBoolean*
guppi_seq_scalar_range_query(const GuppiSeqScalar* seq, double min, double max)
{
  GuppiSeqBoolean* bseq;

  if (min > max) {
    double x = min;
    min = max;
    max = x;
  }

  g_return_val_if_fail(seq != NULL, NULL);

  bseq = GUPPI_SEQ_BOOLEAN(guppi_seq_boolean_new_aligned(GUPPI_SEQ(seq)));
  do_range_query(seq, bseq, min, max, FALSE);

  return bseq;
}

gint
guppi_seq_scalar_in_place_range_query(const GuppiSeqScalar* seq,
				      GuppiSeqBoolean* bseq,
				      double min, double max)
{
  return do_range_query(seq, bseq, min, max, FALSE);
}

gint
guppi_seq_scalar_bitwise_and_range_query(const GuppiSeqScalar* seq,
					 GuppiSeqBoolean* bseq,
					 double min, double max)
{
  return do_range_query(seq, bseq, min, max, TRUE);
}

double
guppi_seq_scalar_min(const GuppiSeqScalar* seq)
{
  const GuppiSeqScalarImpl* impl;
  GuppiSeqScalarImplClass* impl_class;
  double x;

  g_return_val_if_fail(seq != NULL, 0);
  g_return_val_if_fail(guppi_seq_nonempty(GUPPI_SEQ(seq)), 0);

  impl = GUPPI_SEQ_SCALAR_IMPL(guppi_data_impl(GUPPI_DATA(seq)));
  impl_class = GUPPI_SEQ_SCALAR_IMPL_CLASS(GTK_OBJECT(impl)->klass);

  if (impl_class->range) {

    (impl_class->range)(impl, &x, NULL);

  } else {
    gint i0 = guppi_seq_min_index(GUPPI_SEQ(seq));
    gint i1 = guppi_seq_max_index(GUPPI_SEQ(seq));
    gint i;
    double y;

    x = guppi_seq_scalar_get(seq, i0);
    for (i=i0+1; i<=i1; ++i) {
      y = guppi_seq_scalar_get(seq, i);
      if (y < x) x = y;
    }
  }

  return x;
}

double
guppi_seq_scalar_max(const GuppiSeqScalar* seq)
{
  const GuppiSeqScalarImpl* impl;
  GuppiSeqScalarImplClass* impl_class;
  double x;

  g_return_val_if_fail(seq != NULL, 0);
  g_return_val_if_fail(guppi_seq_nonempty(GUPPI_SEQ(seq)), 0);

  impl = GUPPI_SEQ_SCALAR_IMPL(guppi_data_impl(GUPPI_DATA(seq)));
  impl_class = GUPPI_SEQ_SCALAR_IMPL_CLASS(GTK_OBJECT(impl)->klass);

  if (impl_class->range) {

    (impl_class->range)(impl, NULL, &x);

  } else {
    gint i0 = guppi_seq_min_index(GUPPI_SEQ(seq));
    gint i1 = guppi_seq_max_index(GUPPI_SEQ(seq));
    gint i;
    double y;

    x = guppi_seq_scalar_get(seq, i0);
    for (i=i0+1; i<=i1; ++i) {
      y = guppi_seq_scalar_get(seq, i);
      if (y > x) x = y;
    }
  }

  return x;
}

double
guppi_seq_scalar_sum(const GuppiSeqScalar* seq)
{
  const GuppiSeqScalarImpl* impl;
  GuppiSeqScalarImplClass* impl_class;

  g_return_val_if_fail(seq != NULL, 0);
  if (guppi_seq_empty(GUPPI_SEQ(seq)))
    return 0;
  
  impl = GUPPI_SEQ_SCALAR_IMPL(guppi_data_impl(GUPPI_DATA(seq)));
  impl_class = GUPPI_SEQ_SCALAR_IMPL_CLASS(GTK_OBJECT(impl)->klass);

  if (impl_class->sum) {
    return (impl_class->sum)(impl);
  } else {
    gint i0 = guppi_seq_min_index(GUPPI_SEQ(seq));
    gint i1 = guppi_seq_max_index(GUPPI_SEQ(seq));
    gint i;
    double y = 0;

    for (i=i0; i<=i1; ++i) 
      y += guppi_seq_scalar_get(seq, i);

    return y;
  }
}

double
guppi_seq_scalar_mean(const GuppiSeqScalar* seq)
{
  double sum;
  gsize n;

  g_return_val_if_fail(seq != NULL, 0);

  n = guppi_seq_size(GUPPI_SEQ(seq));
  g_return_val_if_fail(n>0, 0);
  sum = guppi_seq_scalar_sum(seq);

  return sum/n;
}

double
guppi_seq_scalar_var(const GuppiSeqScalar* seq)
{
  const GuppiSeqScalarImpl* impl;
  GuppiSeqScalarImplClass* impl_class;
  
  gconstpointer ptr;
  gint stride;

  double x, om, mean=0, sumsq=0;
  gint i, i0, i1, count=0;

  g_return_val_if_fail(seq != NULL, 0);
  
  impl = GUPPI_SEQ_SCALAR_IMPL(guppi_data_impl(GUPPI_DATA(seq)));
  impl_class = GUPPI_SEQ_SCALAR_IMPL_CLASS(GTK_OBJECT(impl)->klass);

  if (impl_class->var) {
    return (impl_class->var)(impl);
  }

  guppi_seq_indices(GUPPI_SEQ(seq), &i0, &i1);

  ptr = guppi_seq_scalar_raw(seq, &stride);
  if (ptr != NULL) {
    ptr = (gconstpointer)(((guint8*)ptr) + stride*i0);
    
    for (i=i0; i<=i1; ++i) {
      x = *(const double*)ptr;
      ++count;
      om = mean;
      mean += (x - mean)/count;
      if (count > 1)
	sumsq += (x - mean)*(x - om);
      ptr = (gconstpointer)(((guint8*)ptr) + stride);
    }
    
  } else {

    for (i=i0; i<=i1; ++i) {
      x = guppi_seq_scalar_get(seq, i);
      ++count;
      om = mean;
      mean += (x - mean)/count;
      if (count > 1)
	sumsq += (x - mean)*(x - om);
    }

  }

  return sumsq / count;
}

double
guppi_seq_scalar_vars(const GuppiSeqScalar* seq)
{
  double v;
  gsize n;

  g_return_val_if_fail(seq != NULL, 0);
  
  n = guppi_seq_size(GUPPI_SEQ(seq));
  g_return_val_if_fail(n>1, 0);
  v = guppi_seq_scalar_var(seq);
  return (n*(n-1))*v;
}

double
guppi_seq_scalar_sdev(const GuppiSeqScalar* seq)
{
  double v;

  g_return_val_if_fail(seq != NULL, 0);
  v = guppi_seq_scalar_var(seq);
  return sqrt(v);
}

double
guppi_seq_scalar_sdevs(const GuppiSeqScalar* seq)
{
  double v;

  g_return_val_if_fail(seq != NULL, 0);
  v = guppi_seq_scalar_vars(seq);
  return sqrt(v);
}

gconstpointer
guppi_seq_scalar_raw(const GuppiSeqScalar* seq, gint* stride)
{
  const GuppiSeqScalarImpl* impl;
  GuppiSeqScalarImplClass* impl_class;

  g_return_val_if_fail(seq != NULL, NULL);
  g_return_val_if_fail(stride != NULL, NULL);

  impl = GUPPI_SEQ_SCALAR_IMPL(guppi_data_impl(GUPPI_DATA(seq)));
  impl_class = GUPPI_SEQ_SCALAR_IMPL_CLASS(GTK_OBJECT(impl)->klass);

  if (impl_class->raw_access == NULL)
    return NULL;

  return (impl_class->raw_access)(impl, stride);
}

/* $Id: guppi-seq-scalar.c,v 1.4 2000/03/04 18:51:36 trow Exp $ */
