/* This is -*- C -*- */
/* $Id: guppi-data.c,v 1.4 2000/02/09 21:04:56 trow Exp $ */

/*
 * guppi-data.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@emccta.com> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "guppi-data.h"

static GtkObjectClass* parent_class = NULL;

enum {
  ARG_0
};

enum {
  CHANGED,
  CHANGED_LABEL,
  CHANGED_SUBDATA,
  LAST_SIGNAL
};
static guint data_signals[LAST_SIGNAL] = { 0 };

static void
guppi_data_get_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_data_set_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_data_destroy(GtkObject* obj)
{
  GuppiData* gd = GUPPI_DATA(obj);

  if (parent_class->destroy)
    parent_class->destroy(obj);

  if (gd->freeze_count > 0)
    g_warning("Destroying frozen GuppiData");

  gtk_object_unref(GTK_OBJECT(gd->impl));
  gd->impl = NULL;

  g_free(gd->label);
  gd->label = NULL;
}

static void
guppi_data_finalize(GtkObject* obj)
{
  if (parent_class->finalize)
    parent_class->finalize(obj);
}

static void
guppi_data_class_init(GuppiDataClass* klass)
{
  GtkObjectClass* object_class = (GtkObjectClass*)klass;

  parent_class = gtk_type_class(GTK_TYPE_OBJECT);

  klass->default_impl = (GtkType)0;
  klass->type_name = "-* GuppiData *-";

  data_signals[CHANGED] =
    gtk_signal_new("changed",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GuppiDataClass, changed),
		   gtk_marshal_NONE__NONE,
		   GTK_TYPE_NONE, 0);

  data_signals[CHANGED_LABEL] =
    gtk_signal_new("changed_label",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GuppiDataClass, changed_label),
		   gtk_marshal_NONE__NONE,
		   GTK_TYPE_NONE, 0);

  data_signals[CHANGED_SUBDATA] =
    gtk_signal_new("changed_subdata",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GuppiDataClass, changed_subdata),
		   gtk_marshal_NONE__NONE,
		   GTK_TYPE_NONE, 0);

  gtk_object_class_add_signals(object_class, data_signals, LAST_SIGNAL);

  object_class->get_arg = guppi_data_get_arg;
  object_class->set_arg = guppi_data_set_arg;
  object_class->destroy = guppi_data_destroy;
  object_class->finalize = guppi_data_finalize;

}

static void
guppi_data_init(GuppiData* gd)
{
  gd->read_only = FALSE;
  gd->freeze_count = 0;

  gd->requested_impl = 0;
  gd->impl = NULL;
}

GtkType
guppi_data_get_type(void)
{
  static GtkType guppi_data_type = 0;
  if (!guppi_data_type) {
    static const GtkTypeInfo guppi_data_info = {
      "GuppiData",
      sizeof(GuppiData),
      sizeof(GuppiDataClass),
      (GtkClassInitFunc)guppi_data_class_init,
      (GtkObjectInitFunc)guppi_data_init,
      NULL, NULL, (GtkClassInitFunc)NULL
    };
    guppi_data_type = gtk_type_unique(GTK_TYPE_OBJECT, &guppi_data_info);
  }
  return guppi_data_type;
}

const gchar*
guppi_data_get_type_name(const GuppiData* gd)
{
  g_return_val_if_fail(gd != NULL, NULL);
  return GUPPI_DATA_CLASS(GTK_OBJECT(gd)->klass)->type_name;
}

GuppiDataImpl*
guppi_data_impl(GuppiData* gd)
{
  g_return_val_if_fail(gd != NULL, NULL);

  if (gd->impl == NULL) {
    GtkType type = gd->requested_impl;
    if (type == 0)
      type = GUPPI_DATA_CLASS(GTK_OBJECT(gd)->klass)->default_impl;
    g_return_val_if_fail(type != 0, NULL);

  /* Somewhere we should check that there isn't a mismatch between
     the data type and the default type of the implementation. */
    gd->impl = GUPPI_DATA_IMPL(gtk_type_new(type));
    g_assert(gd->impl);
  }

  /* We must always have an implementation installed by this point. */
  g_assert(gd->impl != NULL);

  return gd->impl;
}

const gchar*
guppi_data_get_label(const GuppiData* gd)
{
  g_return_val_if_fail(GUPPI_DATA(gd) != NULL, NULL);
  
  return gd->label ? gd->label : _("Unlabeled");
}

void
guppi_data_set_label(GuppiData* gd, const gchar* str)
{
  g_return_if_fail(gd != NULL);
  g_return_if_fail(str != NULL);

  g_free(gd->label);
  gd->label = g_strdup(str);
  gtk_signal_emit(GTK_OBJECT(gd), data_signals[CHANGED_LABEL]);
}

gboolean
guppi_data_is_read_only(const GuppiData* gd)
{
  return gd->read_only;
}

gboolean
guppi_data_can_change(const GuppiData* gd)
{
  return !gd->read_only;
}

const gchar*
guppi_data_get_impl_name(const GuppiData* gd)
{
  GuppiDataImpl* impl;
  g_return_val_if_fail(GUPPI_DATA(gd) != NULL, NULL);
  impl = guppi_data_impl((GuppiData*)gd);
  g_return_val_if_fail(impl != NULL, NULL);
  return guppi_data_impl_get_impl_name(impl);
}

void
guppi_data_freeze(GuppiData* gd)
{
  g_return_if_fail(GUPPI_DATA(gd) != NULL);
  g_assert(gd->freeze_count >= 0);
  ++gd->freeze_count;
}

void
guppi_data_thaw(GuppiData* gd)
{
  GuppiDataClass* klass;

  g_return_if_fail(GUPPI_DATA(gd) != NULL);
  g_return_if_fail(gd->freeze_count != 0);
  g_assert(gd->freeze_count > 0);
  --gd->freeze_count;

  if (gd->freeze_count == 0 && gd->pending_change) {
    klass = GUPPI_DATA_CLASS(GTK_OBJECT(gd)->klass);
    if (klass->emit_pending_signals)
      (klass->emit_pending_signals)(gd);
    guppi_data_changed(gd);

    gd->pending_change = FALSE;
  }
}

gboolean
guppi_data_frozen(GuppiData* gd)
{
  g_return_val_if_fail(GUPPI_DATA(gd) != NULL, FALSE);
  g_assert(gd->freeze_count >= 0);
  return gd->freeze_count > 0;
}

void
guppi_data_changed(GuppiData* gd)
{
  g_return_if_fail(GUPPI_DATA(gd) != NULL);
  if (guppi_data_frozen > 0)
    gd->pending_change = TRUE;
  else
    gtk_signal_emit(GTK_OBJECT(gd), data_signals[CHANGED]);
}

gboolean
guppi_data_has_hook(const GuppiData* gd, const gchar* key)
{
  GuppiDataImpl* impl;

  g_return_val_if_fail(GUPPI_DATA(gd) != NULL, FALSE);
  g_return_val_if_fail(key != NULL, FALSE);
  impl = guppi_data_impl((GuppiData*)gd);
  g_return_val_if_fail(impl != NULL, FALSE);
  
  return guppi_data_impl_has_hook(impl, key);
}

gpointer
guppi_data_call_hook(GuppiData* gd, const gchar* key, gpointer user_data)
{
  GuppiDataImpl* impl;

  g_return_val_if_fail(GUPPI_DATA(gd) != NULL, NULL);
  g_return_val_if_fail(key != NULL, NULL);
  impl = guppi_data_impl(gd);
  g_return_val_if_fail(impl != NULL, NULL);

  return guppi_data_impl_call_hook(impl, key, user_data);
}

GtkWidget*
guppi_data_make_browser_widget(GuppiData* gd)
{
  GuppiDataClass* klass;
  g_return_val_if_fail(GUPPI_DATA(gd) != NULL, NULL);
  klass = GUPPI_DATA_CLASS(GTK_OBJECT(gd)->klass);

  return klass->browser_widget ? (klass->browser_widget)(gd) : NULL;
}

GtkWidget*
guppi_data_make_info_widget(GuppiData* gd)
{
  GuppiDataClass* klass;
  g_return_val_if_fail(GUPPI_DATA(gd) != NULL, NULL);
  klass = GUPPI_DATA_CLASS(GTK_OBJECT(gd)->klass);

  return klass->info_widget ? (klass->info_widget)(gd) : NULL;
}

GuppiData*
guppi_data_copy(const GuppiData* gd)
{
  GuppiDataImpl* impl;
  GuppiDataImplClass* klass;
  GuppiData* copy;

  g_return_val_if_fail(GUPPI_DATA(gd) != NULL, NULL);

  impl = guppi_data_impl((GuppiData*)gd);
  klass = GUPPI_DATA_IMPL_CLASS(GTK_OBJECT(impl)->klass);
  g_assert(klass->copy);

  copy = GUPPI_DATA(gtk_type_new(GTK_OBJECT_TYPE(gd)));
  copy->impl = (klass->copy)(impl);

  copy->read_only = gd->read_only;
  copy->label = g_strdup_printf(_("Copy of %s"), 
				guppi_data_get_label(gd));

  return copy;
}

gint
guppi_data_get_size_in_bytes(GuppiData* gd)
{
  GuppiDataImpl* impl;
  GuppiDataImplClass* klass;

  g_return_val_if_fail(GUPPI_DATA(gd) != NULL, -1);
  impl = guppi_data_impl(gd);

  klass = GUPPI_DATA_IMPL_CLASS(GTK_OBJECT(impl)->klass);
  if (klass->get_size_in_bytes)
    return (klass->get_size_in_bytes)(impl);
  else
    return -1;
}

void
guppi_data_get_size_info(GuppiData* gd, gchar* buf, gsize N)
{
  GuppiDataImpl* impl;
  GuppiDataImplClass* klass;

  g_return_if_fail(GUPPI_DATA(gd) != NULL);
  g_return_if_fail(buf != NULL);
  
  if (N == 0)
    return;
  if (N == 1) {
    buf[0] = '\0';
    return;
  }

  impl = guppi_data_impl(gd);

  klass = GUPPI_DATA_IMPL_CLASS(GTK_OBJECT(impl)->klass);
  if (klass->get_size_info)
    (klass->get_size_info)(impl, buf, N);
  else {
    buf[0] = '?';
    buf[1] = '\0';
  }
}

GList*
guppi_data_get_subdata(GuppiData* data)
{
  GList* subdata_list = NULL;
  GList* iter;
  GuppiData* subdata;
  GuppiDataClass* klass;

  g_return_val_if_fail(data != NULL, NULL);

  klass = GUPPI_DATA_CLASS(GTK_OBJECT(data)->klass);
  if (klass->general_subdata)
    subdata_list = (klass->general_subdata)(data);
  
  iter = data->local_subdata;
  while (iter != NULL) {
    subdata = GUPPI_DATA(iter->data);
    subdata_list = g_list_append(subdata_list, subdata);
    gtk_object_ref(GTK_OBJECT(subdata));
    iter = g_list_next(iter);
  }

  return subdata_list;
}

void
guppi_data_add_subdata(GuppiData* data, GuppiData* subdata)
{
  g_return_if_fail(data != NULL);
  g_return_if_fail(subdata != NULL);

  data->local_subdata = g_list_append(data->local_subdata, subdata);
  gtk_signal_emit(GTK_OBJECT(data), data_signals[CHANGED_SUBDATA]);
}


/* $Id: guppi-data.c,v 1.4 2000/02/09 21:04:56 trow Exp $ */
