/* $Id: guppi.c,v 1.16 2000/02/21 18:35:01 trow Exp $ */

/*
 * guppi.c
 *
 * Copyright (C) 1999 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org>
 * and Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <gnome.h>
#include <glade/glade.h>
#include <signal.h>
#include <time.h>
#include <guppi-data-tree-view.h>
#include "corba_guppi.h"
#include <guppi-useful.h>
#include <guppi-data-init.h>
#include <guppi-plot-init.h>
#include <guppi-plot-plug-in.h>
#include "app_scm.h"
#include "term.h"
#include "file-open.h"
#include "plug-ins.h"
#include "guile-load.h"
#include "guppi.h"

static GnomeApp* guppi_app = NULL;

static void
dummy_uib_signal_connector(GnomeUIInfo* foo, gchar* bar, 
			   GnomeUIBuilderData* foobar)
{ }

void
guppi_add_menubar_item(const gchar* path, const gchar* name)
{
  GnomeUIInfo item[] = {
    GNOMEUIINFO_SUBTREE(NULL, NULL), 
    GNOMEUIINFO_END
  };

  GnomeUIBuilderData uibdata = { dummy_uib_signal_connector,
				 NULL, FALSE, NULL, NULL };

  GtkWidget* parent;
  gint pos;

  g_return_if_fail(name != NULL);

  item[0].label = (gchar*)name;
  /* We need a pointer to a GNOMEUIINFO_END to put in as the contents
     of the empty item we just built, and item[1] is just sitting
     there looking very tempting... */
  item[0].moreinfo = &item[1];

  if (path == NULL || *path == '\0')
    parent = gnome_app_find_menu_pos(guppi_app->menubar, "Help", &pos);
  else
    parent = gnome_app_find_menu_pos(guppi_app->menubar, path, &pos);

  g_return_if_fail(parent != NULL);

  gnome_app_fill_menu_custom(GTK_MENU_SHELL(parent), item, &uibdata,
			     guppi_app->accel_group, TRUE, pos-1);
}

void
guppi_add_menu(const gchar* path,
		    GnomeUIInfo* menuinfo)
{
  gnome_app_insert_menus(guppi_app, path, menuinfo);
}


/****************************************************************************/

/*
 * Build our Application window, hook up the basic callbacks
 */

static void
data_entry_cb(GtkWidget* w, gpointer data)
{
  guppi_unimplemented_function_dialog("Data Entry");
}

static void
save_cb(GtkWidget* w, gpointer data)
{
  guppi_unimplemented_function_dialog("Save");
}

static void
save_as_cb(GtkWidget* w, gpointer data)
{
  guppi_unimplemented_function_dialog("Save as");
}

static void
exit_cb(GtkWidget* w, gpointer data)
{
  guppi_exit();
}

static void
guile_terminal_cb(GtkWidget* w, gpointer data)
{
  guile_terminal();
}

static void
hacked_scatter_plot_cb(GtkWidget* w, gpointer data)
{
  guppi_eval_str("(let ((s (guppi-plot-element-new \"scatter\")) (state (guppi-plot-state-new))) (guppi-plot-state-add-element state s) (guppi-plot-state-view state) (guppi-plot-element-edit s))");
}

static void
guile_load_cb(GtkWidget* w, gpointer data)
{
  guile_select_and_load(GNOME_APP(data));
}

static void
about_cb(GtkWidget* w, gpointer data)
{
  static GtkWidget* dialog = NULL;
  GtkWidget* app;

  app = GTK_WIDGET(data);

  if (dialog != NULL) {
    g_assert(GTK_WIDGET_REALIZED(dialog));
    gdk_window_show(dialog->window);
    gdk_window_raise(dialog->window);
  } else {
    const gchar* authors[] = {
      "Havoc Pennington <hp@pobox.com>", 
      "Jon Trowbridge <trow@gnu.org>",
      NULL
    };

    dialog = gnome_about_new(PACKAGE, VERSION,
			     "(C) 1999 EMC Capital Management, Inc",
			     authors,
			     _("Guppi: Data Analysis/Visualization for the Masses"),
			     NULL);
    
    gtk_signal_connect(GTK_OBJECT(dialog),
		       "destroy",
		       GTK_SIGNAL_FUNC(gtk_widget_destroyed),
		       &dialog);

    gnome_dialog_set_parent(GNOME_DIALOG(dialog), GTK_WINDOW(app));

    gtk_widget_show(dialog);
  }
}

static void
about_plug_in_cb(GtkWidget* w, gpointer data)
{
  loaded_plug_ins();
}

static GtkWidget*
guppi_main_window()
{
  static GnomeUIInfo file_menu[] = {
    GNOMEUIINFO_MENU_OPEN_ITEM(file_open, NULL),
    { GNOME_APP_UI_ITEM,
      N_("Data Entry"), N_("Directly Enter Data For Import"),
      data_entry_cb, NULL, NULL,
      GNOME_APP_PIXMAP_NONE, NULL,
      0, (GdkModifierType)0, NULL },
    GNOMEUIINFO_MENU_SAVE_ITEM(save_cb, NULL),
    GNOMEUIINFO_MENU_SAVE_AS_ITEM(save_as_cb, NULL),
    GNOMEUIINFO_MENU_EXIT_ITEM(exit_cb, NULL),
    GNOMEUIINFO_END
  };

  static GnomeUIInfo plot_menu[] = {
    { GNOME_APP_UI_ITEM,
      N_("New Scatter Plot"), N_("Create a New Scatter Plot Window"),
      hacked_scatter_plot_cb, NULL, NULL,
      GNOME_APP_PIXMAP_NONE, NULL,
      0, (GdkModifierType)0, NULL },
    GNOMEUIINFO_END
  };
    
  static GnomeUIInfo guile_menu[] = {
    { GNOME_APP_UI_ITEM,
      N_("Open Guile Code"), N_("Load and Execute Guile Code"),
      guile_load_cb, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_OPEN,
      0, (GdkModifierType)0, NULL },
    { GNOME_APP_UI_ITEM,
      N_("Guile Terminal"), N_("Open an Interactive Guile Terminal"),
      guile_terminal_cb, NULL, NULL,
      GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_EXEC,
      0, (GdkModifierType)0, NULL },
    GNOMEUIINFO_SEPARATOR,
    GNOMEUIINFO_END
  };

  static GnomeUIInfo help_menu[] = {
    GNOMEUIINFO_HELP("guppi"),
    GNOMEUIINFO_MENU_ABOUT_ITEM(about_cb, NULL),
    { GNOME_APP_UI_ITEM,
      N_("About Plug-Ins"), N_("Find Out About Currently Loaded Plug-Ins"),
      about_plug_in_cb, NULL, NULL,
      GNOME_APP_PIXMAP_NONE, NULL,
      0, (GdkModifierType)0, NULL},
    GNOMEUIINFO_END
  };

  static GnomeUIInfo app_menus[] = {
    GNOMEUIINFO_MENU_FILE_TREE(file_menu),
    { GNOME_APP_UI_SUBTREE,
      _("Plot"), _("Plot"),
      plot_menu, NULL, NULL,
      GNOME_APP_PIXMAP_NONE, NULL,
      0, (GdkModifierType)0, NULL },
    { GNOME_APP_UI_SUBTREE,
      _("Scripting"), _("Scripting"),
      guile_menu, NULL, NULL,
      GNOME_APP_PIXMAP_NONE, NULL,
      0, (GdkModifierType)0, NULL },
    GNOMEUIINFO_MENU_HELP_TREE(help_menu),
    GNOMEUIINFO_END
  };

  GtkWidget* app = NULL;
  GtkWidget* swin = NULL;
  GtkWidget* browser = NULL;

  app = gnome_app_new(PACKAGE, "Guppi");
  gtk_window_set_default_size(GTK_WINDOW(app), 300, 200);

  gnome_app_set_statusbar(GNOME_APP(app), 
			  gnome_appbar_new(FALSE,TRUE,GNOME_PREFERENCES_USER));

  gnome_app_create_menus_with_data(GNOME_APP(app), app_menus, app);
  gnome_app_install_menu_hints(GNOME_APP(app), app_menus);

  swin = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(swin),
				 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

  browser = guppi_data_tree_view_new(guppi_data_tree_main());
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(swin), browser);

  gnome_app_set_contents(GNOME_APP(app), swin);

  gtk_signal_connect(GTK_OBJECT(app), "delete_event",
		     GTK_SIGNAL_FUNC(exit_cb), NULL);
  gtk_signal_connect(GTK_OBJECT(app), "destroy",
		     GTK_SIGNAL_FUNC(gtk_widget_destroyed), app);

  return app;
}

/****************************************************************************/

static void
load_rc_files(void)
{
  gchar buffer[1024];
  const gint buffer_len = 1024;

  g_snprintf(buffer, buffer_len, "%s/.guppirc", gnome_util_user_home());
  if (g_file_exists(buffer)) {
    guppi_splash_message(_("Loading .guppirc"));
    guppi_safe_load(buffer);
  }

  strncpy(buffer, "./.guppilocalrc", buffer_len);
  if (g_file_exists(buffer)) {
    guppi_splash_message(_("Loading .guppilocalrc"));
    guppi_safe_load(buffer);
  }
}

/*
  On the first try, we attempt a graceful exit.
  If a second attempt comes in soon enough, we give up and abort.
*/
static void
exit_signal_handler(gint x)
{
  static time_t then = 0;
  time_t now;
  
  time(&now);
  if (now - then > 10) {
    guppi_exit();
    then = now;
  } else {
    guppi_abort();
  }
}

static void
abort_signal_handler(gint x)
{
  guppi_abort();
}

static void
real_main(void* closure, int argc, char* argv[])
{
  struct sigaction xsh, ash;

  /* Trap ctrl-c from the starting console */
  xsh.sa_handler = exit_signal_handler;
  xsh.sa_flags = 0;

  ash.sa_handler = abort_signal_handler;
  ash.sa_flags = 0;

  sigaction(SIGTERM, &ash, NULL);
  sigaction(SIGINT, &xsh, NULL);
  sigaction(SIGHUP, &ash, NULL);

  bindtextdomain(PACKAGE, GNOMELOCALEDIR);
  textdomain(PACKAGE);

  /* We have many, many libraries and misc. things to initialize */
  guppi_splash_message(_("Initializing libglade"));
  glade_gnome_init();
  
  /*
    guppi_corba_init() calls gnome_CORBA_init(), which in turn calls
    gnome_init().
  */
  guppi_splash_message(_("Initializing CORBA"));
  guppi_corba_init(&argc, argv);
  guppi_exit_connect_shutdown_func(guppi_corba_shutdown, NULL);

  guppi_splash_message(_("Initializing Utility Library"));
  guppi_useful_init();

  guppi_splash_message(_("Initializing Data Library"));
  guppi_data_init();

  guppi_splash_message(_("Initializing Plot Library"));
  guppi_plot_init();

  guppi_splash_message(_("Initializing GuppiAppGuile"));
  guppi_app_scm_init();

  guppi_splash_message(_("Loading Plug-Ins"));
  guppi_plug_in_load_all();

  guppi_splash_message(_("Building Main Guppi Window"));
  guppi_app = GNOME_APP(guppi_main_window());

  /* We load our rc files after building our app window so that we can
     freely monkey with the app's UI. */
  load_rc_files();

  guppi_splash_destroy();

  gtk_widget_show_all(GTK_WIDGET(guppi_app));
  gtk_main();
}

int
main(int argc, char* argv[])
{
  /*
    We need to start up enough of the toolkit to let us put up our
     spash screen.  We'll init the rest of gnome in just a little while.
  */
  gtk_init(&argc, &argv);
  gdk_imlib_init();

  guppi_splash_create();

  guppi_splash_message(_("Booting Guile"));
  scm_boot_guile(argc, argv, real_main, NULL);
  g_assert_not_reached();
  return 1;
}


/* $Id: guppi.c,v 1.16 2000/02/21 18:35:01 trow Exp $ */
