// Copyright (C) 1999-2002 Open Source Telecom Corporation.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// As a special exception to the GNU General Public License, permission is 
// granted for additional uses of the text contained in its release 
// of Common C++.
// 
// The exception is that, if you link the Common C++ library with other
// files to produce an executable, this does not by itself cause the
// resulting executable to be covered by the GNU General Public License.
// Your use of that executable is in no way restricted on account of
// linking the Common C++ library code into it.
// 
// This exception does not however invalidate any other reasons why
// the executable file might be covered by the GNU General Public License.
// 
// This exception applies only to the code released under the 
// name Common C++.  If you copy code from other releases into a copy of
// Common C++, as the General Public License permits, the exception does
// not apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
//
// If you write modifications of your own for Common C++, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice.

#include <cc++/config.h>
#include <cc++/export.h>
#include <cc++/thread.h>
#include <cc++/process.h>
#include <cc++/strchar.h>

#include <cstdlib>
#include <cstdio>
#include <cerrno>

#ifndef	WIN32
#ifdef	HAVE_SYS_TIME_H
#include <sys/time.h>
#endif

#ifdef	HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif

#include <pwd.h>
#include <grp.h>

#ifdef	SIGTSTP
#include <sys/file.h>
#include <sys/ioctl.h>
#endif

#ifndef	_PATH_TTY
#define	_PATH_TTY "/dev/tty"
#endif
#endif

#ifdef	WIN32
#include <process.h>
#include <stdio.h>
#endif

#ifdef	CCXX_NAMESPACES
namespace ost {
#endif

static char *_pUser = NULL;
static char *_pHome = NULL;

#ifdef	WIN32
int Process::spawn(const char *exename, const char **args, bool wait)
{
	int mode = P_NOWAIT;

	if(wait)
			mode = P_WAIT;

	return (int)::spawnvp(mode, (char *)exename, (char **)args);
}

int Process::join(int pid)
{
	int status, result;

	if(pid == -1)
		return pid;

	result = (int)cwait(&status, pid, WAIT_CHILD);
	if(status & 0x0)
		return -1;
	return result;
}

bool Process::cancel(int pid, int sig)
{
	HANDLE hPid = OpenProcess(PROCESS_ALL_ACCESS, TRUE, pid);
	bool rtn = true;

	if(!hPid)
		return false;

	if(!sig)
		sig = 2;

	if(!TerminateProcess(hPid, sig))
		rtn = false;

	CloseHandle(hPid);
	return rtn;
}

#else

#ifndef WEXITSTATUS
#define WEXITSTATUS(status) ((unsigned)(status) >> 8)
#endif

#ifndef WIFEXITED
#define WIFEXITED(status) (((status) & 255) == 0)
#endif

#ifndef WTERMSIG
#define WTERMSIG(status) (((unsigned)(status)) & 0x7F)
#endif

#ifndef WIFSIGNALLED
#define WIFSIGNALLED(status) (((status) & 255) != 0)
#endif

#ifndef WCOREDUMP
#define WCOREDUMP(status) (((status) & 0x80) != 0)
#endif

static void lookup(void)
{
	struct passwd *pw = NULL;
#ifdef	HAVE_GETPWUID_R
	struct passwd pwd;
	char buffer[1024];

	::getpwuid_r(geteuid(), &pwd, buffer, 1024, &pw);
#else
	pw = ::getpwuid(geteuid());
#endif

	if(_pHome)
		delString(_pHome);
	if(_pUser)
		delString(_pUser);

	_pUser = _pHome = NULL;

	if(pw != NULL && pw->pw_dir != NULL)
		_pHome = newString(pw->pw_dir);

	if(pw != NULL && pw->pw_name != NULL)
		_pUser = newString(pw->pw_name);

	endpwent();
}

const char *Process::getUser(void)
{
	if(!_pUser)
		lookup();

	return (const char *)_pUser;
}

const char *Process::getHomeDir(void)
{
	if(!_pHome)
		lookup();

	return (const char *)_pHome;
}

bool Process::setUser(const char *id, bool grp)
{
        struct passwd *pw = NULL;
#ifdef  HAVE_GETPWNAM_R
        struct passwd pwd;
        char buffer[1024];

        ::getpwnam_r(id, &pwd, buffer, 1024, &pw);
#else
        pw = ::getpwnam(id);
#endif
	if(!pw)
		return false;

	if(grp)
		if(setgid(pw->pw_gid))
			return false;

	if(setuid(pw->pw_uid))
		return false;

	lookup();
	return true;
}

bool Process::setGroup(const char *id)
{
	struct group *group = NULL;
#ifdef	HAVE_GETGRNAM_R
	struct group grp;
	char buffer[2048];

	::getgrnam_r(id, &grp, buffer, 1024, &group);
#else
	group = ::getgrnam(id);
#endif
	if(!group)
	{
		endgrent();
		return false;
	}

#ifdef	HAVE_SETEGID
	setegid(group->gr_gid);
#endif
	if(setgid(group->gr_gid))
	{
		endgrent();
		return false;
	}

	endgrent();
	return true;
}

bool Process::cancel(int pid, int sig)
{
	if(!sig)
		sig = SIGTERM;

	if(pid < 1)
		return false;

	if(::kill(pid, sig))
		return false;

	return true;
}

int Process::join(int pid)
{
	int status;

	if(pid < 1)
		return -1;

#ifdef  HAVE_WAITPID
        waitpid(pid, &status, 0);
#else
#ifdef  HAVE_WAIT4
        wait4(pid, &status, 0, NULL);
#else
	int result;
        while((result = wait(&status)) != pid && result != -1)
                ;
#endif
#endif

        if(WIFEXITED(status))
                return WEXITSTATUS(status);
        else if(WIFSIGNALLED(status))
                return -WTERMSIG(status);
        else
                return -1;
}

int Process::spawn(const char *exename, const char **args, bool wait)
{
	int pid;

	pid = vfork();
	if(pid == -1)
		return -1;

	if(!pid)
	{
		execvp((char *)exename, (char **)args);
		_exit(-1);
	}

	if(!wait)
		return pid;

	return join(pid);
}

Process::Trap Process::setInterruptSignal(int signo, Trap func)
{
	struct	sigaction	sig_act, old_act;

	sig_act.sa_handler = func;
	sigemptyset(&sig_act.sa_mask);
	if(signo != SIGALRM)
		sigaddset(&sig_act.sa_mask, SIGALRM);

	sig_act.sa_flags = 0;
#ifdef	SA_INTERRUPT
	sig_act.sa_flags |= SA_INTERRUPT;
#endif
	if(sigaction(signo, &sig_act, &old_act) < 0)
		return SIG_ERR;

	return old_act.sa_handler;
}

Process::Trap Process::setPosixSignal(int signo, Trap func)
{
	struct	sigaction	sig_act, old_act;

	sig_act.sa_handler = func;
	sigemptyset(&sig_act.sa_mask);
	sig_act.sa_flags = 0;
	if(signo == SIGALRM)
	{
#ifdef	SA_INTERRUPT
		sig_act.sa_flags |= SA_INTERRUPT;
#endif
	}
	else
	{
		sigaddset(&sig_act.sa_mask, SIGALRM);
#ifdef	SA_RESTART
		sig_act.sa_flags |= SA_RESTART;
#endif
	}
	if(sigaction(signo, &sig_act, &old_act) < 0)
		return SIG_ERR;
	return old_act.sa_handler;
}

void	Process::detach(void)
{
	attach("/dev/null");
}

void    Process::attach(const char *dev)
{
        int pid;
        int fd;

        if(getppid() == 1)
                return;

	::close(0);
	::close(1);
	::close(2);

#ifdef  SIGTTOU
        ::signal(SIGTTOU, SIG_IGN);
#endif

#ifdef  SIGTTIN
        ::signal(SIGTTIN, SIG_IGN);
#endif

#ifdef  SIGTSTP
        ::signal(SIGTSTP, SIG_IGN);
#endif

        if((pid = fork()) < 0)
                THROW(pid);
        else if(pid > 0)
                exit(0);


#if	defined(SIGTSTP) && defined(TIOCNOTTY)
        if(setpgid(0, getpid()) == -1)
                THROW(-1);

        if((fd = open(_PATH_TTY, O_RDWR)) >= 0)
        {
                ioctl(fd, TIOCNOTTY, NULL);
                close(fd);
        }
#else

#ifdef	HAVE_SETPGRP
        if(setpgrp() == -1)
                THROW(-1);
#else
	if(setpgid(0, getpid()) == -1)
		THROW(-1);
#endif

        ::signal(SIGHUP, SIG_IGN);

        if((pid = fork()) < 0)
                THROW(-1);
        else if(pid > 0)
                exit(0);
#endif

	if(dev && *dev)
	{
       		::open(dev, O_RDWR);
        	::open(dev, O_RDWR);
        	::open(dev, O_RDWR);
	}
}

#endif	// not win32

void Process::setEnv(const char *name, const char *value, bool overwrite)
{
#ifdef	HAVE_SETENV
	::setenv(name, value, (int)overwrite);
#else
	char strbuf[256];

	snprintf(strbuf, sizeof(strbuf), "%s=%s", name, value);
	if(!overwrite)
		if(getenv(strbuf))
			return;

	::putenv(strdup(strbuf));
#endif
}

const char *Process::getEnv(const char *name)
{
	return ::getenv(name);
}

#ifdef	CCXX_NAMESPACES
}
#endif

/** EMACS **
 * Local variables:
 * mode: c++
 * c-basic-offset: 8
 * End:
 */
