/* Copyright (C) 1989, 1990, 1991 Aladdin Enterprises.  All rights reserved.
   Distributed by Free Software Foundation, Inc.

This file is part of Ghostscript.

Ghostscript is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY.  No author or distributor accepts responsibility
to anyone for the consequences of using it or for whether it serves any
particular purpose or works at all, unless he says so in writing.  Refer
to the Ghostscript General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
Ghostscript, but only under the conditions described in the Ghostscript
General Public License.  A copy of this license is supposed to have been
given to you along with Ghostscript so you can know your rights and
responsibilities.  It should be in a file named COPYING.  Among other
things, the copyright notice and this notice must be preserved on all
copies.  */

/* zcolor.c */
/* Color operators for Ghostscript */
#include "ghost.h"
#include "errors.h"
#include "oper.h"
#include "alloc.h"
#include "estack.h"
#include "iutil.h"
#include "store.h"
#include "gsmatrix.h"			/* for gs_state */
#include "gsstate.h"
#include "state.h"

/* Forward declarations */
private int make_color(P1(os_ptr));
private void tri_put(P2(os_ptr, float [3]));
private int remap_color(P0());

/* currentcolorspace */
/* Note: this is NOT the Level 2 P*stScr*pt currentcolorspace operator! */
int
zcurrentcolorspace(register os_ptr op)
{	gs_color_space cs;
	gs_currentcolorspace(igs, &cs);
	push(1);
	make_int(op, (int)cs);
	return 0;
}

/* currentcolortransfer */
int
zcurrentcolortransfer(register os_ptr op)
{	push(4);
	op[-3] = istate.transfer_procs.red;
	op[-2] = istate.transfer_procs.green;
	op[-1] = istate.transfer_procs.blue;
	*op = istate.transfer_procs.gray;
	return 0;
}

/* currentgray */
int
zcurrentgray(register os_ptr op)
{	push(1);
	make_real(op, gs_currentgray(igs));
	return 0;
}

/* currentgscolor */
int
zcurrentgscolor(register os_ptr op)
{	int code;
	push(1);
	if (	(code = make_color(op)) < 0 ||
		(code = gs_currentgscolor(igs, op->value.pcolor)) < 0
	   )
		pop(1);
	return code;
}

/* currenthsbcolor */
int
zcurrenthsbcolor(register os_ptr op)
{	float par[3];
	gs_currenthsbcolor(igs, par);
	push(3);
	tri_put(op, par);
	return 0;
}

/* currentrgbcolor */
int
zcurrentrgbcolor(register os_ptr op)
{	float par[3];
	gs_currentrgbcolor(igs, par);
	push(3);
	tri_put(op, par);
	return 0;
}

/* currenttransfer */
int
zcurrenttransfer(register os_ptr op)
{	push(1);
	*op = istate.transfer_procs.gray;
	return 0;
}

/* setcolortransfer */
int
zsetcolortransfer(register os_ptr op)
{	check_proc(op[-3]);
	check_proc(op[-2]);
	check_proc(op[-1]);
	check_proc(*op);
	istate.transfer_procs.red = op[-3];
	istate.transfer_procs.green = op[-2];
	istate.transfer_procs.blue = op[-1];
	istate.transfer_procs.gray = *op;
	pop(4);
	return remap_color();
}

/* setgray */
int
zsetgray(register os_ptr op)
{	float gray;
	int code;
	if ( (code = real_param(op, &gray)) < 0 ||
	     (code = gs_setgray(igs, gray)) < 0
	   )
		return code;
	pop(1);
	return remap_color();
}

/* setgscolor */
int
zsetgscolor(register os_ptr op)
{	int code;
	check_type(*op, t_color);
	if ( (code = gs_setgscolor(igs, op->value.pcolor)) < 0 )
		return code;
	pop(1);
	return remap_color();
}

/* sethsbcolor */
int
zsethsbcolor(register os_ptr op)
{	float par[3];
	int code;
	if (	(code = num_params(op, 3, par)) < 0 ||
		(code = gs_sethsbcolor(igs, par[0], par[1], par[2])) < 0
	   )
		return code;
	pop(3);
	return remap_color();
}

/* setrgbcolor */
int
zsetrgbcolor(register os_ptr op)
{	float par[3];
	int code;
	if (	(code = num_params(op, 3, par)) < 0 ||
		(code = gs_setrgbcolor(igs, par[0], par[1], par[2])) < 0
	   )
		return code;
	pop(3);
	return remap_color();
}

/* settransfer */
int
zsettransfer(register os_ptr op)
{	check_proc(*op);
	istate.transfer_procs.red = istate.transfer_procs.green =
	  istate.transfer_procs.blue = istate.transfer_procs.gray = *op;
	pop(1);
	return remap_color();
}

/* ------ Initialization procedure ------ */

op_def zcolor_op_defs[] = {
	{"0.currentcolorspace", zcurrentcolorspace},
	{"0currentcolortransfer", zcurrentcolortransfer},
	{"0currentgray", zcurrentgray},
	{"0currentgscolor", zcurrentgscolor},
	{"0currenthsbcolor", zcurrenthsbcolor},
	{"0currentrgbcolor", zcurrentrgbcolor},
	{"0currenttransfer", zcurrenttransfer},
	{"4setcolortransfer", zsetcolortransfer},
	{"1setgray", zsetgray},
	{"1setgscolor", zsetgscolor},
	{"3sethsbcolor", zsethsbcolor},
	{"3setrgbcolor", zsetrgbcolor},
	{"1settransfer", zsettransfer},
	op_def_end(0)
};

/* ------ Internal routines ------ */

/* Make a color object */
private int
make_color(os_ptr op)
{	gs_color *cp = (gs_color *)alloc(1, gs_color_sizeof, "make_color");
	if ( cp == 0 ) return e_VMerror;
	make_tv(op, t_color, pcolor, cp);
	return 0;
}

/* Put 3 reals on the operand stack. */
private void
tri_put(register os_ptr op, float pf3[3])
{	make_real(op - 2, pf3[0]);
	make_real(op - 1, pf3[1]);
	make_real(op, pf3[2]);
}

/* Remap the current color after changing it or the transfer function. */
/* We have to run the transfer function(s). */
extern gs_color_space gx_device_color_space(P1(gs_state *));
private int remap_gray_finish(P1(os_ptr));
private int remap_rgb_finish(P1(os_ptr));
private int
remap_color()
{	gs_state *pgs = igs;
#define push_color(v, p)\
  esp++; ref_assign(esp, &istate.transfer_procs.p);\
  esp++; make_real(esp, v)
	switch ( gx_device_color_space(pgs) )
	   {
	case gs_color_space_DeviceRGB:
	   {	float rgb[3];
		check_estack(7);
		gs_currentrgbcolor(pgs, rgb);
		push_op_estack(remap_rgb_finish);
		push_color(rgb[2], blue);
		push_color(rgb[1], green);
		push_color(rgb[0], red);
	   }	break;
	case gs_color_space_DeviceGray:
	   {	float gray;
		check_estack(3);
		gray = gs_currentgray(pgs);
		push_op_estack(remap_gray_finish);
		push_color(gray, gray);
	   }	break;
	default:
		return e_undefined;
	   }
#undef push_color
	return o_check_estack;
}
/* Finish remapping a gray. */
private int
remap_gray_finish(register os_ptr op)
{	float mgray;
	gs_color* pmcolor =
		(gs_color *)alloc(1, gs_color_sizeof, "remap_gray");
	int code;
	if ( pmcolor == 0 ) return e_VMerror;
	code = num_params(op, 1, &mgray);
	if ( code < 0 ) return code;
	gx_set_gray_only(pmcolor, mgray);
	gx_render_color(pmcolor, igs);
	alloc_free((char *)pmcolor, 1, gs_color_sizeof, "remap_gray");
	pop(1);
	return 0;
}
/* Finish remapping an RGB color. */
private int
remap_rgb_finish(register os_ptr op)
{	float mrgb[3];
	gs_color* pmcolor =
		(gs_color *)alloc(1, gs_color_sizeof, "remap_gray");
	int code;
	if ( pmcolor == 0 ) return e_VMerror;
	code = num_params(op, 3, mrgb);
	if ( code < 0 ) return code;
	gx_set_rgb_only(pmcolor, mrgb[0], mrgb[1], mrgb[2]);
	gx_render_color(pmcolor, igs);
	alloc_free((char *)pmcolor, 1, gs_color_sizeof, "remap_gray");
	pop(3);
	return 0;
}
