/* Copyright (C) 1989, 1990, 1991 Aladdin Enterprises.  All rights reserved.
   Distributed by Free Software Foundation, Inc.

This file is part of Ghostscript.

Ghostscript is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY.  No author or distributor accepts responsibility
to anyone for the consequences of using it or for whether it serves any
particular purpose or works at all, unless he says so in writing.  Refer
to the Ghostscript General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
Ghostscript, but only under the conditions described in the Ghostscript
General Public License.  A copy of this license is supposed to have been
given to you along with Ghostscript so you can know your rights and
responsibilities.  It should be in a file named COPYING.  Among other
things, the copyright notice and this notice must be preserved on all
copies.  */

/* idebug.c */
/* Debugging support for Ghostscript interpreter */
#include "string_.h"
#include "ghost.h"
#include "iutil.h"
#include "dict.h"
#include "name.h"
#include "ostack.h"			/* for opdef.h */
#include "opdef.h"
#include "packed.h"
#include "store.h"		/* for make_oper for opdef.h */

/* Forward references */
void debug_print_string(P2(byte *, ushort));
#define debug_print_name(pnref)\
  debug_print_string((pnref)->value.pname->string_bytes,\
		     (pnref)->value.pname->string_size)

/* Print a ref */
void
debug_print_ref(ref *pref)
{	unsigned size = r_size(pref);
	ushort elt;
	ref nref;
	dprintf1("(%x)", r_type_attrs(pref));
	elt = *(ushort *)pref;
	if ( elt <= packed_max_full_ref )
	  switch ( r_type(pref) )
	   {
	case t_array:
	  dprintf2("array(%u)0x%lx", size, (ulong)pref->value.refs); break;
	case t_boolean:
	  dprintf1("boolean %x", pref->value.index); break;
	case t_color:
	  dprintf1("color 0x%lx", (ulong)pref->value.pcolor); break;
	case t_device:
	  dprintf1("device 0x%lx", (ulong)pref->value.pdevice); break;
	case t_dictionary:
	  dprintf3("dict(%u/%u)0x%lx",
		   dict_length(pref), dict_maxlength(pref),
		   (ulong)pref->value.pdict);
	  break;
	case t_file: dprintf1("file 0x%lx", (ulong)pref->value.pfile); break;
	case t_integer: dprintf1("int %ld", pref->value.intval); break;
	case t_mark: dprintf("mark"); break;
	case t_mixedarray:
	  dprintf2("mixed packedarray(%u)0x%lx", size,
		   (ulong)pref->value.packed); break;
	case t_name:
	  dprintf2("name(0x%lx#%x)", (ulong)pref->value.pname,
		   pref->value.pname->index);
	  debug_print_name(pref);
	  break;
	case t_null: dprintf("null"); break;
	case t_oparray:
	  dprintf1("op_array(0x%x)", size);
	  name_index_ref(op_array_nx_table[size - op_def_count], &nref);
	  debug_print_name(&nref);
	  break;
	case t_operator:
	  dprintf1("op(0x%x", size);
	  if ( size )
	    dprintf1(":%s", (char *)(op_def_table[size]->oname + 1));
	  dprintf1(")0x%lx", (ulong)pref->value.opproc);
	  break;
	case t_real: dprintf1("real %f", pref->value.realval); break;
	case t_shortarray:
	  dprintf2("short packedarray(%u)0x%lx", size,
		   (ulong)pref->value.packed); break;
	case t_string:
	  dprintf2("string(%u)0x%lx", size, (ulong)pref->value.bytes); break;
	default: dprintf1("type 0x%x", r_type(pref));
	   }
	else
	  switch ( elt >> packed_type_shift )
	   {
	case pt_executable_operator:
	  dprintf("<op_name>");
	  elt &= packed_int_mask;
	  op_index_ref(elt, &nref);
	  debug_print_ref(&nref);
	  break;
	case pt_integer:
	  dprintf1("<int> %d", (elt & packed_int_mask) + packed_min_intval);
	  break;
	case pt_literal_name: case pt_literal_name+1:
	  dprintf("<lit_name>"); elt &= packed_max_name_index; goto ptn;
	case pt_executable_name: case pt_executable_name+1:
	  dprintf("<exec_name>"); elt &= packed_max_name_index;
ptn:	  name_index_ref(elt, &nref);
	  dprintf2("(0x%lx#%x)", (ulong)nref.value.pname, elt);
	  debug_print_name(&nref);
	  break;
	   }
}

/* Print a string */
void
debug_print_string(byte *chrs, ushort len)
{	ushort i;
	for ( i = 0; i < len; i++ )
		dputc(chrs[i]);
}

/* Dump a region of memory containing refs */
void
debug_dump_refs(ref *from, ref *to, char *msg)
{	ref *p = from;
	if ( from < to ) dprintf2("%s at 0x%lx:\n", msg, (ulong)from);
	while ( p < to )
	   {	unsigned type = r_type(p), attrs = r_type_attrs(p);
		unsigned btype = r_btype(p);
		static char *ts[] = { type_print_strings };
		static char *as = attr_print_string;
		char *ap = as;
#define buf_size 30
		char buf[buf_size + 1];
		uint plen;
		dprintf2("..%04x: 0x%02x ", (uint)p & 0xffff, type);
		if ( btype >= t_next_index )
			dprintf1("0x%02x?? ", btype);
		else
			dprintf1("%s ", ts[btype]);
		for ( ; *ap; ap++, attrs >>= 1 )
		  if ( *ap != '.' )
		    dputc(((attrs & 1) ? *ap : '-'));
		dprintf2(" 0x%04x 0x%08lx", r_size(p), *(ulong *)&p->value);
		if ( obj_cvs(p, (byte *)buf, buf_size, &plen) >= 0 &&
		     ((buf[plen] = 0), strcmp(buf, "--nostringval--"))
		   )
			dprintf1(" = %s", buf);
		dputc('\n');
		p++;
	   }
}

/* Dump a region of memory */
void
debug_dump_bytes(byte *from, byte *to, char *msg)
{	byte *p = from;
	if ( from < to ) dprintf1("%s:\n", msg);
	while ( p != to )
	   {	byte *q = min(p + 16, to);
		dprintf1("%lx:", (ulong)p);
		while ( p != q ) dprintf1(" %02x", *p++);
		dputc('\n');
	   }
}
