/* Copyright (C) 1989, 1990, 1991 Aladdin Enterprises.  All rights reserved.
   Distributed by Free Software Foundation, Inc.

This file is part of Ghostscript.

Ghostscript is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY.  No author or distributor accepts responsibility
to anyone for the consequences of using it or for whether it serves any
particular purpose or works at all, unless he says so in writing.  Refer
to the Ghostscript General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
Ghostscript, but only under the conditions described in the Ghostscript
General Public License.  A copy of this license is supposed to have been
given to you along with Ghostscript so you can know your rights and
responsibilities.  It should be in a file named COPYING.  Among other
things, the copyright notice and this notice must be preserved on all
copies.  */

/* ghost.h */
/* Common definitions for Ghostscript */
#include "gx.h"

/* The typedef for object references */
typedef struct ref_s ref;

/*
 * Object types.  This should be an enum, but there is no way
 * to declare an enum a subrange of byte rather than int....
 * The types marked with + use the read/write/execute attributes;
 * the rest only use the executable attribute.
 */
#define t_array 0			/* + value.refs, uses size */
#define t_boolean 1			/* value.index */
#define t_dictionary 2			/* + value.pdict */
#define t_file 3			/* + value.pfile */
#define t_fontID 4			/* value.pfont */
#define t_integer 5			/* value.intval */
#define t_mark 6			/* (no value) */
#define t_name 7			/* value.pname */
#define t_null 8			/* (no value) */
#define t_operator 9			/* value.opproc, uses size */
#define t_real 10			/* value.realval */
#define t_save 11			/* value.psave */
#define t_string 12			/* + value.bytes, uses size */
/* The following are the two implementations of packed arrays. */
#define t_mixedarray 13			/* + value.packed, uses size */
#define t_shortarray 14			/* + value.packed, uses size */
/*
 * The following are extensions to the PostScript type set.
 * When adding new types, be sure to edit the table in gs_init.ps
 * (==only operator), the printing routine in idebug.c,
 * and also type_print_strings below.
 */
#define t_color 15			/* value.pcolor */
#define t_device 16			/* value.pdevice */
#define t_oparray 17			/* (no value), uses size */
#define t_next_index 18		/* *** change this if adding types *** */
/*
 * The interpreter uses types starting at t_next_index for representing
 * a few high-frequency operators.
 * Since there are no operations specifically on operators,
 * there is no need for any operators to check specifically for these
 * types.  The r_btype macro takes care of the conversion when required.
 */
/*
 * Define the type names for debugging printout.
 * All names must be the same length, so that columns will line up.
 */
#define type_print_strings\
  "arry","bool","dict","file","font","int ","mark","name","null","oper",\
  "real","save","str ","mpry","spry","colr","devc","opry"
/*
 * Define the type names for the type operator.
 */
#define type_name_strings\
  "arraytype","booleantype","dicttype","filetype","fonttype",\
  "integertype","marktype","nametype","nulltype","operatortype",\
  "realtype","savetype","stringtype","packedarraytype","packedarraytype",\
  "colortype","devicetype","operatortype"

/*
 * The encoding of attributes is constrained by two factors:
 *
 *	- The packed array format requires the high-order bits of the
 *	  type/attributes field to be 0.  (see packed.h)
 *
 *	- The interpreter wants the type, executable bit, and execute
 *	  permission to be adjacent, and in that order from high to low.
 *
 * The layout given below is the one that leads to the most efficient
 * dispatching in the interpreter.
 */

/* Location attributes */
/* Note that these are associated with the *location*, not with the */
/* ref that is *stored* in that location. */
#define l_mark 1			/* mark for garbage collector */
					/* (not used yet) */
#define l_new 2				/* stored into since last save */
/* Object attributes */
#define a_subrange 4			/* subarray or substring */
					/* (set properly, but never read) */
#define a_write 8
#define a_read 0x10
#define a_execute 0x20
#define a_executable 0x40
#define a_all (a_write+a_read+a_execute)
/* Define the attribute names for debugging printout. */
#define attr_print_string "mnswrxe.....????"

/* Abstract types */
typedef struct dict_s dict;
typedef struct name_s name;
/* We define a dummy type for op_proc_p so that */
/* we don't have to include oper.h. */
typedef int (*dummy_op_proc_p)();
#define real_opproc(pref) (*(op_proc_p *)&(pref)->value)

/* Object reference */
/*
 * Note that because of the way packed arrays are represented,
 * the type_attrs member must be the first one in the ref structure.
 */
struct stream_s;
struct gs_font_s;
struct gs_color_s;
struct gx_device_s;
struct vm_save_s;
struct ref_s {

	struct {
		ushort type_attrs;
		ushort rsize;
	} tas;

#define r_size(rp) ((rp)->tas.rsize)
#define r_inc_size(rp,inc) ((rp)->tas.rsize += (inc))
#define r_set_size(rp,siz) ((rp)->tas.rsize = (siz))
#define r_set_subrange_size(rp,siz)\
 (r_set_size(rp, siz), r_set_attrs(rp, a_subrange))
/* type_attrs is a single element for fast dispatching in the interpreter */
#define r_type_shift 7
#define r_type(rp) ((rp)->tas.type_attrs >> r_type_shift)
#define r_has_type(rp,typ)\
 (((rp)->tas.type_attrs & (0x1f << r_type_shift)) == ((typ) << r_type_shift))
#define r_set_type(rp,typ) ((rp)->tas.type_attrs = (typ) << r_type_shift)
#define r_btype(rp)\
 ((rp)->tas.type_attrs >= (t_next_index << r_type_shift) ?\
  t_operator : r_type(rp))
#define r_type_xe(rp) ((rp)->tas.type_attrs >> (r_type_shift - 2))
#define r_type_attrs(rp) ((rp)->tas.type_attrs)	/* reading only */
#define r_has_attrs(rp,mask)		/* optimize 1-bit case */\
  (!(mask) ? 1 :\
   (mask) & ((mask)-1) ? !(~r_type_attrs(rp) & (mask)) :\
   r_type_attrs(rp) & (mask))
#define r_set_attrs(rp,mask) ((rp)->tas.type_attrs |= (mask))
#define r_clear_attrs(rp,mask) ((rp)->tas.type_attrs &= ~(mask))
#define r_set_type_attrs(rp,typ,mask)\
 ((rp)->tas.type_attrs = ((typ) << r_type_shift) + (mask))

	union v {			/* name the union to keep gdb happy */
		long intval;
		ushort index;		/* for enumerated things */
		float realval;
		byte *bytes;
		struct ref_s *refs;
		name *pname;
		dict *pdict;
		ushort *packed;
		dummy_op_proc_p opproc;
		struct stream_s *pfile;
		struct gs_font_s *pfont;
		struct gs_color_s *pcolor;
		struct gx_device_s *pdevice;
		struct vm_save_s *psave;
	} value;
};
