/* Copyright (C) 1990, 1991 Aladdin Enterprises.  All rights reserved.
   Distributed by Free Software Foundation, Inc.

This file is part of Ghostscript.

Ghostscript is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY.  No author or distributor accepts responsibility
to anyone for the consequences of using it or for whether it serves any
particular purpose or works at all, unless he says so in writing.  Refer
to the Ghostscript General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
Ghostscript, but only under the conditions described in the Ghostscript
General Public License.  A copy of this license is supposed to have been
given to you along with Ghostscript so you can know your rights and
responsibilities.  It should be in a file named COPYING.  Among other
things, the copyright notice and this notice must be preserved on all
copies.  */

/* gstype1.c */
/* Adobe Type 1 font routines for Ghostscript library */
#include "math_.h"
#include "memory_.h"
#include "gx.h"
#include "gserrors.h"
#include "gxarith.h"
#include "gxfixed.h"
#include "gxmatrix.h"
#include "gzstate.h"
#include "gzdevice.h"			/* for gxchar */
#include "gxdevmem.h"			/* ditto */
#include "gzpath.h"
#include "gxchar.h"
#include "gxfont.h"
#include "gxtype1.h"
#include "gxop1.h"

/* Encrypt a string. */
int
gs_type1_encrypt(byte *dest, byte *src, uint len, crypt_state *pstate)
{	register crypt_state state = *pstate;
	register byte *from = src;
	register byte *to = dest;
	register uint count = len;
	while ( count )
	   {	encrypt_next(*from, state, *to);
		from++, to++, count--;
	   }
	*pstate = state;
	return 0;
}
/* Decrypt a string. */
int
gs_type1_decrypt(byte *dest, byte *src, uint len, crypt_state *pstate)
{	register crypt_state state = *pstate;
	register byte *from = src;
	register byte *to = dest;
	register uint count = len;
	while ( count )
	   {	/* If from == to, we can't use the obvious */
		/*	decrypt_next(*from, state, *to);	*/
		register byte ch = *from++;
		decrypt_next(ch, state, *to);
		to++, count--;
	   }
	*pstate = state;
	return 0;
}

/* Define the structures for the state of a Type 1 interpreter. */
/* This is the interpreter state that must be saved and restored */
/* when calling a CharString subroutine. */
typedef struct {
	byte *ip;
	crypt_state dstate;
} ip_state;
/* This is the full state of the Type 1 interpreter. */
struct gs_type1_state_s {
		/* The following are set at initialization */
	gs_show_enum *penum;		/* show enumerator */
	gs_state *pgs;			/* graphics state */
	gs_type1_data *pdata;		/* font-specific data */
	int charpath_flag;		/* 0 for show, 1 for false */
					/* charpath, 2 for true charpath */
	int paint_type;			/* 0 for fill, non-0 for stroke */
	fixed_coeff fc;			/* cached fixed coefficients */
		/* The following are updated dynamically */
	fixed ostack[ostack_size];	/* the Type 1 operand stack */
	int os_count;			/* # of occupied stack entries */
#define ipstack_size 10
	ip_state ipstack[ipstack_size+1];	/* control stack */
	int ips_count;			/* # of occupied entries */
	int ip_skip;			/* # of bytes to skip */
	gs_fixed_point lsb;		/* left side bearing */
	gs_fixed_point width;		/* character width */
	int seac_base;			/* base character code for seac, */
					/* or -1 */
};

/* Export the size of the structure */
const uint gs_type1_state_sizeof = sizeof(gs_type1_state);

/* Initialize the cached matrix in a Type 1 interpreter */
/* from the matrix in the graphics state. */
int
gs_type1_init_matrix(register gs_type1_state *pis)
{	gs_matrix ctm;
	int scale = -10000;
	int expt, shift;
	ctm = ctm_only(pis->pgs);
	pis->fc.skewed = 0;
	if ( !is_fzero(ctm.xx) )
	   {	(void)frexp(ctm.xx, &scale);
	   }
	if ( !is_fzero(ctm.xy) )
	   {	(void)frexp(ctm.xy, &expt);
		if ( expt > scale ) scale = expt;
		pis->fc.skewed = 1;
	   }
	if ( !is_fzero(ctm.yx) )
	   {	(void)frexp(ctm.yx, &expt);
		if ( expt > scale ) scale = expt;
		pis->fc.skewed = 1;
	   }
	if ( !is_fzero(ctm.yy) )
	   {	(void)frexp(ctm.yy, &expt);
		if ( expt > scale ) scale = expt;
	   }
	scale = 31 - max_coeff_bits - scale;
	pis->fc.xx = (is_fzero(ctm.xx) ? 0 : (long)ldexp(ctm.xx, scale));
	pis->fc.yy = (is_fzero(ctm.yy) ? 0 : (long)ldexp(ctm.yy, scale));
	if ( pis->fc.skewed )
	   {	pis->fc.xy = (is_fzero(ctm.xy) ? 0 : (long)ldexp(ctm.xy, scale));
		pis->fc.yx = (is_fzero(ctm.yx) ? 0 : (long)ldexp(ctm.yx, scale));
	   }
	else
		pis->fc.xy = pis->fc.yx = 0;
	shift = scale - _fixed_shift;
	if ( shift > 0 )
	   {	pis->fc.shift = shift;
		pis->fc.round = (fixed)1 << (shift - 1);
	   }
	else
	   {	pis->fc.shift = 0;
		pis->fc.round = 0;
		shift = -shift;
		pis->fc.xx <<= shift;
		pis->fc.yy <<= shift;
		if ( pis->fc.skewed )
		   {	pis->fc.xx <<= shift;
			pis->fc.xx <<= shift;
		   }
	   }
#ifdef DEBUG
if ( gs_debug['1'] )
   {	dprintf7("[1]ctm: [%6g %6g %6g %6g ; %6g %6g] scale=%d\n",
		 ctm.xx, ctm.xy, ctm.yx, ctm.yy, ctm.tx, ctm.ty, scale);
	dprintf5("   fc: [%lx %lx %lx %lx] shift=%d\n",
		 pis->fc.xx, pis->fc.xy, pis->fc.yx, pis->fc.yy,
		 pis->fc.shift);
   }
#endif
	return 0;
}

/* Initialize a Type 1 interpreter */
int
gs_type1_init(register gs_type1_state *pis, gs_show_enum *penum,
  int charpath_flag, int paint_type, byte *str, gs_type1_data *pdata)
{	gs_state *pgs = penum->pgs;
	pis->penum = penum;
	pis->pgs = pgs;
	pis->pdata = pdata;
	pis->charpath_flag = charpath_flag;
	pis->paint_type = paint_type;
	pis->os_count = 0;
	pis->ipstack[0].ip = str;
	pis->ipstack[0].dstate = crypt_charstring_seed;
	pis->ips_count = 1;
	pis->ip_skip = pdata->lenIV;
	pis->seac_base = -1;
	gs_type1_init_matrix(pis);
	/* Set the current point of the path to the origin, */
	/* in anticipation of the initial [h]sbw. */
	{ gx_path *ppath = pgs->path;
	  ppath->position.x = pgs->ctm.tx_fixed;
	  ppath->position.y = pgs->ctm.ty_fixed;
	}
	return 0;
}

/* Tracing for type 1 interpreter */
#ifdef DEBUG
#  define dc(str) if ( gs_debug['1'] ) type1_trace(cip, c, str);
private void
type1_trace(byte *cip, byte c, char _ds *str)
{	dprintf3("[1]%lx: %02x %s\n", (ulong)(cip - 1), c, (char *)str);
}
#else
#  define dc(str)
#endif

/* Define the state used by operator procedures. */
/* These macros refer to a current instance (s) of gs_op1_state. */
#define sppath s.ppath
#define sfc s.fc
#define ptx s.px
#define pty s.py
#define csp s.osp
#define clear csp = cstack - 1

/* Accumulate relative coordinates */
/****** THESE ARE NOT ACCURATE FOR NON-INTEGER DELTAS. ******/
/* This probably doesn't make any difference in practice. */
#define c_fixed(d, c)\
  (sfc.shift > 0 ? arith_rshift(arg2int(d) * c + sfc.round, sfc.shift) :\
   arg2int(d) * c)
#define accum_x(dx)\
    ptx += c_fixed(dx, sfc.xx);\
    if ( sfc.skewed ) pty += c_fixed(dx, sfc.xy)
#define accum_y(dy)\
    pty += c_fixed(dy, sfc.yy);\
    if ( sfc.skewed ) ptx += c_fixed(dy, sfc.yx)
#define accum_xy(dx,dy)\
    ptx += c_fixed(dx, sfc.xx);\
    pty += c_fixed(dy, sfc.yy);\
    if ( sfc.skewed ) pty += c_fixed(dx, sfc.xy), ptx += c_fixed(dy, sfc.yx)

/* ------ Operator procedures ------ */

/* We put these before the interpreter to save having to write */
/* prototypes for all of them. */

#define s (*ps)
#define arg2int(i) i

int
gs_op1_vmoveto(register is_ptr ps, int dy)
{	accum_y(dy);
	clear;
	return gx_path_add_point(sppath, ptx, pty);
}

int
gs_op1_rlineto(register is_ptr ps, int dx, int dy)
{	accum_xy(dx, dy);
	clear;
	return gx_path_add_line(sppath, ptx, pty);
}

int
gs_op1_hlineto(register is_ptr ps, int dx)
{	accum_x(dx);
	clear;
	return gx_path_add_line(sppath, ptx, pty);
}

int
gs_op1_vlineto(register is_ptr ps, int dy)
{	accum_y(dy);
	clear;
	return gx_path_add_line(sppath, ptx, pty);
}

int
gs_op1_rrcurveto(register is_ptr ps,
  int dx1, int dy1, int dx2, int dy2, int dx3, int dy3)
{	fixed x1, y1, x2, y2;
	accum_xy(dx1, dy1);
	x1 = ptx, y1 = pty;
	accum_xy(dx2, dy2);
	x2 = ptx, y2 = pty;
	accum_xy(dx3, dy3);
	clear;
	return gx_path_add_curve(sppath, x1, y1, x2, y2, ptx, pty);
}

int
gs_op1_closepath(register is_ptr ps)
{	/* Note that this does NOT reset the current point! */
	int code = gx_path_close_subpath(sppath);
	if ( code < 0 ) return code;
	clear;
	return gx_path_add_point(sppath, ptx, pty);	/* put the point where it was */
}

int
gs_op1_sbw(register is_ptr ps, int sbx, int sby, int wx, int wy)
{	register gs_type1_state *pis = ps->pis;
	pis->lsb.x = int2fixed(sbx), pis->lsb.y = int2fixed(sby);
	pis->width.x = int2fixed(wx), pis->width.y = int2fixed(wy);
	accum_xy(sbx, sby);
	clear;
	return 0;
}

int
gs_op1_hsbw(register is_ptr ps, int sbx, int wx)
{	register gs_type1_state *pis = ps->pis;
	pis->lsb.x = int2fixed(sbx), pis->lsb.y = 0;
	pis->width.x = int2fixed(wx), pis->width.y = 0;
	accum_x(sbx);
	clear;
	return 0;
}

int
gs_op1_rmoveto(register is_ptr ps, int dx, int dy)
{	accum_xy(dx, dy);
	clear;
	return gx_path_add_point(sppath, ptx, pty);
}

int
gs_op1_hmoveto(register is_ptr ps, int dx)
{	accum_x(dx);
	clear;
	return gx_path_add_point(sppath, ptx, pty);
}

int
gs_op1_vhcurveto(register is_ptr ps, int dy1, int dx2, int dy2, int dx3)
{	fixed x1, y1, x2, y2;
	accum_y(dy1);
	x1 = ptx, y1 = pty;
	accum_xy(dx2, dy2);
	x2 = ptx, y2 = pty;
	accum_x(dx3);
	clear;
	return gx_path_add_curve(sppath, x1, y1, x2, y2, ptx, pty);
}

int
gs_op1_hvcurveto(register is_ptr ps, int dx1, int dx2, int dy2, int dy3)
{	fixed x1, y1, x2, y2;
	accum_x(dx1);
	x1 = ptx, y1 = pty;
	accum_xy(dx2, dy2);
	x2 = ptx, y2 = pty;
	accum_y(dy3);
	clear;
	return gx_path_add_curve(sppath, x1, y1, x2, y2, ptx, pty);
}

#undef s
#undef arg2int
#define arg2int(f) fixed2int_var(f)

/* ------ Main interpreter ------ */

/* Continue interpreting a Type 1 CharString. */
/* If str != 0, it is taken as the byte string to interpret. */
/* Return 0 on successful completion, <0 on error, */
/* (code<<1)+1 for seac, or (N+1)<<1 for callothersubr(N). */
int
gs_type1_interpret(register gs_type1_state *pis, byte *str)
{	gs_state *pgs = pis->pgs;
	gs_type1_data *pdata = pis->pdata;
	gs_op1_state s;
#undef csp
	register fixed _ss *csp;
	ip_state *ipsp = &pis->ipstack[pis->ips_count - 1];
	int skip = pis->ip_skip;
	register byte *cip;
	register crypt_state state;		/* decryption state */
	register int c;
	int code;
	fixed ftx = pgs->ctm.tx_fixed, fty = pgs->ctm.ty_fixed;

	sppath = pgs->path;
	s.pis = pis;
	sfc = pis->fc;
	ptx = sppath->position.x;
	pty = sppath->position.y;

	/* Copy the operand stack out of the saved state. */
	if ( pis->os_count == 0 )
	   {	clear;
	   }
	else
	   {	memcpy(cstack, pis->ostack, pis->os_count * sizeof(fixed));
		csp = &cstack[pis->os_count - 1];
	   }

	if ( str != 0 )
		ipsp->ip = str;
itop:	cip = ipsp->ip;
	state = ipsp->dstate;
top:	/* Skip initial random bytes */
	for ( ; skip > 0; --skip )
	   {	decrypt_next(*cip, state, c); ++cip;
	   }
	while ( 1 )
	 { decrypt_next(*cip, state, c); ++cip;
	   switch ( (char_command)c )
	   {
	case c_hstem: dc("hstem")	/* hint, ignore */
		clear; break;
	case c_vstem: dc("vstem")	/* hint, ignore */
		clear; break;
	case c_vmoveto: dc("vmoveto");
		code = gs_op1_vmoveto(&s, ics0);
		goto moved;
	case c_rlineto: dc("rlineto")
		code = gs_op1_rlineto(&s, ics0, ics1);
moved:		if ( code < 0 ) return code;
pp:
#ifdef DEBUG
if ( gs_debug['1'] )
		dprintf2("[1]pt=(%g,%g)\n",
			 fixed2float(ptx), fixed2float(pty));
#endif
		clear; break;
	case c_hlineto: dc("hlineto")
		code = gs_op1_hlineto(&s, ics0);
		goto moved;
	case c_vlineto: dc("vlineto")
		code = gs_op1_vlineto(&s, ics0);
		goto moved;
	case c_rrcurveto: dc("rrcurveto")
		code = gs_op1_rrcurveto(&s, ics0, ics1, ics2, ics3, ics4, ics5);
		goto moved;
	case c_closepath: dc("closepath")
		code = gs_op1_closepath(&s);
		goto moved;
	case c_callsubr: dc("callsubr")
	   {	int index = fixed2int_var(*csp);
		byte *nip;
		code = (*pdata->subr_proc)(pdata, index, &nip);
		if ( code < 0 ) return code;
		--csp;
		ipsp->ip = cip, ipsp->dstate = state;
		++ipsp;
		cip = nip;
	   }
		state = crypt_charstring_seed;
		skip = pis->pdata->lenIV;
		goto top;
	case c_return: dc("return")
		--ipsp;
		goto itop;
	case c_escape: dc("escape:")
		decrypt_next(*cip, state, c); ++cip;
		switch ( (char_extended_command)c )
		   {
		case ce_dotsection: dc("  dotsection")	/* hint, ignore */
			clear; break;
		case ce_vstem3: dc("  vstem3")		/* hint, ignore */
			clear; break;
		case ce_hstem3: dc("  hstem3")		/* hint, ignore */
			clear; break;
		case ce_seac: dc("  seac")
			/* Do the accent now.  When it finishes */
			/* (detected in endchar), do the base character. */
			pis->seac_base = (int)(byte)fixed2int_var(cs3);
			/* Adjust the origin of the coordinate system */
			/* for the accent (endchar puts it back). */
			ptx = ftx, pty = fty;
			cs1 -= cs0;	 /* subtract off asb */
			accum_xy(cs1, cs2);
			sppath->position.x = ptx;
			sppath->position.y = pty;
			clear;
			/* Give control back to the caller, who must */
			/* re-invoke the interpreter with the seac string. */
			ipsp->dstate = crypt_charstring_seed;
			pis->ip_skip = pis->pdata->lenIV;
			return ((int)(byte)fixed2int_var(cs4) << 1) + 1;
		case ce_sbw: dc("  sbw")
			gs_op1_sbw(&s, ics0, ics1, ics2, ics3);
			goto pp;
		case ce_div: dc("  div")
			csp[-1] = float2fixed((float)csp[-1] / (float)*csp);
			--csp; break;
		case ce_undoc15: dc("  undoc15")
			/*
			 * NOTE: this opcode is not documented by Adobe,
			 * but is used in some Adobe fonts.  I have no idea
			 * what it is supposed to do.
			 */
			clear; break;
		case ce_callothersubr: dc("  callothersubr")
		   {	int index = fixed2int_var(*csp);
			int scount = csp - cstack;
			/* Update path position so it will be right */
			/* when we come back in. */
			sppath->position.x = ptx;
			sppath->position.y = pty;
			/* Exit to caller */
			ipsp->ip = cip, ipsp->dstate = state;
			pis->os_count = scount;
			pis->ips_count = ipsp - &pis->ipstack[0] + 1;
			pis->ip_skip = 0;
			if ( scount )
				memcpy(pis->ostack, cstack,
				       scount * sizeof(fixed));
			return (index + 1) << 1;
		   }
		case ce_pop: dc("  pop")
			++csp;
			code = (*pdata->pop_proc)(pdata, csp);
			if ( code < 0 ) return code;
			goto pushed;
		case ce_setcurrentpoint: dc("  setcurrentpoint")
			ptx = ftx, pty = fty;
			accum_xy(cs0, cs1);
			goto pp;
		default:
			return_error(gs_error_invalidfont);
		   }
		break;
	case c_hsbw: dc("hsbw")
		gs_op1_hsbw(&s, ics0, ics1);
		goto pp;
	case c_endchar: dc("endchar")
		if ( pis->seac_base >= 0 )
		   {	/* We just finished the accent of a seac. */
			/* Do the base character. */
			int base_code = pis->seac_base;
			pis->seac_base = -1;
			/* Restore the coordinate system origin */
			sppath->position.x = pgs->ctm.tx_fixed;
			sppath->position.y = pgs->ctm.ty_fixed;
			clear;
			/* Clear the ipstack, in case the accent ended */
			/* inside a subroutine. */
			pis->ips_count = 1;
			ipsp = &pis->ipstack[0];
			/* Give control back to the caller, who must */
			/* re-invoke the interpreter with the */
			/* base character string. */
			ipsp->dstate = crypt_charstring_seed;
			pis->ip_skip = pis->pdata->lenIV;
			return (base_code << 1) + 1;
		   }
		/* Set the current point to the character origin: */
		/* the 'show' loop will take care of adding in */
		/* the width we supply to setcharwidth/cachedevice. */
		gx_path_add_point(sppath,
				  pgs->ctm.tx_fixed, pgs->ctm.ty_fixed);
		if ( pis->charpath_flag )
		   {	code = gs_setcharwidth(pis->penum, pgs,
					fixed2float(pis->width.x),
					fixed2float(pis->width.y));
			if ( code < 0 ) return code;
			/* Merge the path into its parent */
			code = gx_path_merge(sppath, pgs->saved->path);
		   }
		else
		   {	gs_rect bbox;
			code = gs_pathbbox(pgs, &bbox);
			/* Restore the current point to where it was */
			/* at the beginning of the character, so that */
			/* the right thing will happen when we copy */
			/* a just-cached character to the output. */
			gx_path_add_point(sppath, ftx, fty);
			if ( code < 0 )		/* must be a null path */
			   {	bbox.p.x = bbox.p.y = bbox.q.x = bbox.q.y = 0;
			   }
#ifdef DEBUG
if ( gs_debug['1'] )
			dprintf4("[1]bbox=(%g,%g),(%g,%g)\n",
				 bbox.p.x, bbox.p.y, bbox.q.x, bbox.q.y);
#endif
			if ( pis->paint_type )
			   {	/* Expand the bounding box to encompass */
				/* the width of the stroke, plus a little */
				/* to overcome rounding problems. */
				float adjust = gs_currentlinewidth(pgs);
				if ( adjust == 0 ) adjust = 1;
				bbox.p.x -= adjust;
				bbox.p.y -= adjust;
				bbox.q.x += adjust;
				bbox.q.y += adjust;
			   }
			code = gs_setcachedevice(pis->penum, pgs,
					fixed2float(pis->width.x),
					fixed2float(pis->width.y),
					bbox.p.x, bbox.p.y,
					bbox.q.x, bbox.q.y);
			if ( code < 0 ) return code;
			/* We've already constructed the path: */
			/* translate it so it matches the cache device. */
			gx_path_translate(pgs->path, pgs->ctm.tx_fixed - ftx,
					  pgs->ctm.ty_fixed - fty);
			if ( code < 0 ) return code;
			/******
			 ****** The adjust parameter is a hack to make
			 ****** characters come out more bold, since we
			 ****** don't look at the hints.
			 ******/
			gx_color_load(pgs->dev_color, pgs);
			code = (pis->paint_type ? gs_stroke(pgs) :
				gs_fill_adjust(pgs, float2fixed(0.25)));
		   }
		return code;
	case c_rmoveto: dc("rmoveto");
		code = gs_op1_rmoveto(&s, ics0, ics1);
		goto moved;
	case c_hmoveto: dc("hmoveto");
		code = gs_op1_hmoveto(&s, ics0);
		goto moved;
	case c_vhcurveto: dc("vhcurveto")
		code = gs_op1_vhcurveto(&s, ics0, ics1, ics2, ics3);
		goto moved;
	case c_hvcurveto: dc("hvcurveto")
		code = gs_op1_hvcurveto(&s, ics0, ics1, ics2, ics3);
		goto moved;
	/* Fill up the dispatch up to 32 */
	case c_undef0: case c_undef2:
	case c_undef15:
	case c_undef16: case c_undef17: case c_undef18: case c_undef19:
	case c_undef20: case c_undef23:
	case c_undef24: case c_undef25: case c_undef26: case c_undef27:
	case c_undef28: case c_undef29:
		return_error(gs_error_invalidfont);
	default:			/* a number */
		csp++;
		if ( c <= c_max_num1 )
			*csp = int2fixed(c_value_num1(c));
		else
		   {	byte c0;
			decrypt_next(*cip, state, c0); ++cip;
			if ( c <= c_max_num2 )
				*csp = int2fixed(c_value_num2(c, (int)c0));
			else if ( c <= c_max_num3 )
				*csp = int2fixed(c_value_num3(c, (int)c0));
			else			/* c = 255 */
			   {	byte c1, c2;
				long lw;
				decrypt_next(*cip, state, c1); ++cip;
				decrypt_next(*cip, state, c2); ++cip;
				decrypt_next(*cip, state, lw); ++cip;
				lw += (long)c0 << 24;
				lw += (uint)c1 << 16;
				lw += (uint)c2 << 8;
				*csp = int2fixed(lw);
				if ( lw != fixed2int_var(*csp) )
					return_error(gs_error_rangecheck);
			   }
		   }
pushed:
#ifdef DEBUG
if ( gs_debug['1'] )
		dprintf3("[1]%d: (%d) %f\n",
			 (int)(csp - cstack), c, fixed2float(*csp));
#endif
		;			/* in case no debug */
	   }
	 }
}

/* Pop a (fixed) number off the internal stack. */
/* The client uses this to get the arguments for an OtherSubr. */
int
gs_type1_pop(gs_type1_state *pis, fixed *pf)
{	*pf = pis->ostack[--(pis->os_count)];
	return 0;
}
