/* Copyright (C) 1989, 1990, 1991 Aladdin Enterprises.  All rights reserved.
   Distributed by Free Software Foundation, Inc.

This file is part of Ghostscript.

Ghostscript is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY.  No author or distributor accepts responsibility
to anyone for the consequences of using it or for whether it serves any
particular purpose or works at all, unless he says so in writing.  Refer
to the Ghostscript General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
Ghostscript, but only under the conditions described in the Ghostscript
General Public License.  A copy of this license is supposed to have been
given to you along with Ghostscript so you can know your rights and
responsibilities.  It should be in a file named COPYING.  Among other
things, the copyright notice and this notice must be preserved on all
copies.  */

/* gp_dos.c */
/* DOS-specific routines for Ghostscript */
#include <dos.h>
#include <fcntl.h>
#include <dir.h>
#include <signal.h>
#include "string_.h"
#include "gx.h"
#include "gp.h"
#ifdef __OVERLAY__
#  include "overlay.h"
#endif

/* Define the size of the C stack. */
unsigned _stklen = 8000;		/* default is 4096, we need more */

/* Define the size of the overlay buffer, if relevant. */
#ifdef __OVERLAY__
unsigned _ovrbuffer = (1024L * OVLBUFK) / 16;
#endif

/* Forward declarations */
private void handle_FPE(P3(int, int, int *));

/* Do platform-dependent initialization */
#if CPU_TYPE > 86
/* Internal routine to set flags and read them back. */
/* We use __emit__ so we don't require an assembler. */
private int
push_pop_flags(unsigned flags)
{	__emit__(0x8b, 0x46, 6);	/* mov ax,flags */
	__emit__(0x50, 0x9d);		/* push ax; popf */
	__emit__(0x9c, 0x58);		/* pushf; pop ax */
}
#endif
void
gp_init()
{	/*
	 * Detect the processor type using the following algorithms:
	 *	The 8088/8086 truncate shift counts mod 32,
	 *	  the 80186 and up do not.
	 *	The 80186 and below fix FLAGS bits 15-12 to 1,
	 *	  the 80286 and up do not.
	 *	The 80386 allows setting FLAGS bits 14-12,
	 *	  the 80286 and below do not.
	 * We currently can't tell an 80386 from an 80486.
	 * Note that this algorithm will identify an 80386
	 *   running in Virtual 8086 mode as an 80386.
	 *   This is acceptable, because Ghostscript doesn't actually
	 *   use 80286 or 80386 addressing modes, only the additional
	 *   instructions available on these processors.
	 * (This algorithm is derived from the Intel manuals.)
	 */
#if CPU_TYPE > 86
	/* We have to be careful not to turn interrupts off! */
	int result, type;
	result = push_pop_flags(0x202);
	if ( (result & 0xf000) == 0xf000 )
	   {	/* CPU is an 8088/8086/80186 */
		   {	int shc = 33;	/* force shift by variable */
			result = 0xffff << shc;
		   }
		type = (result == 0 ? 186 : 86);
	   }
	else
	   {	/* CPU is an 80286/80386/... */
		result = push_pop_flags(0x7202);
		type = ((result & 0x7000) == 0 ? 286 : 386);
	   }
	/* A 486 is the same as a 386. */
#define CPU_EQUIV (CPU_TYPE == 486 ? 386 : CPU_TYPE)
	if ( type < CPU_EQUIV )
	   {	eprintf1("This executable requires an 80%d or higher.\n",
			 CPU_EQUIV);
		exit(1);
	   }
#endif
	_fmode = O_BINARY;		/* Open files in 'binary' mode */

#ifdef __OVERLAY__
	/* Initialize the overlay machinery. */
	   {	int code;
#  ifdef OVEMS
		code = _OvrInitEms(OVEMS_HANDLE, OVEMS_FIRST, OVEMS_PAGES);
		if ( code )
			eprintf("Attempt to use EMS memory for overlays failed.\n");
#  endif
#  ifdef OVEXT
		code = _OvrInitExt(OVEXT_START, OVEXT_LENGTH);
		if ( code )
			eprintf("Attempt to use extended memory for overlays failed.\n");
#  endif
	   }
#endif
	/* Set up the handler for numeric exceptions. */
	signal(SIGFPE, handle_FPE);
}

/* Trap numeric exceptions.  Someday we will do something */
/* more appropriate with these. */
private void
handle_FPE(int sig, int subcode, int *regs)
{	eprintf("Numeric exception:\n");
	fprintf(estderr,
"AX=%04x  BX=%04x  CX=%04x  DX=%04x  SI=%04x  DI=%04x  BP=%04x\n",
		regs[8], regs[7], regs[6], regs[5], regs[2], regs[1], regs[0]);
	fprintf(estderr,
"DS=%04x  ES=%04x  CS:IP=%04x:%04x\n",
		regs[3], regs[4], regs[10], regs[9]);
	exit(1);
}

/* Read the current date (in days since Jan. 1, 1980) */
/* and time (in milliseconds since midnight). */
void
gp_get_clock(long *pdt)
{	struct date osdate;
	struct time ostime;
	long idate;
	static int mstart[12] =
	   { 0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334 };
	getdate(&osdate);
	gettime(&ostime);
	idate = (long)osdate.da_year * 365 +
		(osdate.da_year / 4 + 1 +	/* account for leap years */
		 mstart[osdate.da_mon - 1] +	/* month is 1-origin */
		 osdate.da_day - 1);		/* day of month is 1-origin */
	if ( osdate.da_mon <= 2 && osdate.da_year % 4 == 0 )		/* Jan. or Feb. of leap year */
		idate--;
	pdt[0] = idate;
	pdt[1] =
		(ostime.ti_hour * 60 + ostime.ti_min) * 60000L +
		(ostime.ti_sec * 100 + ostime.ti_hund) * 10L;
}

/* ------ File name syntax ------ */

/* Define the character used for separating file names in a list. */
char gp_file_name_list_separator = ';';

/* Define the default scratch file name template. */
char gp_scratch_file_name_template[] = "_temp_XXXXXX";

/* Answer whether a file name contains a directory/device specification, */
/* i.e. is absolute (not directory- or device-relative). */
int
gp_file_name_is_absolute(char *fname, uint len)
{	/* A file name is absolute if it contains a drive specification */
	/* (second character is a :) or if it start with / or \. */
	return ( len >= 1 && (*fname == '/' || *fname == '\\' ||
		(len >= 2 && fname[1] == ':')) );
}

/* Answer the string to be used for combining a directory/device prefix */
/* with a base file name.  The file name is known to not be absolute. */
char *
gp_file_name_concat_string(char *prefix, uint plen, char *fname, uint len)
{	if ( plen > 0 )
	  switch ( prefix[plen - 1] )
	   {	case ':': case '/': case '\\': return "";
	   };
	return "\\";
}

/* ------ File enumeration ------ */

struct file_enum_s {
	struct ffblk ffblk;
	char *pattern;
	int first_time;
	gs_memory_procs mprocs;
};

/* Initialize an enumeration.  NEEDS WORK ON HANDLING * ? \. */
file_enum *
gp_enumerate_files_init(char *pat, uint patlen, proc_alloc_t palloc, proc_free_t pfree)
{	file_enum *pfen = (file_enum *)(*palloc)(1, sizeof(file_enum), "gp_enumerate_files");
	char *pattern;
	if ( pfen == 0 ) return 0;
	pattern = (*palloc)(patlen + 1, 1,
			    "gp_enumerate_files(pattern)");
	if ( pattern == 0 ) return 0;
	memcpy(pattern, pat, patlen);
	pattern[patlen] = 0;
	pfen->pattern = pattern;
	pfen->mprocs.alloc = palloc;
	pfen->mprocs.free = pfree;
	pfen->first_time = 1;
	return pfen;
}

/* Enumerate the next file. */
/* DOESN'T PREFIX THE DRIVE & DIRECTORY -- THIS IS WRONG. */
uint
gp_enumerate_files_next(file_enum *pfen, char *ptr, uint maxlen)
{	int code;
	char *p, *q;
	char *dta = getdta();		/* just in case! */
	if ( pfen->first_time )
	   {	code = findfirst(pfen->pattern, &pfen->ffblk, 0);
		pfen->first_time = 0;
	   }
	else
		code = findnext(&pfen->ffblk);
	setdta(dta);
	if ( code < 0 )
	   {	/* All done, clean up. */
		gp_enumerate_files_close(pfen);
		return ~(uint)0;
	   }
	if ( maxlen < 13 ) return maxlen + 1;	/* cop out! */
	for ( p = &pfen->ffblk.ff_name[0], q = ptr; *p; p++ )
	  if ( *p != ' ' ) *q++ = *p;
	return q - ptr;
}

/* Clean up the file enumeration. */
void
gp_enumerate_files_close(file_enum *pfen)
{	proc_free_t pfree = pfen->mprocs.free;
	(*pfree)(pfen->pattern, strlen(pfen->pattern) + 1, 1,
		 "gp_enumerate_files_close(pattern)");
	(*pfree)((char *)pfen, 1, sizeof(file_enum), "gp_enumerate_files_close");
}
