/*
 * SOFA utility methods for inspecting SOFA file metrics and determining HRTF
 * utility compatible layouts.
 *
 * Copyright (C) 2018-2019  Christopher Fitzgerald
 * Copyright (C) 2019  Christopher Robinson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Or visit:  http://www.gnu.org/licenses/old-licenses/gpl-2.0.html
 */

#include "config.h"

#include "sofa-support.h"

#include <algorithm>
#include <array>
#include <cmath>
#include <cstdio>
#include <span>
#include <utility>
#include <vector>

#include "fmt/base.h"
#include "fmt/ranges.h"

#include "mysofa.h"

namespace {

using double3 = std::array<double,3>;


/* Produces a sorted array of unique elements from a particular axis of the
 * triplets array.  The filters are used to focus on particular coordinates
 * of other axes as necessary.  The epsilons are used to constrain the
 * equality of unique elements.
 */
std::vector<double> GetUniquelySortedElems(const std::vector<double3> &aers, const unsigned axis,
    const std::array<const double*,3> &filters, const std::array<double,3> &epsilons)
{
    std::vector<double> elems;
    for(const double3 &aer : aers)
    {
        const double elem{aer[axis]};

        auto j = unsigned{};
        for(j = 0;j < 3;j++)
        {
            if(filters[j] && std::abs(aer[j] - *filters[j]) > epsilons[j])
                break;
        }
        if(j < 3)
            continue;

        auto iter = elems.begin();
        for(;iter != elems.end();++iter)
        {
            const double delta{elem - *iter};
            if(delta > epsilons[axis]) continue;
            if(delta >= -epsilons[axis]) break;

            iter = elems.emplace(iter, elem);
            break;
        }
        if(iter == elems.end())
            elems.emplace_back(elem);
    }
    return elems;
}

/* Given a list of azimuths, this will produce the smallest step size that can
 * uniformly cover the list. Ideally this will be over half, but in degenerate
 * cases this can fall to a minimum of 5 (the lower limit).
 */
double GetUniformAzimStep(const double epsilon, const std::vector<double> &elems)
{
    if(elems.size() < 5) return 0.0;

    /* Get the maximum count possible, given the first two elements. It would
     * be impossible to have more than this since the first element must be
     * included.
     */
    auto count = static_cast<unsigned>(std::ceil(360.0 / (elems[1]-elems[0])));
    count = std::min(count, 255u);

    for(;count >= 5;--count)
    {
        /* Given the stepping value for this number of elements, check each
         * multiple to ensure there's a matching element.
         */
        const double step{360.0 / count};
        bool good{true};
        size_t idx{1u};
        for(auto mult=1u;mult < count && good;++mult)
        {
            const double target{step*mult + elems[0]};
            while(idx < elems.size() && target-elems[idx] > epsilon)
                ++idx;
            good &= (idx < elems.size()) && !(std::abs(target-elems[idx++]) > epsilon);
        }
        if(good)
            return step;
    }
    return 0.0;
}

/* Given a list of elevations, this will produce the smallest step size that
 * can uniformly cover the list. Ideally this will be over half, but in
 * degenerate cases this can fall to a minimum of 5 (the lower limit).
 */
double GetUniformElevStep(const double epsilon, std::vector<double> &elems)
{
    if(elems.size() < 5) return 0.0;

    /* Reverse the elevations so it increments starting with -90 (flipped from
     * +90). This makes it easier to work out a proper stepping value.
     */
    std::ranges::reverse(elems);
    for(auto &v : elems) v *= -1.0;

    auto count = static_cast<unsigned>(std::ceil(180.0 / (elems[1]-elems[0])));
    count = std::min(count, 255u);

    double ret{0.0};
    for(;count >= 5;--count)
    {
        const double step{180.0 / count};
        bool good{true};
        size_t idx{1u};
        /* Elevations don't need to match all multiples if there's not enough
         * elements to check. Missing elevations can be synthesized.
         */
        for(auto mult=1u;mult <= count && idx < elems.size() && good;++mult)
        {
            const double target{step*mult + elems[0]};
            while(idx < elems.size() && target-elems[idx] > epsilon)
                ++idx;
            good &= !(idx < elems.size()) || !(std::abs(target-elems[idx++]) > epsilon);
        }
        if(good)
        {
            ret = step;
            break;
        }
    }
    /* Re-reverse the elevations to restore the correct order. */
    for(auto &v : elems) v *= -1.0;
    std::ranges::reverse(elems);

    return ret;
}

} // namespace


const char *SofaErrorStr(int err)
{
    switch(err)
    {
    case MYSOFA_OK: return "OK";
    case MYSOFA_INVALID_FORMAT: return "Invalid format";
    case MYSOFA_UNSUPPORTED_FORMAT: return "Unsupported format";
    case MYSOFA_INTERNAL_ERROR: return "Internal error";
    case MYSOFA_NO_MEMORY: return "Out of memory";
    case MYSOFA_READ_ERROR: return "Read error";
    }
    return "Unknown";
}

auto GetCompatibleLayout(const std::span<const float> xyzs) -> std::vector<SofaField>
{
    auto aers = std::vector(xyzs.size()/3, double3{});
    for(size_t i{0u};i < aers.size();++i)
    {
        std::array vals{xyzs[i*3], xyzs[i*3 + 1], xyzs[i*3 + 2]};
        mysofa_c2s(vals.data());
        aers[i] = {vals[0], vals[1], vals[2]};
    }

    const auto radii = GetUniquelySortedElems(aers, 2, {}, {0.1, 0.1, 0.001});
    auto fds = std::vector<SofaField>{};
    fds.reserve(radii.size());

    for(const double dist : radii)
    {
        auto elevs = GetUniquelySortedElems(aers, 1, {nullptr, nullptr, &dist}, {0.1, 0.1, 0.001});

        /* Remove elevations that don't have a valid set of azimuths. */
        std::erase_if(elevs, [&dist,&aers](const double ev) -> bool
        {
            const auto azims = GetUniquelySortedElems(aers, 0, {nullptr, &ev, &dist},
                {0.1, 0.1, 0.001});

            if(std::abs(ev) > 89.999)
                return azims.size() != 1;
            if(azims.empty() || !(std::abs(azims[0]) < 0.1))
                return true;
            return GetUniformAzimStep(0.1, azims) <= 0.0;
        });

        double step{GetUniformElevStep(0.1, elevs)};
        if(step <= 0.0)
        {
            if(elevs.empty())
                fmt::println("No usable elevations on field distance {:f}.", dist);
            else
            {
                fmt::println("Non-uniform elevations on field distance {:.3f}.\nGot: {:+.2f}",
                    dist, fmt::join(elevs, ", "));
            }
            continue;
        }

        auto evStart = 0u;
        for(auto ei=0u;ei < elevs.size();ei++)
        {
            if(!(elevs[ei] < 0.0))
            {
                fmt::println("Too many missing elevations on field distance {:f}.", dist);
                return fds;
            }

            const auto eif = (90.0+elevs[ei]) / step;
            const auto ev_start = std::round(eif);

            if(std::abs(eif - ev_start) < (0.1/step))
            {
                evStart = static_cast<unsigned>(ev_start);
                break;
            }
        }

        const auto evCount = static_cast<unsigned>(std::round(180.0 / step)) + 1;
        if(evCount < 5)
        {
            fmt::println("Too few uniform elevations on field distance {:f}.", dist);
            continue;
        }

        SofaField field{};
        field.mDistance = dist;
        field.mEvCount = evCount;
        field.mEvStart = evStart;
        field.mAzCounts.resize(evCount, 0u);
        auto &azCounts = field.mAzCounts;

        for(unsigned ei{evStart};ei < evCount;ei++)
        {
            double ev{-90.0 + ei*180.0/(evCount - 1)};
            auto azims = GetUniquelySortedElems(aers, 0, {nullptr, &ev, &dist}, {0.1, 0.1, 0.001});

            if(ei == 0 || ei == (evCount-1))
            {
                if(azims.size() != 1)
                {
                    fmt::println("Non-singular poles on field distance {:f}.", dist);
                    return fds;
                }
                azCounts[ei] = 1;
            }
            else
            {
                step = GetUniformAzimStep(0.1, azims);
                if(step <= 0.0)
                {
                    fmt::println("Non-uniform azimuths on elevation {:f}, field distance {:f}.",
                        ev, dist);
                    return fds;
                }
                azCounts[ei] = static_cast<unsigned>(std::round(360.0f / step));
            }
        }

        fds.emplace_back(std::move(field));
    }

    return fds;
}
