/***************************************************************************
** Although considerable effort has been expended to make this software   **
** correct and reliable, no warranty is implied; the author disclaims any **
** obligation or liability for damages, including but not limited to      **
** special, indirect, or consequential damages arising out of or in       **
** connection with the use or performance of this software.               **
***************************************************************************/

/*
 *	This file contains the routines that implement the
 *	'special' TeX DVI file command.
 *
 *	The 'paste' command includes graphics in the text:
 *
 *	PASTE file protocol width height [/switch ... ]
 *
 *	where:
 *
 *	    'file' is a file containing graphics primitives data
 *	    represented in a particular way, such as a CORE metafile,
 *	    Tektronix 4014 graphics data file, PostScript, etc.
 *
 *	    'protocol' is a name describing the protocol used. Currently,
 *	    this can be 'tek4014', 'macpaint', 'paint', or 'postscript'.
 *
 *	    'width' is the width of the box to contain the graphics.
 *	    This value must be a recognizable dimension.
 *
 *	    'depth' is the depth of the box to contain the graphics.
 *	    This value must be a recognizable dimension.
 *
 *	The following switches further define the PASTE operation:
 *
 *	/ORIENTATION={UP,DOWN,LEFT,RIGHT}	default: UP
 *	/PICTURE=picture_number			default: 1
 *	/OPTIONS=list                           device dependent
 *
 *	The 'color' command changes the color of the output:
 *
 *	COLOR hue
 *
 *	where 'hue' can one of: RED,GREEN,BLUE,YELLOW,CYAN,MAGENTA,
 *	                        BLACK or WHITE.
 *
 *	Additionally, special hue specifications detailing color mix
 *	can be used. These are:
 *
 *	    RGB=red:green:blue
 *	    HLS=hue:lightness:saturation
 *
 *	The red, green and blue specifications are values between 0 and
 *	100 indicating the percent of each color in the resulting mix. These
 *	values can be fractional, if needed. Alternatively, the hue (0-360),
 *	lightness (0-100) and saturation (0-100) can be specified. Conversion
 *	of an hls specification to rgb specification is done according to
 *	the algorithm described in the CORE specification (ACM SIGGRAPH,
 *	Volume 13, Number 3, August 1979, "Status Report of the Graphics
 *	Standards Planning Committee", Appendix B). The default value
 *	for hue, if it is not specified, is black.
 *
 *	This command has an effect only if the current output device can
 *	support color. On some output devices, this may affect the gray
 *	level of the output.
 *
 *	The 'lwidth' command changes the current linewidth:
 *
 *	LWIDTH width
 *
 *	where 'width' is a dimension specifying the new line width.
 *
 *	The 'lstyle' command changes the current linestyle rendition
 *	on the output device:
 *
 *	LSTYLE style
 *
 *	where 'style' can be one of the names 'solid', 'dotted', 'dashed',
 *	or 'dotdash'; or, it can be a dashline specification of the form:
 *
 *	    n1,n2,n3,...
 *
 *	where 'n1','n2', ... specify the lengths of individual segments
 *	of the line in units of a line's width. The first value
 *	corresponds to a visible segment, the second to an invisible
 *	segment, and so on, alternating between visible and invisible.
 *
 *	The 'xydef' command saves the current (h,v) position for use by
 *	other graphics operators:
 *
 *	XYDEF
 *
 *	The 'line' command draws a line on the output page using the
 *	current linestyle amd linewidth:
 *
 *	LINE (x1,y1) (x2,y2)
 *
 *	where:
 *
 *	    (x1,y1) are the coordinates, relative to the current (h,v)
 *	    position, of the first endpoint of the line.
 *
 *	    (x2,y2) are the coordinates of the second endpoint of the line.
 *
 *	The 'point' command draws a single solid point on the output
 *	device:
 *
 *	POINT (x,y) diameter
 *
 *	where:
 *
 *	    (x,y) specifies the coordinates of the point relative to the
 *	    current (h,v) position.
 *
 *	    'diameter' specifies the diameter of the point (dimension).
 *
 *	The 'arc' command draws a circular arc on the output device using
 *	the current linestyle and linewidth:
 *
 *	ARC (x,y) radius start_ang stop_ang
 *
 *	where:
 *
 *	    (x,y) specifies the coordinates of the center of the arc
 *	    relative to the current (h,v) position.
 *
 *	    'radius' is the radius of the arc segment (dimension).
 *
 *          'start_ang' specifies the beginning angle of the arc. Angles
 *	    are measured positive clockwise from the 3 o'clock position.
 *
 *          'stop_ang' specifies the ending angle of the arc. If it is
 *	    less than the start angle, a counter-clockwise arc is drawn.
 *
 *	The 'fill' command generates a filled polygon:
 *
 *	FILL (x1,y1),...,(xn,yn) columns bitsize pattern [/switch...]
 *
 *	where:
 *
 *	    (x1,y1),...,(xn,yn) are the (x,y) coordinates (dimension relative
 *	    to current (h,v) position) of the vertices of the polygon.
 *
 *	    'columns' is the number of columns (width of each row) in the
 *	    bitmap describing the fill pattern. If this is zero, a solid
 *	    pattern is used.
 *
 *	    'bitsize' is a dimension specifying the size of each 'bit' in
 *	    the pattern bitmap.
 *
 *	    'pattern' is a list of hexadecimal values that specify the
 *	    on (1) and off (0) 'bits' of the fill pattern. The fill pattern
 *	    is used to completely fill the interior of the polygon.
 *
 *	The following switch further define the FILL operation:
 *
 *	/PERIMETER      Specifies that the perimeter of the polygon is to
 *			be drawn using current line attributes.
 *
 *	The 'save' command saves the current line style, width, and color.
 *	It has no parameters.
 *
 *	The 'restore' command restores the current line style, width, and
 *	color from a previously saved state.
 */

#include "types.h"
#include "devtab.h"
#include "csi.h"

#include "arith.p"
#include "raster.p"
#include "strng.p"

char Scratch_String[256];

int CSI_Mask_Word;
#define ORIENTATION 0x0001
#define PICTURE     0x0002
#define OPTIONS     0x0004
#define PERIMETER   0x0001

#define OPTION_SIZE 64

struct Arg_List {
	long H_Pos;
	long V_Pos;
	unsigned long Mag;
	int Reflect_Image;
	struct Device_Table *Device;
};

struct XY_Coord {
	struct XY_Coord *Prev;
	long X_Coord;
	long Y_Coord;
} *XY_Top = 0;

Execute_Special (Special_String, Current_H, Current_V, Magnification, Reflected, Device_Ptr)
char *Special_String;
long Current_H, Current_V;
unsigned long Magnification;
int Reflected;
struct Device_Table *Device_Ptr;
{
	auto   int Status;
	static struct CSI_Status_Block CSI_Block;
	static struct Arg_List Args;
	extern int Paste(), Set_Color(), Draw_Line(), Draw_Point();
	extern int Draw_Arc(), Fill_Poly(), Set_LStyle(), Set_LWidth();
	extern int Define_XY(), Save_Attributes(), Restore_Attributes();
	static struct Verb_Table Verb_List[] = {
		csivb ("PASTE",  2, Paste,      &CSI_Mask_Word, 0, 0),
		csivb ("COLOR",  1, Set_Color,  &CSI_Mask_Word, 0, 0),
		csivb ("LWIDTH", 2, Set_LWidth, &CSI_Mask_Word, 0, 0),
		csivb ("LSTYLE", 2, Set_LStyle, &CSI_Mask_Word, 0, 0),
		csivb ("XYDEF",  1, Define_XY,  &CSI_Mask_Word, 0, 0),
		csivb ("LINE",   2, Draw_Line,  &CSI_Mask_Word, 0, 0),
		csivb ("ARC",    1, Draw_Arc,   &CSI_Mask_Word, 0, 0),
		csivb ("POINT",  2, Draw_Point, &CSI_Mask_Word, 0, 0),
		csivb ("FILL",   1, Fill_Poly,  &CSI_Mask_Word, 0, 0),
		csivb ("SAVE",   1, Save_Attributes,    &CSI_Mask_Word, 0, 0),
		csivb ("RESTORE",1, Restore_Attributes, &CSI_Mask_Word, 0, 0),
		csiend
	};
	static char Partial_Command[16];
	extern int Do_Command(), strlen();
	msgcode DVIOUT_SPECIALABORT;
/*
 *	Parse the command, dispatch to executor function:
 */
	Args.H_Pos = Current_H;
	Args.V_Pos = Current_V;
	Args.Mag = Magnification;
	Args.Reflect_Image = Reflected;
	Args.Device = Device_Ptr;
	CSI_Mask_Word = 0;
	Status = Do_Command (Special_String, Verb_List, &Args, &CSI_Block);
	if (Status == 0) {
		stringcpy_m (Partial_Command, Special_String, sizeof (Partial_Command));
		if (strlen (Special_String) >= sizeof (Partial_Command))
			stringcpy_m (&Partial_Command[sizeof(Partial_Command)-4], "...", 4);
		Message (DVIOUT_SPECIALABORT, Partial_Command, 1);
	}
	return (Status);
}

/*
 *	Routine Paste is called for the PASTE command. This routine
 *	determines the protocol used, parses the command arguments,
 *	then dispatches control to the routine responsible for
 *	handling that particular protocol:
 */

int Paste (Arg_Ptr, CSI_Ptr)
struct Arg_List *Arg_Ptr;
struct CSI_Status_Block *CSI_Ptr;
{
	auto   struct Device_Table *Device_Ptr;
	auto   long Width, Depth;
	auto   unsigned int Index, Picture_Number;
	auto   int Orientation, Status, Argn, Option_Count;
	static char *Orientation_List[4] = { "UP", "LEFT", "DOWN", "RIGHT" };
	static char *Option_List[25];
	static char Picture_Value[8], Protocol_Value[12], Orientation_Value[8];
	static char Option_Value[25][OPTION_SIZE], Width_Value[20], Depth_Value[20];
	static struct Switch_Value_Table Picture_Table[] = {
		csisv (Picture_Value[0], sizeof (Picture_Value), "1"),
		csiend
	}, Orientation_Table[] = {
		csisv (Orientation_Value[0], sizeof (Orientation_Value), "UP"),
		csiend
	}, Option_Table[] = {
		csisv (Option_Value[0][0], OPTION_SIZE, 0), csisv (Option_Value[1][0], OPTION_SIZE, 0),
		csisv (Option_Value[2][0], OPTION_SIZE, 0), csisv (Option_Value[3][0], OPTION_SIZE, 0),
		csisv (Option_Value[4][0], OPTION_SIZE, 0), csisv (Option_Value[5][0], OPTION_SIZE, 0),
		csisv (Option_Value[6][0], OPTION_SIZE, 0), csisv (Option_Value[7][0], OPTION_SIZE, 0),
		csisv (Option_Value[8][0], OPTION_SIZE, 0), csisv (Option_Value[9][0], OPTION_SIZE, 0),
		csisv (Option_Value[10][0], OPTION_SIZE, 0), csisv (Option_Value[11][0], OPTION_SIZE, 0),
		csisv (Option_Value[12][0], OPTION_SIZE, 0), csisv (Option_Value[13][0], OPTION_SIZE, 0),
		csisv (Option_Value[14][0], OPTION_SIZE, 0), csisv (Option_Value[15][0], OPTION_SIZE, 0),
		csisv (Option_Value[16][0], OPTION_SIZE, 0), csisv (Option_Value[17][0], OPTION_SIZE, 0),
		csisv (Option_Value[18][0], OPTION_SIZE, 0), csisv (Option_Value[19][0], OPTION_SIZE, 0),
		csisv (Option_Value[20][0], OPTION_SIZE, 0), csisv (Option_Value[21][0], OPTION_SIZE, 0),
		csisv (Option_Value[22][0], OPTION_SIZE, 0), csisv (Option_Value[23][0], OPTION_SIZE, 0),
		csisv (Option_Value[24][0], OPTION_SIZE, 0),
		csiend
	};
	static struct Switch_Table Switch_List[] = {
		csisw ("PICTURE", 1, PICTURE, 0, Picture_Table),
		csisw ("ORIENTATION", 2, ORIENTATION, 0, Orientation_Table),
		csisw ("OPTIONS", 2, OPTIONS, 0, Option_Table),
		csiend
	};
	extern int TK_Paste(), MP_Paste(), PS_Paste();
	static struct {
		char *Protocol_Name;
		int (*Executor)();
		unsigned int Min_Len;
	} Protocol_Desc[] = {
		{ "TEK4014",    &TK_Paste, 1 },
		{ "PAINT",      &MP_Paste, 2 },
		{ "MACPAINT",   &MP_Paste, 4 },
		{ "POSTSCRIPT", &PS_Paste, 2 },
		{ 0 }
	};
	extern long Parse_Dimension(), Convert_Dimension();
	extern unsigned int Parse_Unsigned_Int();
	extern int Get_Parameter();
	msgcode DVIOUT_INVALORIENT, DVIOUT_NEGZEROSIZE, DVIOUT_UNKPROTOCOL,
		DVIOUT_MISSPECARG;
/*
 *	Retrieve command parameters and switch values:
 */
	strcpy (Picture_Value, "1");
	strcpy (Orientation_Value, "UP");
	for (Index = 0; Index < 25; Index++)
		Option_Value[Index][0] = '\0';
	if ((Status = Get_Parameter (Argn=0, 0, 0, 0, Switch_List, CSI_Ptr)) == 0 ||
	    (Status = Get_Parameter (Argn=1, Scratch_String, sizeof (Scratch_String), 0, Switch_List, CSI_Ptr)) <= 0 ||
	    (Status = Get_Parameter (Argn=2, Protocol_Value, sizeof (Protocol_Value), 0, Switch_List, CSI_Ptr)) <= 0 ||
	    (Status = Get_Parameter (Argn=3, Width_Value, sizeof (Width_Value), 0, Switch_List, CSI_Ptr)) <= 0 ||
	    (Status = Get_Parameter (Argn=4, Depth_Value, sizeof (Depth_Value), 0, Switch_List, CSI_Ptr)) <= 0) {
		if (Status < 0)
			Message (DVIOUT_MISSPECARG, Argn, 0, "PASTE", 1);
		return (0);
	}
/*
 *	Translate command parameters to the required forms:
 */
	if ((Picture_Number = Parse_Unsigned_Int (Picture_Value)) == 0)
		Picture_Number = 1;
	if ((Width = Parse_Dimension (Width_Value, 1)) < 0) {
		Message (DVIOUT_NEGZEROSIZE, Width_Value, 1);
		return (0);
	}
	if ((Depth = Parse_Dimension (Depth_Value, 1)) < 0) {
		Message (DVIOUT_NEGZEROSIZE, Depth_Value, 1);
		return (0);
	}
	Orientation = -1;
	for (Index = 0; Index < 4; Index++)
	if (Orientation_Value[0] != '\0' &&
	    Compare_Keyword_M (Orientation_Value, Orientation_List[Index], 1, 4, 0) == 1) {
		Orientation = Index;
		break;
	}
	if (Orientation == -1) {
		Message (DVIOUT_INVALORIENT, Orientation_Value, 1);
		Orientation = 0;
	}
	Option_Count = 0;
	for (Index = 0; Index < 25; Index++)
	if (Option_Value[Index][0] != '\0')
		Option_List[Option_Count++] = &Option_Value[Index][0];
/*
 *	Determine the protocol, dispatch control to the associated
 *	routine:
 */
	Device_Ptr = Arg_Ptr->Device;
	for (Index = 0; Protocol_Desc[Index].Protocol_Name != 0; Index++)
	if (Protocol_Value[0] != '\0' &&
	    Compare_Keyword_M (Protocol_Value, Protocol_Desc[Index].Protocol_Name,
			       Protocol_Desc[Index].Min_Len, 4, 0) == 1) {
		(*Device_Ptr->Save_Graphics) (&Protocol_Desc[0]);
		Width = Convert_Dimension (Width, Device_Ptr, Arg_Ptr->Mag);
		Depth = Convert_Dimension (Depth, Device_Ptr, Arg_Ptr->Mag);
		Status = ((*Protocol_Desc[Index].Executor) (Device_Ptr,
							    Scratch_String,
							    Arg_Ptr->H_Pos,
							    Arg_Ptr->V_Pos,
							    (Arg_Ptr->Reflect_Image == 0) ? Width : -Width,
							    Depth,
							    Orientation,
							    Picture_Number,
							    Option_List,
							    Option_Count));
		(*Device_Ptr->Restore_Graphics) (&Protocol_Desc[0]);
		return (Status);
	}
	Message (DVIOUT_UNKPROTOCOL, Protocol_Value, 1);
	return (0);
}

int Define_XY (Arg_Ptr, CSI_Ptr)
struct Arg_List *Arg_Ptr;
struct CSI_Status_Block *CSI_Ptr;
{
	auto   struct XY_Coord *XY_Ptr;
	extern char *Mem_Alloc();

	XY_Ptr = (struct XY_Coord *) Mem_Alloc (sizeof (struct XY_Coord));
	XY_Ptr->Prev = XY_Top;
	XY_Ptr->X_Coord = Arg_Ptr->H_Pos;
	XY_Ptr->Y_Coord = Arg_Ptr->V_Pos;
	XY_Top = XY_Ptr;
	return (1);
}

int Draw_Line (Arg_Ptr, CSI_Ptr)
struct Arg_List *Arg_Ptr;
struct CSI_Status_Block *CSI_Ptr;
{
	auto   int Status, Argn;
	auto   long X1, Y1, X2, Y2;
	static char XY_Value1[40], XY_Value2[40];
	extern int Get_Parameter(), Convert_XY();
	msgcode DVIOUT_MISSPECARG;
/*
 *	Obtain command values:
 */
	if ((Status = Get_Parameter (Argn=1, XY_Value1, sizeof (XY_Value1), 0, 0, CSI_Ptr)) <= 0 ||
	    (Status = Get_Parameter (Argn=2, XY_Value2, sizeof (XY_Value2), 0, 0, CSI_Ptr)) <= 0) {
		if (Status < 0)
			Message (DVIOUT_MISSPECARG, Argn, 0, "LINE", 1);
		return (0);
	}
	if (Convert_XY (XY_Value1, Arg_Ptr, &X1, &Y1) == 0 || Convert_XY (XY_Value2, Arg_Ptr, &X2, &Y2) == 0)
		return (0);
	(*Arg_Ptr->Device->Typeset_Line) (X1, Y1, X2, Y2);
	return (1);
}

int Draw_Point (Arg_Ptr, CSI_Ptr)
struct Arg_List *Arg_Ptr;
struct CSI_Status_Block *CSI_Ptr;
{
	auto   int Status, Argn;
	auto   long X, Y, Diameter;
	static char XY_Value[40], Diameter_Value[20];
	extern long Parse_Dimension(), Convert_Dimension();
	extern int Get_Parameter(), Convert_XY();
	msgcode DVIOUT_MISSPECARG, DVIOUT_NEGZEROSIZE;
/*
 *	Obtain command values:
 */
	if ((Status = Get_Parameter (Argn=1, XY_Value, sizeof (XY_Value), 0, 0, CSI_Ptr)) <= 0 ||
	    (Status = Get_Parameter (Argn=2, Diameter_Value, sizeof (Diameter_Value), 0, 0, CSI_Ptr)) <= 0) {
		if (Status < 0)
			Message (DVIOUT_MISSPECARG, Argn, 0, "POINT", 1);
		return (0);
	}
	if (Convert_XY (XY_Value, Arg_Ptr, &X, &Y) == 0)
		return (0);
	if ((Diameter = Parse_Dimension (Diameter_Value, 1)) <= 1) {
		if (Diameter <= 0)
			Message (DVIOUT_NEGZEROSIZE, Diameter_Value, 1);
		return (0);
	}
	(*Arg_Ptr->Device->Typeset_Point) (X, Y, Convert_Dimension (Diameter, Arg_Ptr->Device, Arg_Ptr->Mag));
	return (1);
}

int Draw_Arc (Arg_Ptr, CSI_Ptr)
struct Arg_List *Arg_Ptr;
struct CSI_Status_Block *CSI_Ptr;
{
	auto   char *Ptr;
	auto   int Status, Argn;
	auto   long X, Y, Radius, Start_Ang, Stop_Ang;
	static struct Ratio Start, Stop;
	static char XY_Value[40], Start_Value[20], Stop_Value[20], Radius_Value[20];
	extern char *Parse_Number();
	extern long Parse_Dimension(), Convert_Dimension();
	extern int Get_Parameter(), Convert_XY();
	msgcode DVIOUT_EXTRACHAR, DVIOUT_MISSPECARG, DVIOUT_NEGZEROSIZE;
/*
 *	Obtain command values:
 */
	if ((Status = Get_Parameter (Argn=1, XY_Value, sizeof (XY_Value), 0, 0, CSI_Ptr)) <= 0 ||
	    (Status = Get_Parameter (Argn=2, Radius_Value, sizeof (Radius_Value), 0, 0, CSI_Ptr)) <= 0 ||
	    (Status = Get_Parameter (Argn=3, Start_Value, sizeof (Start_Value), 0, 0, CSI_Ptr)) <= 0 ||
	    (Status = Get_Parameter (Argn=4, Stop_Value, sizeof (Stop_Value), 0, 0, CSI_Ptr)) <= 0) {
		if (Status < 0)
			Message (DVIOUT_MISSPECARG, Argn, 0, "ARC", 1);
		return (0);
	}
	if (Convert_XY (XY_Value, Arg_Ptr, &X, &Y) == 0)
		return (0);
	if (*(Ptr = Parse_Number (Start_Value, &Start)) != '\0')
		Message (DVIOUT_EXTRACHAR, Ptr, 1);
	if (*(Ptr = Parse_Number (Stop_Value, &Stop)) != '\0')
		Message (DVIOUT_EXTRACHAR, Ptr, 1);
	if ((Radius = Parse_Dimension (Radius_Value, 0)) < 0) {
		Message (DVIOUT_NEGZEROSIZE, Radius_Value, 1);
		return (0);
	}
	Start_Ang = XN_Div_D_R_M (1000, Start.Numerator, Start.Denominator);
	Stop_Ang = XN_Div_D_R_M (1000, Stop.Numerator, Stop.Denominator);
	if (Arg_Ptr->Reflect_Image != 0) {
		Start_Ang = 180000 - Start_Ang;
		Stop_Ang = 180000 - Stop_Ang;
	}
	(*Arg_Ptr->Device->Typeset_Arc) (X, Y, Convert_Dimension (Radius, Arg_Ptr->Device, Arg_Ptr->Mag),
					 Start_Ang, Stop_Ang);
	return (1);
}

int Fill_Poly (Arg_Ptr, CSI_Ptr)
struct Arg_List *Arg_Ptr;
struct CSI_Status_Block *CSI_Ptr;
{
	auto   struct Device_Table *Device_Ptr;
	auto   long *Poly_X, *Poly_Y;
	auto   unsigned char *Pattern, *Magnified_Pattern;
	auto   char *Row_Value;
	auto   long Bit_Size;
	auto   unsigned int Index, N, Row_Size, Hex_Row_Size, H_Repl, W_Repl;
	auto   unsigned int Pat_Width, Pat_Height;
	auto   int Status, Argn;
	static char XY_Value[40], Col_Value[10], Size_Value[20];
	static struct Switch_Table Switch_List[] = {
		csisw ("PERIMETER", 1, PERIMETER, NEGATABLE, 0),
		csiend
	};
	extern char *Mem_Alloc();
	extern long Parse_Dimension(), Convert_Dimension();
	extern unsigned int Parse_Unsigned_Int();
	extern int Get_Parameter(), Get_Parameter_Count(), Convert_XY();
	msgcode DVIOUT_NEGZEROSIZE, DVIOUT_MISSPECARG, DVIOUT_INVALHEX;
/*
 *	Obtain required command values:
 */
	N = Get_Parameter_Count (1, CSI_Ptr);
	if ((Status = Get_Parameter (Argn=0, 0, 0, 0, Switch_List, CSI_Ptr)) == 0 ||
	    (Status = Get_Parameter (Argn=1, XY_Value, sizeof (XY_Value), 0, Switch_List, CSI_Ptr)) <= 0 ||
	    (Status = Get_Parameter (Argn=2, Col_Value, sizeof (Col_Value), 0, Switch_List, CSI_Ptr)) <= 0) {
		if (Status < 0)
			Message (DVIOUT_MISSPECARG, Argn, 0, "FILL", 1);
		return (0);
	}
/*
 *	Obtain list of (x,y) values:
 */
	Poly_X = (long *) Mem_Alloc (N * 2 * sizeof (long *));
	Poly_Y = &Poly_X[N];
	Index = 0;
	do {
		if (Convert_XY (XY_Value, Arg_Ptr, &Poly_X[Index], &Poly_Y[Index]) == 0) {
			Mem_Free (Poly_X);
			return (0);
		}
	} while (++Index < N &&
	         (Status = Get_Parameter (1, XY_Value, sizeof (XY_Value), 0, Switch_List, CSI_Ptr)) > 0);
	if (Status == 0) {	/* String too long */
		Mem_Free (Poly_X);
		return (0);
	}
/*
 *	Obtain fill pattern:
 */
	if ((Pat_Width = Parse_Unsigned_Int (Col_Value)) == 0) {	/* Use solid pattern */
		Pat_Height = 0;
		Pattern = 0;
	} else if ((Status = Get_Parameter (3, Size_Value, sizeof (Size_Value), 0, Switch_List, CSI_Ptr)) <= 0) {
		if (Status < 0)
			Message (DVIOUT_MISSPECARG, 3, 0, "FILL", 1);
		Mem_Free (Poly_X);
		return (0);
	} else if ((Bit_Size = Parse_Dimension (Size_Value, 1)) <= 1) {
		if (Bit_Size <= 0)
			Message (DVIOUT_NEGZEROSIZE, Size_Value, 1);
		Mem_Free (Poly_X);
		return (0);
	} else if ((Pat_Height = Get_Parameter_Count (4, CSI_Ptr)) == 0) {
		Message (DVIOUT_MISSPECARG, 4, 0, "FILL", 1);
		Mem_Free (Poly_X);
		return (0);
	} else {
		Hex_Row_Size = ((Pat_Width + 3) >> 2) + 1;
		Row_Value = Mem_Alloc (Hex_Row_Size);
		Row_Size = (Pat_Width + 7) >> 3;
		Pattern = (unsigned char *) Mem_Alloc (Row_Size * Pat_Height);
		Index = 0;
		while ((Status = Get_Parameter (4, Row_Value, Hex_Row_Size, 0, Switch_List, CSI_Ptr)) > 0) {
			if (Convert_Hex_to_Raster_M (Pat_Width, Row_Value, &Pattern[Index]) == 0) {
				Message (DVIOUT_INVALHEX, Row_Value, 1);
				Status = 0;
				break;
			}
			Index += Row_Size;
		}
		Mem_Free (Row_Value);
		if (Status == 0) {	/* Syntax error or string too big */
			Mem_Free (Poly_X);
			Mem_Free (Pattern);
			return (0);
		}
	}
/*
 *	Now it is necessary to scale the pattern raster to the appropriate
 *	size. This is done in two steps. The first step is to magnify
 *	the raster so that each 'bit' in the pattern raster is expanded
 *	so that it is 'Bit_Size' width and height, in device pixels. The
 *	second step is to replicate the raster rows enough times to make
 *	sure that the resulting raster description is an even multiple of
 *	the device-dependent requirement for the allowable widths that
 *	the fill pattern can be (e.g. 8 for PostScript, 32 for imPRESS).
 *	We must make sure that the resulting raster size does not exceed
 *	65535 in width or height.
 */
	Device_Ptr = Arg_Ptr->Device;
	if (Pat_Width != 0) {
		if (Arg_Ptr->Reflect_Image != 0)
			Reflect_Raster_M (Pat_Width, Pat_Height, Pattern);
		if ((Bit_Size = (Convert_Dimension (Bit_Size, Device_Ptr, Arg_Ptr->Mag) + 1023) >> 10) == 0)
			Bit_Size = 1;
		if (Bit_Size == 1)
			Magnified_Pattern = Pattern;
		else {
			Row_Size = (Pat_Width * Bit_Size + 7) >> 3;
			Magnified_Pattern = (unsigned char *) Mem_Alloc (Row_Size * Pat_Height * Bit_Size);
			Magnify_Raster_M (Pat_Width, Pat_Height, Bit_Size, Pattern, Magnified_Pattern);
			Mem_Free (Pattern);
			Pat_Width *= Bit_Size;
			Pat_Height *= Bit_Size;
		}
		W_Repl = Device_Ptr->Mask_Resolution[0] / Gcd_M (Pat_Width, Device_Ptr->Mask_Resolution[0]);
		H_Repl = Device_Ptr->Mask_Resolution[1] / Gcd_M (Pat_Height, Device_Ptr->Mask_Resolution[1]);
		if (W_Repl == 1 && H_Repl == 1)
			Pattern = Magnified_Pattern;
		else {
			Pattern = (unsigned char *) Mem_Alloc (((Pat_Width * W_Repl + 7) >> 3) * Pat_Height * H_Repl);
			Replicate_Raster_M (Pat_Width, Pat_Height, Magnified_Pattern, W_Repl, H_Repl, Pattern);
			Mem_Free (Magnified_Pattern);
			Pat_Width *= W_Repl;
			Pat_Height *= H_Repl;
		}
	}
/*
 *	Whew! Finally, call the device driver routine to output the filled
 *	polygon:
 */
	(*Device_Ptr->Typeset_Filled) (N, Poly_X, Poly_Y, Pat_Width, Pat_Height, Pattern,
				       ((CSI_Mask_Word & PERIMETER) == 0) ? 0 : 1);
	Mem_Free (Poly_X);
	if (Pattern != 0)
		Mem_Free (Pattern);
	return (1);
}

int Set_LWidth (Arg_Ptr, CSI_Ptr)
struct Arg_List *Arg_Ptr;
struct CSI_Status_Block *CSI_Ptr;
{
	auto   struct Device_Table *Device_Ptr;
	auto   unsigned long Width;
	auto   int Status;
	static char Width_Value[20];
	extern int Get_Parameter();
	msgcode DVIOUT_MISSPECARG, DVIOUT_NEGZEROSIZE;

	if ((Status = Get_Parameter (1, Width_Value, sizeof (Width_Value), 0, 0, CSI_Ptr)) <= 0) {
		if (Status < 0)
			Message (DVIOUT_MISSPECARG, 1, 0, "LWIDTH", 1);
		Status = 0;
	} else if ((Width = Parse_Dimension (Width_Value, 1)) <= 1) {
		if (Width <= 0)
			Message (DVIOUT_NEGZEROSIZE, Width_Value, 1);
		Status = 0;
	} else {
		Device_Ptr = Arg_Ptr->Device;
		(*Device_Ptr->Set_Linewidth) (Convert_Dimension (Width, Device_Ptr, Arg_Ptr->Mag));
		Status = 1;
	}
	return (Status);
}

int Set_LStyle (Arg_Ptr, CSI_Ptr)
struct Arg_List *Arg_Ptr;
struct CSI_Status_Block *CSI_Ptr;
{
	auto   struct Device_Table *Device_Ptr;
	auto   unsigned int *Spec_Ptr;
	auto   unsigned int Spec_Len, Total_Len, Index;
	auto   int Status;
	static unsigned int Dash_Spec[20];
	static char Spec_Value[12];
	static struct {
		char *Name;
		unsigned int Len;
		unsigned int Spec[20];
	} Predefined_Linestyles[] = {
		{ "",        0 },	/* Null argument => solid */
		{ "SOLID",   0 },
		{ "DOTTED",  2, { 1, 1 } },
		{ "DASHED",  2, { 3, 3 } },
		{ "DOTDASH", 4, { 1, 1, 3, 1 } }
	};
	extern unsigned int Parse_Unsigned_Int();
	extern int Get_Parameter(), strcmp();
	msgcode DVIOUT_MISSPECARG, DVIOUT_TOOMANYDASH, DVIOUT_ZERODASHLEN;
/*
 *	Obtain first in possible list of parameters:
 */
	if ((Status = Get_Parameter (1, Spec_Value, sizeof (Spec_Value), 0, 0, CSI_Ptr)) <= 0) {
		if (Status < 0)
			Message (DVIOUT_MISSPECARG, 1, 0, "LSTYLE", 1);
		return (0);
	}
/*
 *	First, check if the linestyle is one the pre-defined ones:
 */
	Device_Ptr = Arg_Ptr->Device;
	for (Index = 0; Index < arraysize (Predefined_Linestyles); Index++)
	if (strcmp (Spec_Value, Predefined_Linestyles[Index].Name) == 0) {
		(*Device_Ptr->Set_Linestyle) (Predefined_Linestyles[Index].Len,
					      Predefined_Linestyles[Index].Spec, 0);
		return (1);
	}
/*
 *	Parse the dashline specification:
 */
	Spec_Len = 0;
	Total_Len = 0;
	do {
		Dash_Spec[Spec_Len] = Parse_Unsigned_Int (Spec_Value);
		Total_Len += Dash_Spec[Spec_Len];
		Spec_Len++;
	} while (Spec_Len < 20 &&
		 (Status = Get_Parameter (1, Spec_Value, sizeof (Spec_Value), 0, 0, CSI_Ptr)) > 0);
/*
 *	Check final status before setting linestyle at device:
 */
	if (Status > 0) {
		Message (DVIOUT_TOOMANYDASH);
		Status = -1;
	}
	if (Status != 0) {
		if (Total_Len == 0 && Spec_Len > 0) {
			Message (DVIOUT_ZERODASHLEN);
			Spec_Len = 0;
		}
		(*Device_Ptr->Set_Linestyle) (Spec_Len, Dash_Spec, 0);
	}
	return (1);
}

int Convert_XY (XY_Str, Arg_Ptr, X_Ptr, Y_Ptr)
char *XY_Str;
struct Arg_List *Arg_Ptr;
long *X_Ptr, *Y_Ptr;
{
	auto   struct XY_Coord *XY_Ptr;
	auto   char *Ptr_X, *Ptr_Y, *Ptr;
	auto   long X;
	auto   int Error;
	auto   char Term_Char;
	extern long Parse_Dimension(), Convert_Dimension();
	msgcode DVIOUT_EXTRACHAR, DVIOUT_XYSTACKEMPTY, DVIOUT_XYSYNTAX;

	Error = 0;
	if (XY_Str[0] == '*') {
		if ((XY_Ptr = XY_Top) == 0) {
			Message (DVIOUT_XYSTACKEMPTY);
			Error++;
		} else {
			*X_Ptr = XY_Ptr->X_Coord;
			*Y_Ptr = XY_Ptr->Y_Coord;
			XY_Top = XY_Ptr->Prev;
			Mem_Free (XY_Ptr);
			if (XY_Str[1] != '\0')
				Message (DVIOUT_EXTRACHAR, XY_Str, 1);
		}
	} else {
		Ptr_X = XY_Str;
		if (*Ptr_X == '(') {
			Ptr_X++;
			Term_Char = ')';
		} else
			Term_Char = '\0';
		for (Ptr_Y = Ptr_X; *Ptr_Y != ',' && *Ptr_Y != '\0'; Ptr_Y++)
			;
		if (*Ptr_Y != ',') {
			Message (DVIOUT_XYSYNTAX, XY_Str, 1);
			Error++;
		} else {
			*Ptr_Y++ = '\0';
			for (Ptr = Ptr_Y; *Ptr != Term_Char && *Ptr != '\0'; Ptr++)
				;
			if (*Ptr != Term_Char)
				Message (DVIOUT_XYSYNTAX, Ptr_Y, 1);
			else if (*Ptr == ')' && Ptr[1] != '\0')
				Message (DVIOUT_EXTRACHAR, Ptr_Y, 1);
			*Ptr = '\0';
			X = Parse_Dimension (Ptr_X, 0);
			*X_Ptr = Convert_Dimension ((Arg_Ptr->Reflect_Image == 0) ? X : -X,
						    Arg_Ptr->Device, Arg_Ptr->Mag) + Arg_Ptr->H_Pos;
			*Y_Ptr = Convert_Dimension (Parse_Dimension (Ptr_Y, 0),
						    Arg_Ptr->Device, Arg_Ptr->Mag) + Arg_Ptr->V_Pos;
		}
	}
	return ((Error == 0) ? 1 : 0);
}

long Convert_Dimension (Dimen, Device_Ptr, Magnification)
long Dimen;
struct Device_Table *Device_Ptr;
unsigned long Magnification;
{
	return (XN_Div_D_R_M (XN_Div_D_R_M (Dimen, Device_Ptr->Device_Resolution.Numerator * 100,
					    Device_Ptr->Device_Resolution.Denominator * 462528),
			      Magnification, 1000));
}

/*
 *	Routine Set_Color obtains the color specification, does any
 *	necessary conversion, then sets the color at the output device:
 */

int Set_Color (Arg_Ptr, CSI_Ptr)
struct Arg_List *Arg_Ptr;
struct CSI_Status_Block *CSI_Ptr;
{
	auto   unsigned int Index, Red, Green, Blue;
	static struct Ratio Mix[3];
	static char Hue[10], Hue_Value[3][10];
	static struct Switch_Value_Table Hue_Table[] = {
		csisv (Hue_Value[0][0], sizeof (Hue_Value[0]), "0"),
		csisv (Hue_Value[1][0], sizeof (Hue_Value[1]), "0"),
		csisv (Hue_Value[2][0], sizeof (Hue_Value[2]), "0"),
		csiend
	};
	static struct {
		char *Hue_Name;
		unsigned int Min_Chars;
		unsigned int Red_Mix, Green_Mix, Blue_Mix;
	} Hue_Desc[] = {
		{ "RGB",     2, 0, 0, 0 },		/* Must be first */
		{ "HLS",     1, 0, 0, 0 },		/* Must be second */
		{ "RED",     2, 1000,    0,    0 },
		{ "GREEN",   1,    0, 1000,    0 },
		{ "BLUE",    3,    0,    0, 1000 },
		{ "YELLOW",  1, 1000, 1000,    0 },
		{ "CYAN",    1,    0, 1000, 1000 },
		{ "MAGENTA", 1, 1000,    0, 1000 },
		{ "BLACK",   3,    0,    0,    0 },
		{ "WHITE",   1, 1000, 1000, 1000 },
		{ 0 }
	};
	static int RGB_Max[3] = { 100, 100, 100 };
	static int HLS_Max[3] = { 360, 100, 100 };
	extern int Get_Parameter();
	msgcode DVIOUT_INVALHUE;
/*
 *	Obtain the hue from the command string:
 */
	strcpy (Hue, "BLACK");
	if (Get_Parameter (1, Hue, sizeof (Hue), Hue_Table, 0, CSI_Ptr) == 0)
		return (0);
/*
 *	Determine what the hue value is; convert to rgb system, if
 *	necessary:
 */
	for (Index = 0; Hue_Desc[Index].Hue_Name != 0; Index++)
	if (Hue[0] != '\0' && Compare_Keyword_M (Hue, Hue_Desc[Index].Hue_Name,
					         Hue_Desc[Index].Min_Chars, 4, 0) == 1)
		break;
	if (Hue_Desc[Index].Hue_Name == 0) {
		Message (DVIOUT_INVALHUE, Hue, 1);
		Red = Green = Blue = 0;
	} else if (Index == 0) {	/* RGB specification */
		Convert_Mix (Hue_Value, Mix, RGB_Max);
		Red = XN_Div_D_R_M (10, Mix[0].Numerator, Mix[0].Denominator);
		Green = XN_Div_D_R_M (10, Mix[1].Numerator, Mix[1].Denominator);
		Blue = XN_Div_D_R_M (10, Mix[2].Numerator, Mix[2].Denominator);
	} else if (Index == 1) {	/* HLS specification */
		Convert_Mix (Hue_Value, Mix, HLS_Max);
		Convert_HLS_To_RGB (Mix);
		Red = XN_Div_D_R_M (10, Mix[0].Numerator, Mix[0].Denominator);
		Green = XN_Div_D_R_M (10, Mix[1].Numerator, Mix[1].Denominator);
		Blue = XN_Div_D_R_M (10, Mix[2].Numerator, Mix[2].Denominator);
	} else {
		Red = Hue_Desc[Index].Red_Mix;
		Green = Hue_Desc[Index].Green_Mix;
		Blue = Hue_Desc[Index].Blue_Mix;
	}
/*
 *	Finally, set the color at the output device:
 */
	(*Arg_Ptr->Device->Set_Color) (Red, Green, Blue);
	return (1);
}

Convert_Mix (Char_Values, Mix_Values, Max_Values)
char Char_Values[3][10];
struct Ratio Mix_Values[3];
int Max_Values[3];
{
	auto   struct Ratio *Mix_Ptr;
	auto   char *Ptr;
	auto   unsigned int Index;
	extern char *Parse_Number();
	msgcode DVIOUT_EXTRACHAR, DVIOUT_INVALMIX;

	for (Index = 0; Index < 3; Index++) {
		Mix_Ptr = &Mix_Values[Index];
		Ptr = Parse_Number (Char_Values[Index], Mix_Ptr);
		if (*Ptr != '\0')
			Message (DVIOUT_EXTRACHAR, Ptr, 1);
		if (Mix_Ptr->Numerator < 0) {
			Message (DVIOUT_INVALMIX, Char_Values[Index], 1);
			Mix_Ptr->Numerator = 0;
			Mix_Ptr->Denominator = 1;
		} else if (Mix_Ptr->Numerator > Mix_Ptr->Denominator * Max_Values[Index]) {
			Message (DVIOUT_INVALMIX, Char_Values[Index], 1);
			Mix_Ptr->Numerator = Max_Values[Index];
			Mix_Ptr->Denominator = 1;
		}
	}
}

/*
 *	Routine Convert_HLS_To_RGB converts a color mix specified
 *	in terms of hue, lightness and saturation to the equivalent
 *	red, green, blue mix. This algorithm detailed in Appendix B
 *	of ACM SIGGRAPH Computer Graphics, Volume 13, Number 3.
 */

Convert_HLS_To_RGB (Mix)
struct Ratio Mix[3];
{
	auto   double H, L, S, m, M, hue, Value;
	auto   int Index;

	H = (double) Mix[0].Numerator / (double) (long) Mix[0].Denominator;
	L = (double) Mix[1].Numerator / (double) (long) Mix[1].Denominator / 100.0;
	S = (double) Mix[2].Numerator / (double) (long) Mix[2].Denominator / 100.0;
	if (L <= 0.5)
		M = (S + 1.0) * L;
	else
		M  = L + S - L * S;
	m = 2.0 * L - M;
	for (Index = 0; Index < 3; Index++) {
		if ((hue = H - (double) Index * 120.0) < 0.0)
			hue += 360.0;
		if (hue < 60.0)
			Value = (M - m) * hue / 60.0 + m;
		else if (hue < 180.0)
			Value = M;
		else if (hue < 240.0)
			Value = (M - m) * (240.0 - hue) / 60.0 + m;
		else
			Value = m;
		Mix[Index].Numerator = (long) (Value * 1000.0 + 0.5);
		Mix[Index].Denominator = 10;
	}
}

int Save_Attributes (Arg_Ptr, CSI_Ptr)
struct Arg_List *Arg_Ptr;
struct CSI_Status_Block *CSI_Ptr;
{
	(*Arg_Ptr->Device->Save_Graphics) (&CSI_Mask_Word);
	return (1);
}

int Restore_Attributes (Arg_Ptr, CSI_Ptr)
struct Arg_List *Arg_Ptr;
struct CSI_Status_Block *CSI_Ptr;
{
	return ((*Arg_Ptr->Device->Restore_Graphics) (&CSI_Mask_Word));
}

int Collapse_XY_Stack ()
{
	auto   struct XY_Coord *XY_Ptr;
	auto   int Count;

	Count = 0;
	while ((XY_Ptr = XY_Top) != 0) {
		XY_Top = XY_Ptr->Prev;
		Mem_Free (XY_Ptr);
		Count++;
	}
	return (Count);
}
