#!/usr/bin/perl -w

# mailgraph -- a postfix statistics rrdtool frontend
# copyright (c) 2000-2004 David Schweikert <dws@ee.ethz.ch>
# released under the GNU General Public License

use RRDs;
use POSIX qw(uname);

my $VERSION = "1.8";

my $host = (POSIX::uname())[1];
my $scriptname = 'mailgraph.cgi';
my $xpoints = 540;
my $points_per_sample = 3;
my $ypoints = 150;
my $ypoints_err = 80;
my $rrd = 'mailgraph.rrd'; # path to where the RRD database is
my $rrd_virus = 'mailgraph_virus.rrd'; # path to where the Virus RRD database is
my $tmp_dir = '/tmp/mailgraph'; # temporary directory where to store the images

my @graphs = (
	{ title => 'Day Graph',   seconds => 3600*24,        },
	{ title => 'Week Graph',  seconds => 3600*24*7,      },
	{ title => 'Month Graph', seconds => 3600*24*31,     },
	{ title => 'Year Graph',  seconds => 3600*24*365, },
);

my %color = (
	sent     => '000099', # rrggbb in hex
	received => '009900',
	rejected => 'AA0000', 
	bounced  => '000000',
	virus    => 'DDBB00',
	spam     => '999999',
);

sub graph($$$)
{
	my $range = shift;
	my $file = shift;
	my $title = shift;
	my $step = $range*$points_per_sample/$xpoints;

	my ($graphret,$xs,$ys) = RRDs::graph($file,
		'--imgformat', 'PNG',
		'--width', $xpoints,
		'--height', $ypoints,
		'--start', "-$range",
		'--vertical-label', 'msgs/min',
		'--title', $title,
		'--lazy',
		"DEF:sent=$rrd:sent:AVERAGE",
		"DEF:recv=$rrd:recv:AVERAGE",
		"DEF:msent=$rrd:sent:MAX",
		"DEF:mrecv=$rrd:recv:MAX",
		"CDEF:rsent=sent,60,*",
		"CDEF:rrecv=recv,60,*",
		"CDEF:rmsent=msent,60,*",
		"CDEF:rmrecv=mrecv,60,*",
		"CDEF:dsent=sent,UN,0,sent,IF,$step,*",
		"CDEF:ssent=PREV,UN,dsent,PREV,IF,dsent,+",
		"CDEF:drecv=recv,UN,0,recv,IF,$step,*",
		"CDEF:srecv=PREV,UN,drecv,PREV,IF,drecv,+",
		"AREA:rsent#$color{sent}:Sent    ",
		'GPRINT:ssent:MAX:total\: %8.0lf msgs',
		'GPRINT:rsent:AVERAGE:average\: %.2lf msgs/min',
		'GPRINT:rmsent:MAX:max\: %.0lf msgs/min\l',
		"LINE2:rrecv#$color{received}:Received",
		'GPRINT:srecv:MAX:total\: %8.0lf msgs',
		'GPRINT:rrecv:AVERAGE:average\: %.2lf msgs/min',
		'GPRINT:rmrecv:MAX:max\: %.0lf msgs/min\l',
		'HRULE:0#000000',
		'COMMENT:\s',
		'COMMENT:graph created on '.localtime(time).'\r',
	);
	my $ERR=RRDs::error;
	die "ERROR: $ERR\n" if $ERR;
}

sub graph_err($$$)
{
	my $range = shift;
	my $file = shift;
	my $title = shift;
	my $step = $range*$points_per_sample/$xpoints;

	my ($graphret,$xs,$ys) = RRDs::graph($file,
		'--imgformat', 'PNG',
		'--width', $xpoints,
		'--height', $ypoints_err,
		'--start', "-$range",
		'--vertical-label', 'msgs/min',
		'--title', "$title of Errors",
		'--lazy',
		"DEF:bounced=$rrd:bounced:AVERAGE",
		"DEF:rejected=$rrd:rejected:AVERAGE",
		"DEF:virus=$rrd_virus:virus:AVERAGE",
		"DEF:spam=$rrd_virus:spam:AVERAGE",
		"DEF:mbounced=$rrd:bounced:MAX",
		"DEF:mrejected=$rrd:rejected:MAX",
		"DEF:mvirus=$rrd_virus:virus:MAX",
		"DEF:mspam=$rrd_virus:spam:MAX",
		"CDEF:rbounced=bounced,60,*",
		"CDEF:rrejected=rejected,60,*",
		"CDEF:rvirus=virus,60,*",
		"CDEF:rspam=spam,60,*",

		"CDEF:dbounced=bounced,UN,0,bounced,IF,$step,*",
		"CDEF:sbounced=PREV,UN,dbounced,PREV,IF,dbounced,+",
		"CDEF:drejected=rejected,UN,0,rejected,IF,$step,*",
		"CDEF:srejected=PREV,UN,drejected,PREV,IF,drejected,+",
		"CDEF:dvirus=virus,UN,0,virus,IF,$step,*",
		"CDEF:svirus=PREV,UN,dvirus,PREV,IF,dvirus,+",
		"CDEF:dspam=spam,UN,0,spam,IF,$step,*",
		"CDEF:sspam=PREV,UN,dspam,PREV,IF,dspam,+",
		
		"CDEF:rmbounced=mbounced,60,*",
		"CDEF:rmrejected=mrejected,60,*",
		"CDEF:rmvirus=mvirus,60,*",
		"CDEF:rmspam=mspam,60,*",
		"AREA:rrejected#$color{rejected}:Rejected",
		'GPRINT:srejected:MAX:total\: %5.0lf msgs',
		'GPRINT:rrejected:AVERAGE:average\: %.2lf msgs/min',
		'GPRINT:rmrejected:MAX:max\: %.0lf msgs/min\l',
		"STACK:rbounced#$color{bounced}:Bounced ",
		'GPRINT:sbounced:MAX:total\: %5.0lf msgs',
		'GPRINT:rbounced:AVERAGE:average\: %.2lf msgs/min',
		'GPRINT:rmbounced:MAX:max\: %.0lf msgs/min\l',
		"STACK:rvirus#$color{virus}:Viruses ",
		'GPRINT:svirus:MAX:total\: %5.0lf msgs',
		'GPRINT:rvirus:AVERAGE:average\: %.2lf msgs/min',
		'GPRINT:rmvirus:MAX:max\: %.0lf msgs/min\l',
		"STACK:rspam#$color{spam}:Spam    ",
		'GPRINT:sspam:MAX:total\: %5.0lf msgs',
		'GPRINT:rspam:AVERAGE:average\: %.2lf msgs/min',
		'GPRINT:rmspam:MAX:max\: %.0lf msgs/min\l',
		'HRULE:0#000000',
		'COMMENT:\s',
		'COMMENT:graph created on '.localtime(time).'\r',
	);
	my $ERR=RRDs::error;
	die "ERROR: $ERR\n" if $ERR;
}

sub print_html()
{
	print "Content-Type: text/html\n\n";

	print <<HEADER;
<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.0 Transitional//EN" "http://www.w3.org/TR/REC-html40/loose.dtd">
<HTML>
<HEAD>
<TITLE>Mail Statistics for $host</TITLE>
</HEAD>
<BODY BGCOLOR="#FFFFFF">
HEADER

	print "<H1>Mail Statistics for $host</H1>\n";
	for my $n (0..$#graphs) {
		print "<H2>$graphs[$n]{title}</H2>\n";
		print "<P><IMG BORDER=\"0\" SRC=\"$scriptname/mailgraph_${n}.png\" ALT=\"mailgraph\">\n";
		print "<P><IMG BORDER=\"0\" SRC=\"$scriptname/mailgraph_${n}_err.png\" ALT=\"mailgraph\">\n";
	}

	print <<FOOTER;
<table border="0" width="400"><tr><td align="left">
<A href="http://people.ee.ethz.ch/~dws/software/mailgraph">Mailgraph</A> $VERSION
by <A href="http://people.ee.ethz.ch/~dws/">David Schweikert</A></td>
<td ALIGN="right">
<a HREF="http://people.ee.ethz.ch/~oetiker/webtools/rrdtool/"><img border="0" src="http://people.ethz.ch/~oetiker/webtools/rrdtool/.pics/rrdtool.gif" alt="" width="120" height="34"></a>
</td></tr></table>
</BODY></HTML>
FOOTER
}

sub send_image($)
{
	my $file = shift;
	-r $file or do {
		print "Content-type: text/plain\n\nERROR: can't find $file\n";
		exit 1;
	};

	print "Content-type: image/png\n";
	print "Content-length: ".((stat($file))[7])."\n";
	print "\n";
	open(IMG, $file) or die;
	my $data;
	print $data while read IMG, $data, 1024;
}

sub main()
{
	if($ENV{PATH_INFO}) {
		my $uri = $ENV{REQUEST_URI};
		$uri =~ s/\/[^\/]+$//;
		$uri =~ s/\//,/g;
		$uri =~ s/(\~|\%7E)/tilde,/g;
		mkdir $tmp_dir, 0777 unless -d $tmp_dir;
		mkdir "$tmp_dir/$uri", 0777 unless -d "$tmp_dir/$uri";
		my $file = "$tmp_dir/$uri$ENV{PATH_INFO}";
		if($ENV{PATH_INFO} =~ /^\/mailgraph_(\d+)\.png$/) {
			graph($graphs[$1]{seconds}, $file, $graphs[$1]{title});
		}
		elsif($ENV{PATH_INFO} =~ /^\/mailgraph_(\d+)_err\.png$/) {
			graph_err($graphs[$1]{seconds}, $file, $graphs[$1]{title});
		}
		else {
			print "Content-Type: text/plain\n\nERROR: unknown image $ENV{PATH_INFO}\n";
			exit 1;
		}
		send_image($file);
	}
	else {
		print_html;
	}
}

main;
