#include <stdio.h>
#include <fcntl.h>
#include <malloc.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <string.h>
#include <unistd.h>
#include <linux/soundcard.h>

#include "play.h"

#define USEIPC 0

int header_check(wave_header *head) {
	unsigned p;
	
	p = head->length - head->data_length;

	if ( (p != 36) && (p != 37) ) { 
		err(0,"Wrong length difference.");
		return 0;
	}

	if ( strncmp(head->main_chunk,"RIFF",4) ) {
		err(0,"Not in RIFF format.");
 		return 0;
	}

	if ( strncmp(head->chunk_type,"WAVE",4) ) {
		err(0,"Not a WAVE type.");
		return 0;
	}
		
	if ( strncmp(head->sub_chunk,"fmt ",4) ) {
		err(0,"Not a WAVE format.");
		return 0;
	}

	if ( (head->sample_fq * head->byte_p_spl) != head->byte_p_sec ) 
		err(0,"Incorrect sampling rate.");

	return 1;
}


static stop_flag = 0;

void	recplay_stop(void)
{
    stop_flag = 1;
}

/*********************************************************************
 * Record or Play samples :
 *
 * mode = PLAY / RECORD
 * data = command_line options
 *********************************************************************/
int recplay(char *filename,int mode,header_data data,int ipc, void (*loopfunc)(void)) {
	int count;			/* Total # of bytes */
	int cnt;			/* bytes for this I/O */
	int n;				/* Bytes read/written */
	int omode = (mode == PLAY) ? O_WRONLY : O_RDONLY;	
	int um = 0666;			/* Current umask() value */
	int fd = -1;			/* read/write file descriptor */
	int audio = -1;			/* /dev/dsp audio file descriptor */
	int abuf_size;			/* audio buffer size for I/O */
	char *iobuf;			/* allocated I/O buffer */
	char tbuf[80];			/* Temporary buffer */
	wave_header whdr;		/* header to/from file */

	stop_flag = 0;

	/*************************************************************
	 * Find out what are umask setting is :
	 *************************************************************/

	um = umask(0666);
	umask(um);		/* Restore umask */

	/*************************************************************
	 * Lock the device :
	 *************************************************************/
#if USEIPC
	if ( ipc >= 0 ) {
		if ( LockDSP(ipc,mode==PLAY?0:1) ) {
			ipc = -1;		/* Don't unlock now */
			err(1,mode==PLAY
				? "Locking audio device for play."
				: "Locking audio device for record.");
			goto errxit;
		}
	}
#endif
	
	if ( mode == PLAY ) {
		/*****************************************************
		 * Play a sample :
		 *****************************************************/

		if ( filename == NULL ) {
			fd = 0;
			filename = "stdin";
		} else if ( (fd = open(filename,O_RDONLY,0)) < 0 ) {
			sprintf(tbuf,"Error opening %s",filename);
			err(1,tbuf);
			return 1;
		}

		if ( (n = read(fd,&whdr,sizeof(whdr))) < 0 ) {
			sprintf(tbuf,"Reading %s.",filename);
			err(1,tbuf);
			return 1;

		} else if ( n == 0 )
			return 1;	/* Perhaps its /dev/null */

		else if ( (unsigned) n < sizeof(whdr) ) {
			err(0,"Short header: can't be wav file.");
			goto errxit;
		}
		
		if ( !header_check(&whdr) )
			goto errxit; 
		
		if ( !data.force_stereo )
			data.stereo = (whdr.modus == 2);

		if ( !data.force_speed )
			data.speed = whdr.sample_fq;

		if ( !data.force_sample_size ) 
			data.sample_size = whdr.bit_p_spl;
			
		/*****************************************************
		 * set upper border accordingly
		 *****************************************************/
			
		if ( data.time_limit == 0.0 ) 
			data.upper_border = whdr.data_length;
		else	{
			if (!data.force_speed)
				data.upper_border = (long unsigned ) data.time_limit * whdr.sample_fq;
			else	data.upper_border = (long  unsigned) data.time_limit * data.speed;
		
			if ( data.stereo )
				data.upper_border *= 2;
			if ( data.sample_size != 8 )
				data.upper_border *= 2;
		}
	} else	{				/* mode = RECORD */
		/*****************************************************
		 * Record a sample :
		 *****************************************************/

		if ( filename == NULL ) {
			fd = 1;
			filename = "stdout";
		} else	{
			remove(filename);
			if ( (fd = open(filename,O_WRONLY|O_CREAT,0666&~um)) < 0 ) {
				sprintf(tbuf,"Opening %s for write.",filename);
				err(1,tbuf);
				return 1;
			}
		}

		/*****************************************************
		 * Create header :
		 *****************************************************/

		strncpy(whdr.main_chunk,"RIFF",4);
		strncpy(whdr.chunk_type,"WAVE",4);
		strncpy(whdr.sub_chunk,"fmt ",4);
		strncpy(whdr.data_chunk,"data",4);

		whdr.length_chunk = 16;
		whdr.format = 1;
		whdr.modus = data.stereo ? 2 : 1;		
		whdr.sample_fq = data.speed;

		if ( !data.stereo ) {
			/*********************************************
			 * Mono :
			 *********************************************/
			if ( data.sample_size == 8 )
				whdr.byte_p_spl = 1;
			else 	whdr.byte_p_spl = 2;
		} else	{
			/*********************************************
			 * Sterio :
			 *********************************************/
			if ( data.sample_size == 8 )
				whdr.byte_p_spl = 2;
			else	whdr.byte_p_spl = 4;
		}

		whdr.byte_p_sec = data.speed * whdr.byte_p_spl;
		whdr.data_length = (long unsigned)
		    (whdr.byte_p_sec * data.time_limit);

		whdr.bit_p_spl = data.sample_size;
		whdr.length = whdr.data_length + 36;
		data.upper_border = (long unsigned) data.time_limit * data.speed
			* whdr.byte_p_spl;
		data.lower_border = 0;

		if ( write(fd,&whdr,sizeof(whdr)) != sizeof(whdr) ) {
			err(1,"Writing WAVE header.");
		   	return 1;
		}
	}

	count = data.upper_border - data.lower_border;

	if ( !data.quiet_mode ) {
		if ( mode == PLAY )
			fprintf(stderr,"Playing: %s",filename);
		else	fprintf(stderr,"Recording: %s",filename);

		if ( data.time_limit == 0.0 ) 
			fprintf(stderr," (in full)\n");
		else	fprintf(stderr," (%5f secs)\n",data.time_limit);

		if ( data.stereo ) 
			fprintf(stderr,"Stereo");
		else	fprintf(stderr,"Mono");
		fprintf(stderr,", rate %lu Hz, %lu bits.\n", 
			data.speed, data.sample_size);
	}

	if ( (audio = open(AUDIODEV,omode,0)) < 0 ) {
		err(1,"Error opening audio-device");
		return 1;
	}

	if ( ioctl(audio,SNDCTL_DSP_GETBLKSIZE,&abuf_size) < 0 ) {
		err(1,"Obtaining dsp buffer size.");
		goto errxit;
	}

	if ( abuf_size < 4096 || abuf_size > 65536 ) {
		sprintf(tbuf,"Invalid audio buffer size (%d bytes).",(int)abuf_size);
		err(1,tbuf);
		goto errxit;
	} 

	if ( (iobuf = (char *) malloc(abuf_size)) == NULL ) {
		err(1,"Unable to allocate audio buffer.");
		goto errxit;
	}


#if OBSOLETE
        tmp = data.sample_size;
        if ( ioctl(audio, SNDCTL_DSP_SAMPLESIZE, &data.sample_size) < 0 ) {
        	err(1,"Unable to set audio sample size.");
		goto errxit;
	}

	if ( (unsigned) tmp != data.sample_size ) { /* Is this test needed? */
		err(0,"Unable to set sample size");
		goto errxit;
	}
#endif
	if ( ioctl(audio, SNDCTL_DSP_STEREO, &data.stereo) < 0 ) {
		err(1,"Unable to set sterio/mono.");
		goto errxit;
	}		
      
	if ( ioctl(audio,SNDCTL_DSP_SPEED,&data.speed) < 0 ) {
		err(1,"Unable to set audio sampling rate.");
		goto errxit;
	}

	if ( mode == PLAY ) {
		/*****************************************************
		 * Play :
		 *****************************************************/

		while( count > 0 && ! stop_flag )
		{
		    cnt = count > abuf_size ? abuf_size : count;

			if ( (n = read(fd,iobuf,cnt)) < 0 ) {
				err(1,"Reading audio sample.");
				goto errxit;
			}

			if ( n == 0 ) {
			    //err(0,"Unexpected EOF reading samples.");
				goto errxit;
			}

			if ( write(audio,iobuf,n) != n ) {
				err(1,"Writing audio device.");
				goto errxit;
			}

			if(loopfunc)	// process events
			    loopfunc();
			
			count -= n;
		}
	} else	{
		/*****************************************************
		 * Record :
		 *****************************************************/

		while ( count > 0 && ! stop_flag) {
			cnt =  count > abuf_size ? abuf_size : count;

			if ( (n = read(audio,iobuf,cnt)) < 0 ) {
				err(1,"Reading from audio device.");
				goto errxit;
			}

			if ( write(fd,iobuf,n) != n ) {
				err(1,"Writing to output.");
				goto errxit;
			}

			if(loopfunc)	// process events
			    loopfunc();

			count -= n;
		}
	}

	/*************************************************************
	 * Normal Exit :
	 *************************************************************/
	close(fd);
	close(audio);

#if USEIPC
	if ( ipc >= 0 ) {
		/*****************************************************
		 * Unlock so other processes can use it now :
		 *****************************************************/
		if ( UnlockDSP(ipc,mode==PLAY?0:1) ) {
			err(1,mode==PLAY
				? "Unlocking for audio play."
				: "Unlocking for audio record.");
		}
	}
#endif
	
	return 0;

	/*************************************************************
	 * Error Exit :
	 *************************************************************/

errxit:
	if ( fd > 0 )
	    close(fd);
        close(audio);

#if USEIPC
	if ( ipc >= 0 )
		UnlockDSP(ipc,mode==PLAY?0:1);	/* Best effort basis */
#endif
	return 1;
}
