/*
 *	Diane - a small memo-recorder for X-Windows written with wxWindows
 *	(C) 1996 Karsten Ballder
 *			(karsten@pimajestix.physik.uni-karlsruhe.de)
 *	This source is protected by the GNU General Public License.
 *
 *	$Revision: 1.0 $
 */

#include	<stdio.h>
#include	<string.h>
#include	<stdlib.h>
#include	<signal.h>

#include	<fstream.h>

#define _NO_USES_DEFINES
#include <wx.h>

#include	"diane.xpm"  /* char diane_icon[] */

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <unistd.h>
#include <dirent.h>
#include <string.h>
#include <time.h>


#include	"play.h"


#define	TMPFILE	"tmpfile.wav"
#define DIANEDIR ".diane"
#define OPTSFILE	"diane-options"

static char *samplingRatesStrings[] =
{
    "4000", "8000", "12000", "16000", "22000", "44000", NULL
};

#define NUM_SAMPLINGRATES	6
#define DEFAULT_SAMPLINGRATE	4

static char *bitStrings[] = { "8", "12", "16" };

#define NUM_BITSTRINGS		3
#define DEFAULT_BITS		0

#define DEFAULT_TIMELIMIT       60


#if ! defined (GZIP_COMMAND)
#	define	GZIP_COMMAND	"/usr/bin/gzip"
#endif


class DianePanel : public wxPanel
{
    wxListBox	*memoList;
    wxButton    *stopButton,	
	*recordButton,
	*playButton,
	*infoButton,
	*quitButton,
	*eraseButton;
    wxFrame	*frame;
    wxChoice	*samplingRatesChoice, *bitsChoice;
    wxCheckBox	*stereoCheckBox;
    wxSlider	*timelimitSlider;

    void	loadOptions(const char *filename);
    void	saveOptions(const char *filename);
    
    int		compressFlag;

    void	record(void);
    void	play(void);
    void	erase(void);
    void	stop(void);
    void	showInfo(void);
    const char *check_for_selection();
    void	OnPaint(void)
    {
	wxPanel::OnPaint();
    }
// sound settings:
    struct header_data	soundheader;

    void 	OnCommand(wxWindow &win, wxCommandEvent &event);
    void 	OnDefaultAction(wxItem *item);
public:
    
    DianePanel(wxFrame *frame);
    ~DianePanel()
    {
	saveOptions(OPTSFILE);
    }
};


class DianeFrame : public wxFrame
{
    void ShowMemos(const char *dirname = "/tmp");
public:
    DianePanel	*panel;
    
    DianeFrame(wxFrame *frame, int x, int y, int w, int h);
    Bool	OnClose(void)
    {
	delete this;
	return FALSE;
    }
    
};

class DianeApp : public wxApp
{
public:
    DianeFrame	*dianeFrame;
    wxFrame *OnInit(void);
};



// This statement initialises the whole application
DianeApp dianeApp;


void
DianePanel::OnCommand(wxWindow &win, wxCommandEvent &event)
{
    wxItem *item = (wxItem *) &win;

    if(item == recordButton)
	record();
    else if(item == playButton)
	play();
    else if(item == stopButton)	// stop any playing/recording, set buttons
	stop();
    else if(item == infoButton)
	showInfo();
    else if(item == quitButton)
	delete frame;
    else if(item == eraseButton)
	erase();
}


void	process_events(void)
{
    while(dianeApp.Pending())
	dianeApp.Dispatch();
}

void	err(int syserrflag, char *msg)
{
    dianeApp.dianeFrame->SetStatusText(msg);
}


void
DianePanel::OnDefaultAction(wxItem *item)
{
    if(item == memoList)
	play();
}

void
DianePanel::saveOptions(const char *filename)
{
    ofstream out(filename);

    if(! out)
	return;
    out << samplingRatesChoice->GetSelection() << endl
	<< stereoCheckBox->GetValue() << endl
	<< bitsChoice->GetSelection() << endl
	<< timelimitSlider->GetValue() << endl;
    
}

void
DianePanel::loadOptions(const char *filename)
{
    ifstream in(filename);

    int dummy;

    in >> dummy; if(in) samplingRatesChoice->SetSelection(dummy);
    in >> dummy; if(in) stereoCheckBox->SetValue(dummy);
    in >> dummy; if(in) bitsChoice->SetSelection(dummy);
    in >> dummy; if(in) timelimitSlider->SetValue(dummy);
}



void
DianePanel::record(void)
{
    char	*memoname, buffer[41], buffer2[200], *ptr;
    time_t	mytime;
    struct tm	*my_tm;

    char 	command[1024], *cptr;
    
    stopButton->Enable(TRUE);
    recordButton->Enable(FALSE);
    playButton->Enable(FALSE);


    soundheader.speed	= atol(samplingRatesChoice->GetStringSelection());
    soundheader.stereo	= stereoCheckBox->GetValue();
    soundheader.sample_size = atol(bitsChoice->GetStringSelection());
    soundheader.time_limit  = (float) timelimitSlider->GetValue();
    
    err(0,"Recording memo...");

    recplay(TMPFILE, RECORD, soundheader,
	    0,process_events);

    memoname = wxGetTextFromUser(
	"Please enter a short description for this note:",
	"DIANE: Input text",
	"",
	dianeApp.dianeFrame);
    if(!memoname)
    {
	unlink(TMPFILE);
	return;
    }

    cptr=memoname;
    do
	*cptr = (*cptr == '/') ? '_' : *cptr;
    while(*(++cptr));
    
    strncpy(buffer,memoname,40);
    time(&mytime);
    my_tm = localtime(&mytime);

    sprintf(buffer2,"%s     (%02d-%02d-%02d %02d:%02d)",
	    buffer,
	    my_tm->tm_mday,my_tm->tm_mon+1,my_tm->tm_year,
	    my_tm->tm_hour, my_tm->tm_min);
	
    ptr = buffer2;
    do
    {
	if(*ptr == '\n')
	{
	    *ptr = '\0';
	    break;
	}
    }while(*++ptr);


    err(0,"Compressing memo...");
    process_events();
    
    sprintf(command,GZIP_COMMAND" -c <\"%s\" >\"%s\"", TMPFILE, buffer2);
    system(command);
    //rename(TMPFILE,buffer2);
    unlink(TMPFILE);
    
    memoList->Append(buffer2);
    memoList->SetSelection(memoList->FindString(buffer2));
    stopButton->Enable(FALSE);
    recordButton->Enable(TRUE);
    if(memoList->Number())
	playButton->Enable(TRUE);

}

void
DianePanel::stop(void)
{
    stopButton->Enable(FALSE);
    recplay_stop();
    recordButton->Enable(TRUE);
    if(memoList->Number())
	playButton->Enable(TRUE);
}


void
DianePanel::erase(void)
{
    const char	*memoname;
    
    if((memoname = check_for_selection()) == NULL)
	return;
    unlink(memoname);
    memoList->Delete(memoList->FindString((char *)memoname));
}

void
DianePanel::showInfo(void)
{
    wxMessageBox(
	"DIANE Copyright (C) 1997 by\n"
	"\n"
	"Karsten Ballder\n"
	"Lessingstrasse 2\n"
	"32756 Detmold, Germany\n"
	"(karsten@pimajestix.physik.uni-karlsruhe.de)\n"
	"\n"
	"DIANE has been written using the cross-platform\n"
	"GUI class library wxWindows (Xt port) which is\n"
	"freely available under the GNU Library Public\n"
	"License.\n"
	"\n"
	"GUI:\n"
	"(c) Markus Holzem, Schlossparkstrasse 3,\n"
	"52072 Aachen, Germany\n"
	"Refer to http://www.aiai.ed.ac.uk for more information.\n"
	"\n",
	"DIANE: Info",wxOK,frame);
}

const char *
DianePanel::check_for_selection(void)
{
    char *memoname;

    if(memoList->Number() == 0)
    {
	wxMessageBox(
	    "There is no memo on the list yet.",
	    "DIANE: Errormessage",
	    wxOK|wxCENTRE|wxICON_EXCLAMATION,
	    frame);
	return NULL;
    }

    memoname = memoList->GetStringSelection();
    if(! memoname)
    {
	wxMessageBox(
	    "You must select a memo first.",
	    "DIANE: Errormessage",
	    wxOK|wxCENTRE|wxICON_EXCLAMATION,
	    frame);
	return NULL;
    }
    return memoname;
}

void
DianePanel::play(void)
{
    const char	*memoname;
    char         command[1024];

    if((memoname = check_for_selection()) == NULL)
       return;
    
    stopButton->Enable(TRUE);
    recordButton->Enable(FALSE);
    playButton->Enable(FALSE);

    err(0,"Decompressing memo...");
    process_events();	// because next command might take some time
    sprintf(command,"/usr/bin/gzip -dc <\"%s\" >\"%s\"", memoname, TMPFILE);
    system(command);

    err(0,"Playing memo...");
    process_events();

    recplay(TMPFILE, PLAY, soundheader, 0,process_events);

    unlink(TMPFILE);
    
    stopButton->Enable(FALSE);
    recordButton->Enable(TRUE);
    playButton->Enable(TRUE); // here, it's alway possible
    
}

DianePanel::DianePanel(wxFrame *iframe)
{
    DIR	*dir;
    struct dirent      *myentry;
    
    
    
    Create(iframe);
    frame = iframe;

    compressFlag = 1;
    
    // first, do sound init:

    soundheader.quiet_mode 	= TRUE; 
    soundheader.speed		= 22000;
    soundheader.force_speed	= FALSE;
    soundheader.stereo		= FALSE;
    soundheader.force_stereo	= FALSE;
    soundheader.sample_size	= 220000;
    soundheader.force_sample_size = FALSE;
    soundheader.time_limit	= DEFAULT_TIMELIMIT;	// this is important!
    soundheader.lower_border	= 0;	// these we don't need
    soundheader.upper_border	= 0;
    
    recordButton = new wxButton(this, NULL ,"Record");
    
    (playButton = new wxButton(this,NULL,"Play"))->SetDefault();

    (stopButton = new wxButton(this,NULL,"Stop"))->Enable(FALSE);

    eraseButton = new wxButton(this,NULL,"Erase");    

    infoButton = new wxButton(this,NULL,"Info");
    
    quitButton = new wxButton(this,NULL,"Quit");

    NewLine();


    SetLabelPosition(wxVERTICAL);
    memoList = new wxListBox(this,NULL,"Recorded memos:",wxSINGLE,
			     -1,-1,300,150,0,NULL,
			     wxALWAYS_SB|wxSINGLE|wxHSCROLL,
			     "memoList");

    samplingRatesChoice = new wxChoice(this,NULL,"Sampling rate",
				       320,40,100,-1,NUM_SAMPLINGRATES,
				       samplingRatesStrings);
    samplingRatesChoice->SetSelection(DEFAULT_SAMPLINGRATE);

    SetLabelPosition(wxHORIZONTAL);
    bitsChoice = new wxChoice(this,NULL,"Bits ",
			      320,85,-1,-1,NUM_BITSTRINGS,
			      bitStrings);
    bitsChoice->SetSelection(DEFAULT_BITS);
    
    
    stereoCheckBox = new wxCheckBox(this,NULL,"Stereo",
				    320,110);
    stereoCheckBox->SetValue(FALSE);

    //NewLine();
    SetLabelPosition(wxVERTICAL);
    timelimitSlider = new wxSlider(this,NULL,"Stop after...secs",
				   DEFAULT_TIMELIMIT,
				   1,300,100,320,140);


    loadOptions(OPTSFILE);
    
    Fit();

    if((dir = opendir(".")) != NULL)	// read current dir
    {
	while((myentry = readdir(dir)) != NULL)
	{
	    if(strcmp(myentry->d_name, ".") != 0 &&
	       strcmp(myentry->d_name, "..") != 0 &&
		strcmp(myentry->d_name, OPTSFILE) != 0)
		memoList->Append(myentry->d_name);
	}
	closedir(dir);
    }
}

DianeFrame::DianeFrame(wxFrame *frame, int x, int y, int w, int h)
    :wxFrame(frame,"DIANE  (C) 1997 by K. Ballder",x, y,w,h)
{
    SetIcon(new wxIcon(diane_icon));
    panel = new DianePanel(this);
    CreateStatusLine(1);
    Fit();
    Show(TRUE);
}

wxFrame *DianeApp :: OnInit(void)	/*	main 	*/
{
    chdir(getenv("HOME"));
    if(chdir(DIANEDIR) == -1 && errno == ENOENT)
    {
	if(mkdir(DIANEDIR,0700) == -1 || chdir(DIANEDIR) == -1)
	{
	    // rwx for owner only
	    wxMessageBox(
		"DIANE cannot access/create the directory\n"
		"$HOME/"DIANEDIR"\n"
		"\nPlease make sure it exists and try again.",
		"DIANE: Errormessage",
		wxOK|wxCENTRE|wxICON_EXCLAMATION,
		NULL);
	    return NULL;
	}
    }
	    
    if(argc > 1 && strcmp(argv[1],"--iconic") == 0)
    {
	dianeFrame = new DianeFrame(NULL, 0,0,460,220);
	dianeFrame->Show(True);
	dianeFrame->Iconize(True);
    }
    else
	dianeFrame = new DianeFrame(NULL, -1,-1,460,220);

    return dianeFrame;
}











