/***************************************************************************
                          setigraph.cpp  -  description
                             -------------------
    begin                : Thu Oct 19 2000
    copyright            : (C) 2000 by Gordon Machel
    email                : gmachel@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
#include <klocale.h>
#include <kfiledialog.h>
#include <kmessagebox.h>
#include <kpopupmenu.h>

#include <qpopupmenu.h>

#include "setigraph.h"

SetiGraph::SetiGraph(SetiLoc* loc, bool shwPop, bool allowModes, QWidget *parent, const char *name )
          : QWidget(parent,name)
{
// Make a backup of the SetiLoc class and the boolean switches
location     = (SetiLoc*)loc;
showPopup    = shwPop;
allowModeSel = allowModes;

if(showPopup)
  {
  gp_options = new KPopupMenu("Seti Graph");
  gp_options->setCheckable(true);
  if(allowModeSel)
    {
    gp_options->insertItem(i18n("Bar Chart"), BarChart);
    gp_options->insertItem(i18n("Scatter Chart"), ScatterChart);
    marker_lines = new QPopupMenu();
    marker_lines->setCheckable(true);
    marker_lines->insertItem(i18n("Marker"), 0);
    marker_lines->insertItem(i18n("Connecting Lines"), 1);
    gp_options->insertItem(i18n("Style"), marker_lines, Style);
    gp_options->insertSeparator();
    connect(marker_lines, SIGNAL(activated(int)), SLOT(handleMarkerLines(int)));
    }
  gp_options->insertItem(i18n("Show Info"), ShowInfo);
  info_pos = new QPopupMenu();
  info_pos->setCheckable(true);
  info_pos->insertItem(i18n("Top-Left"), 0);
  info_pos->insertItem(i18n("Top-Right"), 1);
  gp_options->insertItem(i18n("Text Position"), info_pos, TextPosition);
  gp_options->insertSeparator();
  gp_options->insertItem(i18n("Save Graph..."), SaveGraph);
  connect(gp_options, SIGNAL(activated(int)), SLOT(handlePopupCommand(int)));
  connect(gp_options, SIGNAL(aboutToShow()), SLOT(checkPopupStatus()));
  connect(info_pos, SIGNAL(activated(int)), SLOT(handleInfoPos(int)));
  }

// default settings
plot_mode = Bars;
show_info = true;
marker    = true;
lines     = true;
text_pos  = TopLeft;
yscale    = 1.05;
infoText  = "";

}

SetiGraph::~SetiGraph()
{
}

/////////////////////////////////////////////////////////////////////////////
void SetiGraph::paintEvent(QPaintEvent* e)
{
QPainter p;

p.begin(this);
p.translate(0, rect().bottom());
drawCurve(data, plot_mode, red, &p);
if(show_info) printInfo(infoText, &p);
p.end();
}

/////////////////////////////////////////////////////////////////////////////
void SetiGraph::drawCurve(const QArray<double>& d, int mode, const QColor& c, QPainter* p)
{
QBrush bru;
bru.setStyle(SolidPattern);
bru.setColor(c);

QPen pen;
pen.setColor(c);

p->setPen(pen);
p->setBrush(bru);

int dsize = d.size();
switch(mode)
	{
	case Bars: // bar chart
		{
		int x = 0;
		int y = -(int)((d[0]*height())/(yscale*dataMax));
		p->moveTo(x, y);
		x = width()/dsize;
		p->lineTo(x, y);
		for(int i=1;i<dsize;i++)
			{
			y = -(int)((d[i]*height())/(yscale*dataMax));
			p->lineTo(x, y);
			x = (i+1)*width()/dsize;
			p->lineTo(x, y);
			}
		break;
		}
	case Lines: // scatter chart
	case Markers:
	case MarkersAndLines:
    {
		int x = (int)((0.5*width())/dsize);
		int y = -(int)((d[0]*height())/(yscale*dataMax));
		p->moveTo(x, y);
		if(mode == Markers || mode == MarkersAndLines) p->drawEllipse(x-3, y-3, 6, 6);
		for(int i=1;i<dsize;i++)
			{
			x = (int)(((i+0.5)*width())/dsize);
			y = -(int)((d[i]*height())/(yscale*dataMax));
			if(mode == Lines || mode == MarkersAndLines)
				p->lineTo(x, y);
			else
				p->moveTo(x, y);
			if(mode == Markers || mode == MarkersAndLines) p->drawEllipse(x-3, y-3, 6, 6);
			}
		break;
		}
	}
}

/////////////////////////////////////////////////////////////////////////////
double SetiGraph::maxValue(const QArray<double>& array) const
{
double maxval(0.0);

for(unsigned int i=0;i<array.size();i++) if(array[i] > maxval) maxval = array[i];

return maxval;
}

/////////////////////////////////////////////////////////////////////////////
void SetiGraph::mousePressEvent(QMouseEvent* e)
{
if(e->button() == RightButton)
	{
	gp_options->popup(QCursor::pos());
	}
}

/////////////////////////////////////////////////////////////////////////////
void SetiGraph::handlePopupCommand(int id)
{
switch(id)
	{
	case BarChart:
		plot_mode = Bars;
		break;
	case ScatterChart:
    if(marker && !lines) plot_mode = Markers;
    if(!marker && lines) plot_mode = Lines;
    if(marker && lines)  plot_mode = MarkersAndLines;
		break;
	case ShowInfo:
		show_info = !show_info;
		break;
	case SaveGraph: // save the Gaussian widget to disk
		{
		QPixmap gg = QPixmap::grabWindow(this->winId());
		QString fn = KFileDialog::getSaveFileName(location->directory(), "*.png *.PNG");
		if(!fn.isEmpty())
			{
			if(strcmp((const char*)fn.right(4),".png") && strcmp((const char*)fn.right(4),".PNG"))
				fn.append(".png");
			if(QFile::exists((const char*)fn))
				{
				int ret(0); // 1 is equivalent to cancel
				ret = KMessageBox::warningYesNo(this, i18n("File already exists. Overwrite?"));
				if(ret == KMessageBox::No) break;
				}
			if(gg.save((const char*)fn, "PNG") == false)
				{
				KMessageBox::error(this, i18n("Could not save the image. Probably\n"
                                      "you don't have write permission."));
				}
			}
		break;
		}
	}
repaint();
}

/////////////////////////////////////////////////////////////////////////////
void SetiGraph::checkPopupStatus()
{
gp_options->setItemChecked(BarChart, false);
gp_options->setItemChecked(ScatterChart, false);
gp_options->setItemChecked(ShowInfo, false);
if(plot_mode == Bars)
  gp_options->setItemChecked(BarChart, true);
else
  gp_options->setItemChecked(ScatterChart, true);

if(plot_mode != Bars)
	{
	gp_options->setItemEnabled(Style, true);
	marker_lines->setItemChecked(0, marker);
	marker_lines->setItemChecked(1, lines);
	}
else
	gp_options->setItemEnabled(Style, false);
gp_options->setItemChecked(ShowInfo, show_info);
gp_options->setItemEnabled(TextPosition, show_info);
if(show_info == true)
	{
	info_pos->setItemChecked(0, false);
	info_pos->setItemChecked(1, false);
	info_pos->setItemChecked(text_pos, true);
	}
}

/////////////////////////////////////////////////////////////////////////////
void SetiGraph::handleMarkerLines(int id)
{
switch(id)
	{
	case 0:
		{
		if(lines == true) marker = !marker;
		break;
		}
	case 1:
		{
		if(marker == true) lines = !lines;
		break;
		}
	}
if(marker && !lines) plot_mode = Markers;
if(!marker && lines) plot_mode = Lines;
if(marker && lines) plot_mode = MarkersAndLines;

repaint();
}

/////////////////////////////////////////////////////////////////////////////
void SetiGraph::handleInfoPos(int id)
{
text_pos = id;
repaint();
}

/////////////////////////////////////////////////////////////////////////////
void SetiGraph::setTextPosition(int pos)
{
if(pos > TopRight) pos = TopRight;
handleInfoPos(pos);
}

/////////////////////////////////////////////////////////////////////////////
void SetiGraph::printInfo(const QString& info, QPainter* p)
{
p->setFont(QFont("system", 8));

QPen pen;

QString wu;
wu = "WU: " + location->wuName() + "\n";

pen.setColor(red);
p->setPen(pen);
switch(text_pos)
	{
	case 0: // top-left
		{
		p->drawText(0, -height(), width(), 16, AlignLeft|DontClip, wu);
		pen.setColor(white);
		p->setPen(pen);
		p->drawText(0, -height()+16, width(), height()-16, AlignLeft|DontClip, info);
		break;
		}
	case 1: // top-right
		{
		p->drawText(0, -height(), width(), 16, AlignRight|DontClip,wu);
		pen.setColor(white);
		p->setPen(pen);
		p->drawText(0, -height()+16, width(), height()-16, AlignRight|DontClip, info);
		break;
		}
	}		
}

/////////////////////////////////////////////////////////////////////////////
void SetiGraph::setInfoText(const QString& info)
{
if(!info.isNull()) infoText = info;
}

/////////////////////////////////////////////////////////////////////////////
void SetiGraph::setCurve(const double* dat, int len)
{
data.duplicate((const double*)dat, len);
dataMax = maxValue(data);
}

/////////////////////////////////////////////////////////////////////////////
void SetiGraph::setCurve(const QArray<double>& dat)
{
data.duplicate(dat);
dataMax = maxValue(data);
}

/////////////////////////////////////////////////////////////////////////////
void SetiGraph::setPlotMode(int mode)
{
if(mode > MarkersAndLines) mode = MarkersAndLines;
plot_mode = mode;
}

/////////////////////////////////////////////////////////////////////////////
void SetiGraph::setYScale(double scale)
{
yscale = scale;
}

/////////////////////////////////////////////////////////////////////////////
double SetiGraph::maximumValue()
{
return(data.size()>0 ? dataMax : 0.0);
}

/////////////////////////////////////////////////////////////////////////////
int SetiGraph::arraySize()
{
return(data.size());
}

/////////////////////////////////////////////////////////////////////////////
double SetiGraph::yScale()
{
return(yscale);
}

#include "setigraph.moc"
