/************************************************************************
 * kinst - KDE Application Installer - install.cpp                      *
 * Copyright (C) 2000  Steffen Sobiech                                  *
 *                                                                      *
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation; either version 2 of the License, or    *
 * (at your option) any later version.                                  *
 *                                                                      *
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program; if not, write to the Free Software          *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
 ************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include <fcntl.h>
#include <libgen.h>
#include <qwidget.h>
#include <qmessagebox.h>
#include <qdir.h>

#include "install.h"
#include "stringfunc.h"
#include "gui.h"

#ifndef EOL
#define EOL 10
#endif

tKIFInstall::tKIFInstall(char* SourceFileDir, QWidget* parent)
{
  char Temp_Filename[FILENAME_MAX];

  InstallCommand_LogFile = NULL;

  tmpnam(Temp_Filename);
#ifdef _DEBUG_
  printf("KIFInstall:  Temp_Filename : %s\n", Temp_Filename);
#endif

  strncpy(Tmp_LogFileName, Temp_Filename, FILENAME_MAX);

  _SourceFileDir = SourceFileDir;

  guiparent = parent;

  YesNoDlg = new tYesNoDlg(guiparent, _i18n("kinst - Upgrade"), "upgradeyn");
  YesNoDlg->hide();

  KISS = new tKISS(NULL);
}

tKIFInstall::~tKIFInstall()
{
  delete InstallOptions;

  if(!(InstallCommand_LogFile == NULL)) fclose(InstallCommand_LogFile);
  InstallCommand_LogFile = NULL;
}

int tKIFInstall::Read_Line(char* Readed_Line, int length)
{
  char Readed_Char;
  int i = 0;

  Readed_Char = 0;

  while((!feof(InstallCommand_LogFile)) && (!(Readed_Char == EOL)) &&
	(i < length))
    {
      if(fread(&Readed_Char,sizeof(char),1,InstallCommand_LogFile) < 1)
	return(0);
      if(!(Readed_Char == EOL))
	Readed_Line[i] = Readed_Char; else Readed_Line[i] = 0;
      i++;
    }

  return(1);
}

int tKIFInstall::Open_LogFile()
{
  if(InstallCommand_LogFile == NULL)
    InstallCommand_LogFile = fopen(Tmp_LogFileName, "r");
  if(InstallCommand_LogFile == NULL) return(0); else return(1);
}

void tKIFInstall::Close_LogFile()
{
  if(!(InstallCommand_LogFile == NULL)) fclose(InstallCommand_LogFile);
  InstallCommand_LogFile = NULL;
}

int tKIFInstall::CheckFileExist(char* filename)
{
  FILE* check_dummy;

  if(strlen(filename) == 0) return(0);

#ifdef _DEBUG_
  printf("Check file exist : %s\n", filename);
#endif

  check_dummy = fopen(filename, "r");

  if(errno == ENOENT)
    {
      if(check_dummy != NULL) fclose(check_dummy);
      return(0);
    } 
  else 
    {
      if(check_dummy != NULL) fclose(check_dummy);
      return(1);
    }
}

int tKIFInstall::CheckRPMExist(char* RPM_Name)
{
  if(strlen(RPM_Name) == 0) return(0);

#ifdef _DEBUG_
  printf("Check RPM exist : %s\n", RPM_Name);
#endif

  char InvokeCommand[1024];
  char* Line;
  int err;
  unsigned int i;

  strncpy(InvokeCommand, "rpm", 1024);
  _strncat(InvokeCommand, " -q \"", 1024);
  _strncat(InvokeCommand, RPM_Name, 1024);
  _strncat(InvokeCommand, "\" &> ", 1024);
  _strncat(InvokeCommand, Tmp_LogFileName, 1024);

  err = system(InvokeCommand);

  if(err == 127)
    {
      fprintf(stderr, _i18n("Error invoking shell.\n"));
      QMessageBox::warning(guiparent, "kinst",
			   _i18n("Error invoking shell.\n"));
      return(0);
    }
  if(err == -1)
    {
      fprintf(stderr, _i18n("General failure while utilizing rpm.\n"));
      QMessageBox::warning(guiparent, "kinst",
			   _i18n("General failure while utilizing rpm.\n"));
      return(0);
    }

  if(!Open_LogFile()) return(0);

  // Check the log
  Line = InvokeCommand;
  Read_Line(Line, 1024);
  for(i = 0 ; i < strlen(RPM_Name) ; i++)
    if(Line[i] != RPM_Name[i]) return(0);

  Close_LogFile();

  return(1);
}

int tKIFInstall::CreateVersionTag(char* filename, tKIFOptions opt, int extnr)
{
  char tagfilename[FILENAME_MAX];
  FILE* tagfile;

  if(extnr < 0) sprintf(tagfilename, "%s.ver.%i.%i.%i", filename,
			opt.VersionMaj, opt.VersionMin,
			opt.VersionRel);
  else sprintf(tagfilename, "%s.ver.%i.%i.%i", filename,
	       opt.VersionMaj2[extnr],
	       opt.VersionMin2[extnr],
	       opt.VersionRel2[extnr]);

  tagfile = fopen(tagfilename, "w");
  if(tagfile == NULL) return(0);
  fclose(tagfile);

  return(1);
}

int tKIFInstall::CompareVersion(char* filename, tKIFOptions opt, int extnr)
{
  char *temp;
  char filename1[FILENAME_MAX];
  char fileversion1[65535];
  char filter[FILENAME_MAX];
  int ver_maj, ver_min1, ver_min2;
  int need_maj, need_min1, need_min2;
  char s_ver_maj[100];
  char s_ver_min1[100];
  char s_ver_min2[100];
  long int i, w, start;
  QDir* dir;
  QFileInfo* VerSearchFi;
  int found;

  strncpy(filename1, filename, FILENAME_MAX);

  found = 0;

  if(extnr < 0) strncpy(filter, opt.CheckPriorFile, FILENAME_MAX);
  else strncpy(filter, opt.CheckPriorFile2[extnr], FILENAME_MAX);
  _strncat(filter, "*.ver.*", FILENAME_MAX);

  dir = new QDir(dirname(filename1), filter);

  QFileInfoList* VerSearchList;
  VerSearchList = (QFileInfoList*) dir->entryInfoList();
  QFileInfoListIterator VerSearchIt(*VerSearchList);

  while( (VerSearchFi=VerSearchIt.current()) && !found)
    {
      if((qstrcmp(VerSearchFi->fileName().data(), "."))
	 && (qstrcmp(VerSearchFi->fileName().data(), ".."))
	 && (VerSearchFi->isFile()))
	{
	  qstrncpy(filename1, VerSearchFi->fileName().data(), FILENAME_MAX);
	  temp = strstr(filename1, ".ver.");
	  if(temp != NULL) found = 1; else found = 0;
	}
      ++VerSearchIt;
    }

  if(!found) return(0);

  ver_maj = 0;
  ver_min1 = 0;
  ver_min2 = 0;

  if(extnr < 0) 
    {
      need_maj = opt.VersionMaj;
      need_min1 = opt.VersionMin;
      need_min2 = opt.VersionRel;
    }
  else
    {
      need_maj = opt.VersionMaj2[extnr];
      need_min1 = opt.VersionMin2[extnr];
      need_min2 = opt.VersionRel2[extnr];
    }

  if(strlen(filename1) > 65534) return(0);

  temp = strstr(filename1, ".ver.");
  if(temp != NULL) strcpy(fileversion1, temp + 5);
  else return(0);

  // -- ver
  start = 0;
  w = 0;
  strcpy(s_ver_maj, "");
  for(i = start;i <= (long int) strlen(fileversion1);i++)
    {
      if(fileversion1[i] == 0)
	{
	  s_ver_maj[w] = 0;
	  i = strlen(fileversion1);
	  start = strlen(fileversion1);
	}
      else
	if(fileversion1[i] != '.')
	  {
	    s_ver_maj[w] = fileversion1[i];
	    w++;
	  }
	else
	  {
	    s_ver_maj[w] = 0;
	    start = i + 1;
	    i = strlen(fileversion1);
	  }
    }
  ver_maj = atoi(s_ver_maj);

  w = 0;
  strcpy(s_ver_min1, "");
  for(i = start;i <= (long int) strlen(fileversion1);i++)
    {
      if(fileversion1[i] == 0)
	{
	  s_ver_min1[w] = 0;
	  i = strlen(fileversion1);
	  start = strlen(fileversion1);
	}
      else
	if(fileversion1[i] != '.')
	  {
	    s_ver_min1[w] = fileversion1[i];
	    w++;
	  }
	else
	  {
	    s_ver_min1[w] = 0;
	    start = i + 1;
	    i = strlen(fileversion1);
	  }
    }
  ver_min1 = atoi(s_ver_min1);

  w = 0;
  strcpy(s_ver_min2, "");
  for(i = start;i <= (long int) strlen(fileversion1);i++)
    {
      if(fileversion1[i] == 0)
	{
	  s_ver_min2[w] = 0;
	  i = strlen(fileversion1);
	  start = strlen(fileversion1);
	}
      else
	if(fileversion1[i] != '.')
	  {
	    s_ver_min2[w] = fileversion1[i];
	    w++;
	  }
	else
	  {
	    s_ver_min2[w] = 0;
	    start = i + 1;
	    i = strlen(fileversion1);
	  }
    }
  ver_min2 = atoi(s_ver_min2);

#ifdef _DEBUG_
  printf("install:  Actual Version : %i.%i.%i\n", ver_maj, ver_min1,
	 ver_min2);
  printf("install:  New Version : %i.%i.%i\n", need_maj, need_min1,
	 need_min2);
#endif

  if(ver_maj > need_maj)
    {
#ifdef _DEBUG_
      printf("install:  program on host is newer.\n");
#endif
      return(0);
    }

  if(ver_maj < need_maj)
    {
#ifdef _DEBUG_
      printf("install:  program on host is older.\n");
#endif
      return(1);
    }

  if(ver_min1 > need_min1)
    {
#ifdef _DEBUG_
      printf("install:  program on host is newer.\n");
#endif
      return(0);
    }

  if(ver_min1 < need_min1)
    {
#ifdef _DEBUG_
      printf("install:  program on host is older.\n");
#endif
      return(1);
    }

  if(ver_min2 > need_min2)
    {
#ifdef _DEBUG_
      printf("install:  program on host is newer.\n");
#endif
      return(0);
    }

  if(ver_min2 < need_min2)
    {
#ifdef _DEBUG_
      printf("install:  program on host is older.\n");
#endif
      return(1);
    }

#ifdef _DEBUG_
  printf("install:  program on host matches new version.\n");
#endif

  return(0);
}

int tKIFInstall::Install_TGZ(char* TGZ_filename, const char* install_to,
			     void* parent)
{
  char CompleteTGZPath[FILENAME_MAX];
  char InvokeCommand[65535];
  char orig_cwd[FILENAME_MAX];
  int filecount;
  int actfilecount;
  char filename[FILENAME_MAX];
  FILE* log;
  int c;

  c = 0;

  filecount = 0; actfilecount = 0;
  strcpy(filename, "");

#ifdef _DEBUG_
  printf("_SourceFileDir : %s\n", _SourceFileDir);
#endif

  strncpy(CompleteTGZPath, _SourceFileDir, FILENAME_MAX);
  _strncat(CompleteTGZPath, "/", FILENAME_MAX);
  _strncat(CompleteTGZPath, TGZ_filename, FILENAME_MAX);

#ifdef _DEBUG_
  printf("CompleteTGZPath : %s\n", CompleteTGZPath);
#endif

  strncpy(InvokeCommand, "tar tfvvz \"", 65535);
  _strncat(InvokeCommand, CompleteTGZPath, 65535);
  _strncat(InvokeCommand, "\"", 65535);

#ifdef _DEBUG_
  printf("TAR GZIP count command : %s\n", InvokeCommand);
#endif

  getcwd(orig_cwd, FILENAME_MAX);

  if(!(mkdir(install_to, 0755) == 0))
    {
      fprintf(stderr, _i18n("Warning: Unable to create directory %s\n"),
	      install_to);
    }
  if(!(chdir(install_to) == 0))
    {
      fprintf(stderr, _i18n("Error changing to directory %s\n"), install_to);
      QMessageBox::warning(guiparent, "kinst",
			   _i18n("The specified directory does not exist and could not be created"));
      return(0);
    }

  log = NULL;
  log = popen(InvokeCommand, "r");
  if(log == NULL)
    {
      fprintf(stderr, _i18n("General failure while utilizing tar.\n"));
      QMessageBox::warning(guiparent, "kinst", _i18n("General failure while utilizing tar.\n"));
      return(0);
    }

  while(c != EOF)
    {
      c = fgetc(log);
      if(c == EOL) filecount++;
    }

  pclose(log);

  c = 0;

  strncpy(InvokeCommand, "tar xfvz \"", 65535);
  _strncat(InvokeCommand, CompleteTGZPath, 65535);
  _strncat(InvokeCommand, "\"", 65535);

#ifdef _DEBUG_
  printf("TAR GZIP install command : %s\n", InvokeCommand);
#endif

  log = NULL;
  log = popen(InvokeCommand, "r");
  if(log == NULL)
    {
      fprintf(stderr, _i18n("General failure while utilizing tar.\n"));
      QMessageBox::warning(guiparent, "kinst", _i18n("General failure while utilizing tar.\n"));
      return(0);
    }

  setTotalSteps(filecount, parent);

  while(c != EOF)
    {
      c = fgetc(log);
      while((c != EOL) && (c != EOF))
	{
	  _strappend(filename, (unsigned char) c, FILENAME_MAX);
	  c = fgetc(log);
	}
      if(c != EOF)
	{
	  _strappend(filename, 0, FILENAME_MAX);
	  actfilecount++;
	  setStatus(actfilecount, filename, parent);
	}

      strcpy(filename, "");
    }

  pclose(log);

  chdir(orig_cwd);

  return(1);
}

int tKIFInstall::Install_RPM(char* RPM_filename, const char* install_to,
			     void* parent)
{
  char CompleteRPMPath[FILENAME_MAX];
  char InvokeCommand[65535];
  int err;
  int actcount;
  int c;
  FILE* log;

  actcount = 0;

  c = 0;

#ifdef _DEBUG_
  printf("_SourceFileDir : %s\n", _SourceFileDir);
#endif

  strncpy(CompleteRPMPath, _SourceFileDir, FILENAME_MAX);
  _strncat(CompleteRPMPath, "/", FILENAME_MAX);
  _strncat(CompleteRPMPath, RPM_filename, FILENAME_MAX);

#ifdef _DEBUG_
  printf("CompleteRPMPath : %s\n", CompleteRPMPath);
#endif

  strncpy(InvokeCommand, "rpm", 65535);
  if(strlen(install_to) != 0)
    {
      _strncat(InvokeCommand, " --prefix ", 65535);
      _strncat(InvokeCommand, install_to, 65535);
    }
  _strncat(InvokeCommand, " -i \"", 65535);
  _strncat(InvokeCommand, CompleteRPMPath, 65535);
  _strncat(InvokeCommand, "\"", 65535);

#ifdef _DEBUG_
  printf("RPM install command : %s\n", InvokeCommand);
#endif

  if(strlen(install_to) != 0)
    if(!(mkdir(install_to, 0755) == 0))
      fprintf(stderr, _i18n("Warning: Unable to create directory %s\n"),
	      install_to);

  log = NULL;
  log = popen(InvokeCommand, "r");
  if(log == NULL)
    {
      fprintf(stderr, _i18n("General failure while utilizing tar.\n"));
      QMessageBox::warning(guiparent, "kinst", _i18n("General failure while utilizing rpm.\n"));
      return(0);
    }

  setTotalSteps(100, parent);

  while(c != EOF)
    {
      c = fgetc(log);
      if(c != '#')
	{
	  fprintf(stderr, _i18n("RPM returned with the following errors:\n"));
	  fprintf(stderr, "%c", c);
	  while(c != EOF)
	    {
	      c = fgetc(log);
	      fprintf(stderr, "%c", c);
	    }
	  pclose(log);
	  return(0);
	}
      actcount++;
      setStatus(actcount, "", parent);
    }

  pclose(log);

  return(1);
}

int tKIFInstall::Install(tKIFOptions given_options, const char* destfolder,
			 int extnr, void* parent)
{
  int inst_err;
  char CompleteCheckPath[FILENAME_MAX];
  char SourceFile[KIFOPTIONS_FIELD_LENGTH];
  char CheckPriorFile[KIFOPTIONS_FIELD_LENGTH];
  char CheckPriorRPM[KIFOPTIONS_FIELD_LENGTH];
  char _destfolder[FILENAME_MAX];

  strncpy(_destfolder, destfolder, FILENAME_MAX);

  if(extnr < 0)
    {
      strncpy(SourceFile, given_options.SourceFile, KIFOPTIONS_FIELD_LENGTH);
      strncpy(CheckPriorFile, given_options.CheckPriorFile,
	      KIFOPTIONS_FIELD_LENGTH);
      strncpy(CheckPriorRPM, given_options.CheckPriorRPM,
	      KIFOPTIONS_FIELD_LENGTH);
    }
  else
    {
      strncpy(SourceFile, given_options.SourceFile2[extnr],
	      KIFOPTIONS_FIELD_LENGTH);
      strncpy(CheckPriorFile, given_options.CheckPriorFile2[extnr],
	      KIFOPTIONS_FIELD_LENGTH);
      strncpy(CheckPriorRPM, given_options.CheckPriorRPM2[extnr],
	      KIFOPTIONS_FIELD_LENGTH);
    }

  inst_err = 0;

  // Install a TGZ
  if(strcasecmp(given_options.SourceFileType, "TGZ") == 0)
    {
      if(strlen(SourceFile) != 0)
	{
	  strncpy(CompleteCheckPath, destfolder, FILENAME_MAX);
	  if(CompleteCheckPath[strlen(destfolder) - 1] != '/')
	    _strncat(CompleteCheckPath, "/", FILENAME_MAX);
	  _strncat(CompleteCheckPath, CheckPriorFile, FILENAME_MAX);
	} else strcpy(CompleteCheckPath, "");
      if(!CheckFileExist(CompleteCheckPath))
	{
	  inst_err = Install_TGZ(SourceFile, destfolder, parent);
	  if(inst_err)
	    CreateVersionTag(CompleteCheckPath, given_options, extnr);
	}
      else // check versions now
	if(!CompareVersion(CompleteCheckPath, given_options, extnr))
	  inst_err = 4;
	else // ask for upgrade
	  if(YesNoDlg->getYesNo(_i18n("There is an older version of this application\nalready installed on this host.\nShall I upgrade it?")))
	    {
	      KISS->setOptions(&given_options);
	      KISS->DeleteFromKISS(extnr);
	      inst_err = Install_TGZ(SourceFile, destfolder, parent);
	      if(inst_err)
		CreateVersionTag(CompleteCheckPath, given_options, extnr);
	    }
	  else
	    inst_err = 4;
    }

  // Install a RPM
  if(strcasecmp(given_options.SourceFileType, "RPM") == 0)
    {
      if(destfolder[strlen(destfolder) - 1] == '/')
	_destfolder[strlen(destfolder) - 1] = 0;
      if(!CheckRPMExist(CheckPriorRPM))
	inst_err = Install_RPM(SourceFile, _destfolder, parent);
      else inst_err = 4;
    }

  if(remove(Tmp_LogFileName) != 0) fprintf(stderr,
					   _i18n("Warning: Could not remove temporary file\n"));

  return(inst_err);
}
