# -*- coding: utf-8 -*-

# Copyright (C) 2004-2006 Johan Svedberg <johan@svedberg.com>

# This file is part of ontv.

# ontv is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.

# ontv is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with ontv; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

from gettext import gettext as _

import gtk
import gobject

import GUI
from Reminders import Reminder

TIMEOUT = 100

class SearchDialog:
    def __init__(self, xmltvfile, reminders):
        xmltvfile.connect("loaded", self.__xmltvfile_loaded)
        self.reminders = reminders

        self.__get_widgets()
        self.__make_channels_combobox()
        self.__make_search_treeview()
        self.__connect_widgets()

        self.dialog.set_icon(GUI.icon)
        self.search_label.set_mnemonic_widget(self.search_entry)

        self.__timeout_id = 0

    def __get_widgets(self):
        self.dialog = GUI.xml.get_widget("search_dialog")
        self.add_reminder_button2 = GUI.xml.get_widget("add_reminder_button2")
        self.closebutton = GUI.xml.get_widget("closebutton2")

        self.search_label = GUI.xml.get_widget("search_label")
        self.search_entry = GUI.xml.get_widget("search_entry")
        self.channels_combobox = GUI.xml.get_widget("channels_combobox")

        self.search_treeview = GUI.xml.get_widget("search_treeview")

    def __make_channels_combobox(self):
        crt = gtk.CellRendererText()
        self.channels_combobox.pack_start(crt, True)
        self.channels_combobox.add_attribute(crt, 'text', 0)

    def __make_search_treeview(self):
        search_model = gtk.ListStore(object)
        self.search_treeview.set_model(search_model)

        crt = gtk.CellRendererText()
        crt.props.xpad = 3
        crt.props.ypad = 3
        time_column = gtk.TreeViewColumn("Air time", crt)
        time_column.set_cell_data_func(crt, self.__crt_cell_data_func)
        self.search_treeview.append_column(time_column)

        crtext = gtk.CellRendererText()
        crtext.props.xpad = 3
        crtext.props.ypad = 3
        program_column = gtk.TreeViewColumn("Program", crtext)
        program_column.set_cell_data_func(crtext, self.__crtext_cell_data_func)
        self.search_treeview.append_column(program_column)

    def __crt_cell_data_func(self, column, cell, model, iter):
        program = model.get_value(iter, 0)
        markup = "<b>%s-%s</b>\n<i>%s</i>"
        cell.props.markup = markup % (program.start_time, program.stop_time,
                                      program.date)

    def __crtext_cell_data_func(self, column, cell, model, iter):
        program = model.get_value(iter, 0)
        markup = "<b>%s</b>\n<i>%s</i>"
        cell.props.markup = markup % (program.markup_escaped_title,
                                      program.channel.markup_escaped_name)

    def __connect_widgets(self):
        self.dialog.connect("close", self.__dialog_close)
        self.closebutton.connect("clicked", self.__closebutton_click)

        self.search_entry.connect("changed", self.__search_entry_changed)

        selection = self.search_treeview.get_selection()
        selection.connect("changed", self.__search_treeview_selection_changed)

        self.add_reminder_button2.connect("clicked",
                                         self.__add_reminder_button_clicked)

    def __dialog_close(self, dialog):
        self.dialog.hide()

    def __closebutton_click(self, button):
        self.dialog.hide()

    def __search_entry_changed(self, entry):
        if self.__timeout_id > 0:
            gobject.source_remove(self.__timeout_id)

        self.__timeout_id = gobject.timeout_add(TIMEOUT, self.__search)

    def __search(self):
        search_model = self.search_treeview.get_model()
        search_model.clear()
        program_column = self.search_treeview.get_column(1)
        search = self.search_entry.get_text().strip().lower().split()
        active = self.channels_combobox.get_active()
        channels_model = self.channels_combobox.get_model()
        channel = channels_model[active][0]

        if len(search) > 0:
            matches = self.listings.search_program(search, channel)
            iter = search_model.get_iter_first()
            for row in search_model:
                program = row[0]
                if not program in matches:
                    search_model.remove(iter)
                else:
                    matches.remove(program)
                    search_model.iter_next(iter)

            for program in matches:
                search_model.append([program])
                program_column.queue_resize()

        program_column.queue_resize()
        self.search_treeview.set_cursor(0)
        return False

    def __search_treeview_selection_changed(self, selection):
        (search_model, search_iter) = selection.get_selected()
        if search_iter:
            program = search_model.get_value(search_iter, 0)
            reminder = Reminder(program.title,
                                program.channel.markup_escaped_name)
            if not reminder in self.reminders.reminders and \
               not self.add_reminder_button2.props.sensitive:
                self.add_reminder_button2.set_sensitive(True)
            else:
                self.add_reminder_button2.set_sensitive(False)
        else:
            self.add_reminder_button2.set_sensitive(False)

    def __add_reminder_button_clicked(self, button):
        selection = self.search_treeview.get_selection()
        (search_model, search_iter) = selection.get_selected()
        program = search_model.get_value(search_iter, 0)
        reminder = Reminder(program.title, program.channel.markup_escaped_name)
        if self.reminders.add(reminder):
            self.reminders_model.append([reminder])
            self.reminders.save()
            self.add_reminder_button2.set_sensitive(False)

    def __xmltvfile_loaded(self, xmltvfile, listings):
        if listings:
            self.listings = listings
            self.search_entry.emit("changed")

    def set_all_as_combo_active(self, combobox):
        channels_model = combobox.get_model()
        channels_iter = channels_model.get_iter_first()
        for row in channels_model:
            if row[0] == _("All"):
                combobox.set_active_iter(channels_iter)
                break
            channels_iter = channels_model.iter_next(channels_iter)

    def show(self, uicomponent=None, verb=None):
        self.search_treeview.set_cursor(0)
        self.search_entry.grab_focus()
        self.dialog.grab_focus()
        self.dialog.show()

# vim: set sw=4 et sts=4 tw=79 fo+=l:
