/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */

package org.gnu.gtk;

import org.gnu.glib.Type;
import org.gnu.glib.Handle;

/**
 * The EventBox widget is a container with a seethrough window that can receive
 * events for the container widget - normally one that cannot recvieve its own.
 * Events are sent to the windows, so a widget without windows cannot recieve
 * events.
 */
public class EventBox extends Bin {
    /**
     * Construct a new EventBox object.
     */
    public EventBox() {
        super(gtk_event_box_new());
    }

    /**
     * Construct an eventbox using a handle to a native resource.
     */
    public EventBox(Handle handle) {
        super(handle);
    }

    /**
     * Internal static factory method to be used by Java-Gnome only.
     */
    public static EventBox getEventBox(Handle handle) {
        if (handle == null)
            return null;

        EventBox obj = (EventBox) getGObjectFromHandle(handle);
        if (obj == null)
            obj = new EventBox(handle);

        return obj;
    }

    /**
     * Sets whether the EventBox has a visible window.
     * 
     * @param isVisible
     */
    public void setWindowVisible(boolean isVisible) {
        gtk_event_box_set_visible_window(getHandle(), isVisible);
    }

    /**
     * Returns whether the EventBox has a visible window
     * 
     * @return true if the EventBox has a visible window.
     */
    public boolean getWindowVisible() {
        return gtk_event_box_get_visible_window(getHandle());
    }

    /**
     * Returns whether the EventBox is above or below the windows of its' child.
     * 
     * @param aboveChild
     */
    public void setAboveChild(boolean aboveChild) {
        gtk_event_box_set_above_child(getHandle(), aboveChild);
    }

    /**
     * Returns whether the EventBox is above the window of its' child.
     */
    public boolean getAboveChild() {
        return gtk_event_box_get_above_child(getHandle());
    }

    /**
     * Retrieve the runtime type used by the GLib library.
     */
    public static Type getType() {
        return new Type(gtk_event_box_get_type());
    }

    native static final protected int gtk_event_box_get_type();

    native static final protected Handle gtk_event_box_new();

    native static final protected void gtk_event_box_set_visible_window(
            Handle ebox, boolean visible);

    native static final protected boolean gtk_event_box_get_visible_window(
            Handle ebox);

    native static final protected void gtk_event_box_set_above_child(
            Handle ebox, boolean above);

    native static final protected boolean gtk_event_box_get_above_child(
            Handle ebox);
}
