/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */

package org.gnu.pango;
import org.gnu.glib.GObject;
import org.gnu.glib.Handle;
/**
 * The PangoContext structure stores global information used to control the itemization process. It contains the following fields: 
 */
public class Context extends GObject 
{
	
	/**
	 * Create a new Context passing a handle that was created in the
	 * native layer.
	 * 
	 * @param handle The handle to a native resource.
	 */
	public Context(Handle handle) {
		super(handle);
	}

	/**
	 * Retrieve the default font description for the context.
	 * @return default font description.
	 */
	public FontDescription getFontDescription(){
		return new FontDescription( pango_context_get_font_description(getHandle()) );
	}

	/**
	 * Sets the font description
	 */
	public void setFontDescription(FontDescription desc){
		pango_context_set_font_description(getHandle(), desc.getHandle());
	}

	/**
	 * Retrieves the global language tag for the context.
	 */
	public Language getLanguage(){
		return new Language(pango_context_get_language(getHandle()));
	}

	/**
	 * Sets the language
	 */
	public void setLanguage(Language lang){
		pango_context_set_language(getHandle(), lang.getHandle());
	}

	/**
	 * Retrieves the base direction for the context.
	 */
	public Direction getBaseDir(){
		return Direction.intern( pango_context_get_base_dir(getHandle()) );
	}

	/**
	 * Sets the base direction for the context.
	 */
	public void setBaseDir(Direction dir){
		pango_context_set_base_dir(getHandle(), dir.getValue());
	}

	/**
	 * Loads the font in one of the fontmaps in the context that is the closest
	 * match for desc.
	 * @param desc A FontDescription describing the font to load
	 * @return The font loaded, or <code>null</code> if no font matched.
	 */
	public FontDescription loadFont(FontDescription desc){
		if( pango_context_load_fontNULL(getHandle(), desc.getHandle()))
			return null;
		return new FontDescription( pango_context_load_font(getHandle(), desc.getHandle()));
	}

	/**
	 * Get overall metric information for a font particular font description.
	 * Since the metrics may be substantially different for different scripts, a
	 * language tag can be provided to indicate that the metrics should be
	 * retrieved that correspond to the script(s) used by that language.  <p>The
	 * PangoFontDescription is interpreted in the same way as by
	 * pango_itemize(), and the family name may be a comma separated list of
	 * figures. If characters from multiple of these families would be used to
	 * render the string, then the returned fonts would be a composite of the
	 * metrics for the fonts loaded for the individual families.
	 */
	public FontMetrics getMetrics(FontDescription desc, Language language){
		return new FontMetrics( pango_context_get_metrics (getHandle(), desc.getHandle(), language.getHandle()));
	}

	/**
	 * List all families for a Context.
	 */
	public FontFamily[] listFamilies() {
	    Handle[] hndls = pango_context_list_families(getHandle());
	    if (null == hndls)
	        return null;
	    FontFamily[] fams = new FontFamily[hndls.length];
	    for (int i = 0; i < hndls.length; i++) 
	        fams[i] = new FontFamily(hndls[i]);
	    return fams;
	}
	
	

    native static final protected int pango_context_get_type ();
    native static final protected Handle[] pango_context_list_families (Handle context);
    native static final protected Handle pango_context_load_font (Handle context, Handle desc);
    native static final protected Handle pango_context_get_metrics (Handle context, Handle desc, Handle language);
    native static final protected void pango_context_set_font_description (Handle context, Handle desc);
    native static final protected Handle pango_context_get_font_description (Handle context);
    native static final protected Handle pango_context_get_language (Handle context);
    native static final protected void pango_context_set_language (Handle context, Handle language);
    native static final protected void pango_context_set_base_dir (Handle context, int direction);
    native static final protected int pango_context_get_base_dir (Handle context);
	native static final protected boolean 	pango_context_load_fontNULL(Handle handle, Handle desc);

}

