/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */

package org.gnu.gtk;

import org.gnu.glib.GObject;
import org.gnu.glib.Handle;
import org.gnu.glib.Type;

/**
 * Paned is the base class for widgets with two panes arranged either
 * horizontally (HPaned) or vertically (VPaned).  The division between
 * the two children is set by default from the size request of the 
 * children but can be adjusted by the user.
 * 
 * @see HPaned
 * @see VPaned
 */
public class Paned extends Container {
	
	/**
	 * Construct a new Paned from a handle to a native resource.
	 */
	protected Paned(Handle handle) {
		super(handle);
	}

	/**
	 * Add a child to the top or left pane with default parameters.
	 * 
	 * @param child The child Widget to add.
	 */
	public void add1(Widget child) {
		Paned.gtk_paned_add1(getHandle(), child.getHandle());
	}
	
	/**
	 * Add a child to the bottom or right pane with default parameters.
	 * 
	 * @param child The child Widget to add.
	 */
	public void add2(Widget child) {
		Paned.gtk_paned_add2(getHandle(), child.getHandle());
	}
	
	/**
	 * Adds a child to the top or left pane.
	 * 
	 * @param child The child Widget to add.
	 * @param resize Should this child expand when the widget is resized.
	 * @param shrink can this child be made smaller than its request.
	 */
	public void pack1(Widget child, boolean resize, boolean shrink) {
		Paned.gtk_paned_pack1(getHandle(), child.getHandle(), resize, shrink);
	}
	
	/**
	 * Adds a child to the bottom or right pane.
	 * 
	 * @param child The child Widget to add.
	 * @param resize Should this child expand when the widget is resized.
	 * @param shrink can this child be made smaller than its request.
	 */
	public void pack2(Widget child, boolean resize, boolean shrink) {
		Paned.gtk_paned_pack2(getHandle(), child.getHandle(), resize, shrink);
	}
	
	/**
	 * Set the position of the divider between the two panes.
	 * 
	 * @param position The pixel position of the divider; a negative value
	 * means that the position is unset.
	 */
	public void setPosition(int position) {
		Paned.gtk_paned_set_position(getHandle(), position);
	}
	
	/**
	 * Get the position of the divider between the two panes.
	 * 
	 * @return The position of the divider between the two panes.
	 */
	public int getPosition() {
		return Paned.gtk_paned_get_position(getHandle());
	}
	
	public Widget getChild1() {
	    Handle hndl = gtk_paned_get_child1(getHandle());
		if (null == hndl)
			return null;
		GObject obj = retrieveGObject(hndl);
		if (null != obj)
			return (Widget)obj;
		return new Widget(hndl);
	}
	
	public Widget getChild2() {
	    Handle hndl = gtk_paned_get_child2(getHandle());
		if (null == hndl)
			return null;
		GObject obj = retrieveGObject(hndl);
		if (null != obj)
			return (Widget)obj;
		return new Widget(hndl);
	}
	
	/**
	 * Retrieve the runtime type used by the GLib library.
	 */
	public static Type getType() {
		return new Type(gtk_paned_get_type());
	}


	native static final protected int gtk_paned_get_type();
	native static final protected void gtk_paned_add1(Handle paned, Handle child);
	native static final protected void gtk_paned_add2(Handle paned, Handle child);
	native static final protected void gtk_paned_pack1(Handle paned, Handle child, boolean resize, boolean shrink);
	native static final protected void gtk_paned_pack2(Handle paned, Handle child, boolean resize, boolean shirnk);
	native static final protected int gtk_paned_get_position(Handle paned);
	native static final protected void gtk_paned_set_position(Handle paned, int position);
	native static final protected Handle gtk_paned_get_child1(Handle paned);
	native static final protected Handle gtk_paned_get_child2(Handle paned);

}
