#include <config.h>

#include <sys/stat.h>
#include <unistd.h>
/* #include <asm/page.h> */
#include <string.h>
/* #include <proc/readproc.h> */

#include <glibtop.h>
#include <glibtop/xmalloc.h>
#include <glibtop/union.h>

#include <gnome.h>

#include "memusage.h"
#include "graph.h"
#include "proc.h"
#include "global.h"

#define PROC_CMD_LEN 40

struct _MemProcInfo {
	char          *cmd;
	gint           n;
	unsigned long  value;
};

typedef struct _MemProcInfo MemProcInfo;

enum _FieldType {
	RESIDENT = 0,
	SHARED,
	SIZE,
	VIRTUAL,
	SWAP
};

typedef enum _FieldType FieldType;

static GtkAdjustment *adjustments [MEMUSAGE_FIELDS];

static void memusage_type_set (GtkWidget *, gpointer);

GnomeUIInfo memUsageMenu [] = {
  {GNOME_APP_UI_ITEM, N_("Resident"), NULL, memusage_type_set, (gpointer)RESIDENT, NULL,
   GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
  {GNOME_APP_UI_ITEM, N_("Shared"), NULL, memusage_type_set, (gpointer)SHARED, NULL,
   GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
  {GNOME_APP_UI_ITEM, N_("Size"), NULL, memusage_type_set, (gpointer)SIZE, NULL,
   GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
  {GNOME_APP_UI_ITEM, N_("Virtual"), NULL, memusage_type_set, (gpointer)VIRTUAL,
   NULL, GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
  {GNOME_APP_UI_ITEM, N_("Swap"), NULL, memusage_type_set, (gpointer)SWAP, NULL,
   GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
  GNOMEUIINFO_END
};

static GtkWidget *sw;
static Graph     *graph;
static GtkWidget *f;

/* static proc_t **proc_tab = NULL; */
static unsigned *proc_tab = NULL;
static MemProcInfo *memusage_data = NULL;
static unsigned long value_total;
static unsigned long mem_kernel;
static gint cfg_run;
static gint run_tag = -1;
static FieldType ftype = RESIDENT;
static gchar *graph_head;
static gchar *graph_tail;

static void     memusage_handler (GtkNotebook *notebook, GtkNotebookPage *page, GTopPageSignal s);
static gpointer memusage_data_fn (GraphCmd cmd, gpointer data);
static gint     memusage_update ();
static void     memusage_init ();
static void     memusage_start_stop (gint start);

static void	memusage_properties_init (GTopPropertiesHook *hook, GtkWidget *win);
static void	memusage_properties_apply (GTopPropertiesHook *hook);
static void	memusage_properties_update (GTopPropertiesHook *hook);
static void	memusage_properties_load (GTopPropertiesHook *hook);
static void	memusage_properties_save (GTopPropertiesHook *hook);

GTopPropertiesHook MemUsagePropertiesHook = {
	memusage_properties_init,
	memusage_properties_apply,
	memusage_properties_update,
	memusage_properties_load,
	memusage_properties_save,
	NULL
};

void *
addMemUsageView ()
{
	GtkWidget *gw;

	sw = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
					GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_container_border_width (GTK_CONTAINER (sw), GNOME_PAD_SMALL);
	gtk_widget_set_name (sw, "MemUsageGraph");

	f = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME (f), GTK_SHADOW_NONE);
	gtk_container_border_width (GTK_CONTAINER (f), GNOME_PAD_SMALL << 1);
 
	graph = graph_new (memusage_data_fn);
	gw = graph_widget (graph);
	graph_colors_set (graph, graph_default_colors, GRAPH_DEFAULT_COLORS);

	gtk_container_add (GTK_CONTAINER (f), gw);
	gtk_container_add (GTK_CONTAINER (sw), f);

	gtk_notebook_append_page (GTK_NOTEBOOK (notebook), sw,
				  gtk_label_new (_("Memory usage")));

	gtk_widget_show (gw);
	gtk_widget_show (f);
	gtk_widget_show (sw);

	memusage_init ();

	return memusage_handler;
}

static void
memusage_cfg_save ()
{
	/* printf ("proview_cfg_save\n"); */

	//gnome_config_set_int ("/gtop/procview/run", cfg_run);
	//gnome_config_set_int ("/gtop/procview/summary", !cfg_summary);
	/*gnome_config_set_int ("/gtop/procview/sort_field", cfg_sf);*/
	/*gnome_config_set_int ("/gtop/procview/sort_order", !cfg_order);*/

	gnome_config_set_int ("/gtop/memusage/field_type", ftype);
	gnome_config_set_int ("/gtop/memusage/run", cfg_run);

	gnome_config_sync ();
}

static void
memusage_time_cb (GtkWidget *w, gpointer gp)
{
	cfg_run = !cfg_run;
	memusage_start_stop (cfg_run);
}

static void
memusage_handler (GtkNotebook *notebook, GtkNotebookPage *page, GTopPageSignal s)
{
	/*printf ("procview handler %x %x\n", page->child, vbox);*/

	/* check if it is for me */
	if (page->child == sw) {

		switch (s) {

		case GTOP_PAGE_CFG_SAVE:

			memusage_cfg_save ();
			break;

		case GTOP_PAGE_ENTER:

			/*printf ("procview page entered\n");*/

			gtop_time_cb = memusage_time_cb;
			memusage_update ();
			memusage_start_stop (cfg_run);

			gnome_app_menu_show (GNOME_APP(window),
					     GTOP_MENU_MEMUSAGE, -1);

			break;

		case GTOP_PAGE_LEAVE:

			/*printf ("procview page leaved\n");*/

			gtop_time_cb = NULL;
			memusage_start_stop (0);

			gnome_app_menu_hide (GNOME_APP(window),
					     GTOP_MENU_MEMUSAGE, -1);
			break;

		default:
			break;
		}
	}
}

static void
memusage_type_set (GtkWidget *w, gpointer p)
{
	ftype = (FieldType)p;

	switch (ftype) {
	case RESIDENT:
		graph_head = _("Resident Sizes of Processes");
		graph_tail = _("Sum of Resident Sizes: %ldk");
		break;
	case SHARED:
		graph_head = _("Shared Sizes of Processes");
		graph_tail = _("Sum of Shared Sizes: %ldk");
		break;
	case SIZE:
		graph_head = _("Total Sizes of Processes");
		graph_tail = _("Sum of Total Sizes: %ldk");
		break;
	case VIRTUAL:
		graph_head = _("Virtual Sizes of Processes");
		graph_tail = _("Sum of Virtual Sizes: %ldk");
		break;
	case SWAP:
		graph_head = _("Swapped Sizes of Processes");
		graph_tail = _("Sum of Swapped Sizes: %ldk");
		break;
	}

	memusage_update ();
}

static gpointer
memusage_data_fn (GraphCmd cmd, gpointer data)
{
	MemProcInfo *info = data;
	static gchar buf [256];
	static gchar tail [256];

	switch (cmd) {

	case GRAPH_FIRST:

		return (memusage_data) ? (memusage_data [0].cmd) ? memusage_data : NULL : NULL;

	case GRAPH_NEXT:

		return (info [1].cmd) ? info+1 : NULL;

	case GRAPH_VALUE:

		return (gpointer)(info->value);

	case GRAPH_LABEL:

		if (info->n > 1)
			sprintf (buf, "%30s (%2d): %14ldk", info->cmd, info->n, info->value);
		else
			sprintf (buf, "%30s     : %14ldk", info->cmd, info->value);

		return buf;

	case GRAPH_HEAD:

		return graph_head;

	case GRAPH_TAIL:

		sprintf (tail, graph_tail,
			 value_total >> 10);

		return tail;

	default:

		return NULL;
	}
}

static int
memusage_cmd_cmp (const void *i1,
		  const void *i2)
{
	return strcmp (((MemProcInfo *)i1)->cmd, ((MemProcInfo *)i2)->cmd);
}

static gint
memusage_update ()
{
	MemProcInfo *ti;
	ProcInfo info;
	gint n = 0, i, j, k = 0, l;
	unsigned long value, threshold;
	char *cmd;
	static gchar bts [256];
	glibtop_proclist proclist;

	if (proc_tab)
		glibtop_free (proc_tab);

	switch (gtop_properties.proc_select) {
	case GTOP_PROC_SELECT_ALL:
		proc_tab = glibtop_get_proclist (&proclist, 0, 0);
		break;
	case GTOP_PROC_SELECT_USER:
		proc_tab = glibtop_get_proclist
			(&proclist,
			 GLIBTOP_KERN_PROC_UID,
			 getuid ());
		break;
	case GTOP_PROC_SELECT_TTY:
		proc_tab = glibtop_get_proclist
			(&proclist,
			 GLIBTOP_KERN_PROC_UID | GLIBTOP_EXCLUDE_NOTTY,
			 getuid ());
		break;
	}

	n = proclist.number;

	/* temporary info */

	ti = g_new (MemProcInfo, n);

	value = 0; /* keep gcc happy */
	value_total = 0;

	for (i = 0; i < n; i++) {
		glibtop_proc_state procstate;
		glibtop_proc_mem procmem;

		glibtop_get_proc_state (&procstate, proc_tab [i]);

		ti [i].cmd = glibtop_strdup (procstate.cmd);

		glibtop_get_proc_mem (&procmem, proc_tab [i]);

		switch (ftype) {
		case RESIDENT:
			value = procmem.rss;
			break;
		case SHARED:
			value = procmem.share;
			break;
		case SIZE:
			value = procmem.size;
			break;
		case VIRTUAL:
			value = procmem.vsize;
			break;
		case SWAP:
			value = (procmem.size > procmem.resident) ?
				(procmem.size - procmem.resident) : 0;
			break;
		}
		ti [i].value = value >> 10;
		value_total += value;
	}

	threshold = gtop_properties.memusage_thresholds [ftype];

	/*
	 * sort info by cmd
	 * we need same processes cmd grouped
	 *
	 */

	qsort (ti, n, sizeof (MemProcInfo), memusage_cmd_cmp);

	/*
	 * compute # of processes > threshold to k
	 *
	 */

	for (i=0, k=0; i<n;) {
		
		value = 0;
		cmd = ti [i].cmd;

		do {
			value += ti [i].value;
			i++;

		} while (i<n && !strcmp (cmd, ti [i].cmd));

		if (value >= threshold)
			k++;
	}

	if (memusage_data)
		g_free (memusage_data);

	memusage_data = g_new (MemProcInfo, k+2);

	sprintf (bts, "< %ldk", threshold);
	memusage_data [k].cmd = bts;
	memusage_data [k].value = 0;
	memusage_data [k].n = 0;

	memusage_data [k+1].cmd = NULL;

	/*
	 * second pass
	 *
	 *   store data and fill Below field
	 *
	 */
	
	for (i=0, j=0; i<n;) {

		value = 0;
		l   = 0;
		cmd = ti [i].cmd;

		do {
			value += ti [i].value;

			i++;
			l++;

		} while (i<n && !strcmp (cmd, ti [i].cmd));

		if (value >= threshold) {

			memusage_data [j].cmd = cmd;
			memusage_data [j].n = l;
			memusage_data [j].value = value;
			j++;

		} else {
			memusage_data [k].value += value;
			memusage_data [k].n ++;
		}
	}

	g_free (ti);
	
	/* memusage_data --; */

	graph_minimal_height_set
		(graph, sw->allocation.height - 8*GNOME_PAD_SMALL);
	graph_update (graph);

	return TRUE;
}

static void
memusage_start_stop (gint start)
{
	/* printf ("memusage_start_stop %d\n", start); */

	if (!start) {
		if (run_tag != -1) {
			gtk_timeout_remove (run_tag);
			run_tag = -1;
		}

	} else {

		if (run_tag != -1)
			return;

		memusage_update ();
		run_tag = gtk_timeout_add (3000, memusage_update, NULL);
	}

	gnome_stock_set_icon ((GnomeStock*) (TBC (RUN)->icon),
			      (start) ? GNOME_STOCK_PIXMAP_TIMER :
			      GNOME_STOCK_PIXMAP_TIMER_STOP);

	gtk_widget_queue_draw (GTK_WIDGET (TBC (RUN)->widget));
}

static void
memusage_init ()
{
	ProcInfo info;
	gboolean def;

	proc_read_mem (&info);

	ftype   = gnome_config_get_int_with_default ("/gtop/memusage/field_type=0", &def);
	cfg_run = gnome_config_get_int ("/gtop/memusage/run=1");

	if (def)
		ftype = RESIDENT;
	
	memusage_type_set (NULL, (gpointer)ftype);
}

static void
memusage_properties_update (GTopPropertiesHook *hook)
{
	memusage_update ();
}

static void
adjustment_changed_cb (GtkWidget *widget, GtkWidget *adjustment)
{
	gtop_properties_changed ();
}

static void
memusage_properties_apply (GTopPropertiesHook *hook)
{
	gint i;
	
	for (i = 0; i < MEMUSAGE_FIELDS; i++)
		gtop_temp_properties.memusage_thresholds [i] =
			adjustments [i]->value;
}

static void
radio_procall_cb (GtkWidget *widget, GtkWidget *button)
{
	if (!GTK_TOGGLE_BUTTON (button)->active)
		return;

	gtop_temp_properties.proc_select = GTOP_PROC_SELECT_ALL;
	gtop_properties_changed ();
}

static void
radio_procuser_cb (GtkWidget *widget, GtkWidget *button)
{
	if (!GTK_TOGGLE_BUTTON (button)->active)
		return;

	gtop_temp_properties.proc_select = GTOP_PROC_SELECT_USER;
	gtop_properties_changed ();
}

static void
radio_proctty_cb (GtkWidget *widget, GtkWidget *button)
{
	if (!GTK_TOGGLE_BUTTON (button)->active)
		return;

	gtop_temp_properties.proc_select = GTOP_PROC_SELECT_TTY;
	gtop_properties_changed ();
}

static void
memusage_properties_init (GTopPropertiesHook *hook, GtkWidget *win)
{
	GtkWidget *vb, *frame, *label, *button, *spin, *table;
	GtkObject *adjustment;
	GSList *group;
	gint i;

	vb    = gtk_vbox_new (FALSE, 0);
	gtk_container_border_width (GTK_CONTAINER (vb), GNOME_PAD_SMALL);

	frame = gtk_frame_new (_("Select Processes"));
	table = gtk_table_new (3, 1, TRUE);
	gtk_table_set_col_spacings (GTK_TABLE (table), GNOME_PAD << 2);

	button = gtk_radio_button_new_with_label
		(NULL, _("Show all processes"));
	if (gtop_temp_properties.proc_select == GTOP_PROC_SELECT_ALL)
		gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
	gtk_signal_connect
		(GTK_OBJECT (button), "toggled", radio_procall_cb, button);
	gtk_table_attach_defaults (GTK_TABLE (table), button, 0, 1, 0, 1);

	group = gtk_radio_button_group (GTK_RADIO_BUTTON (button));
	button = gtk_radio_button_new_with_label
		(group, _("Only show user processes"));
	if (gtop_temp_properties.proc_select == GTOP_PROC_SELECT_USER)
		gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
	gtk_signal_connect
		(GTK_OBJECT (button), "toggled", radio_procuser_cb, button);
	gtk_table_attach_defaults (GTK_TABLE (table), button, 0, 1, 1, 2);

	group = gtk_radio_button_group (GTK_RADIO_BUTTON (button));
	button = gtk_radio_button_new_with_label
		(group, _("Only show processes with a controlling tty"));
	if (gtop_temp_properties.proc_select == GTOP_PROC_SELECT_TTY)
		gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
	gtk_signal_connect
		(GTK_OBJECT (button), "toggled", radio_proctty_cb, button);
	gtk_table_attach_defaults (GTK_TABLE (table), button, 0, 1, 2, 3);

	gtk_container_add (GTK_CONTAINER (frame), table);
	gtk_box_pack_start (GTK_BOX (vb), frame, FALSE, TRUE, 0);

	frame = gtk_frame_new ("");
	table = gtk_table_new (5, 2, TRUE);
	gtk_table_set_col_spacings (GTK_TABLE (table), GNOME_PAD << 2);
	
	label = gtk_label_new (_("Minimum Resident Memory Size"));
	gtk_misc_set_alignment (GTK_MISC (label), 1, 0.5);
	gtk_table_attach_defaults (GTK_TABLE (table), label, 0, 1, 0, 1);

	spin = gtk_spin_button_new (NULL, 1.0, 0);
	adjustment = gtk_adjustment_new (0, 0, INT_MAX, 1, 256, 256);
	gtk_spin_button_set_adjustment
		(GTK_SPIN_BUTTON (spin), GTK_ADJUSTMENT (adjustment));
	gtk_spin_button_set_value
		(GTK_SPIN_BUTTON (spin),
		 gtop_temp_properties.memusage_thresholds [RESIDENT]);
	gtk_table_attach_defaults (GTK_TABLE (table), spin, 1, 2, 0, 1);

	gtk_signal_connect
		(GTK_OBJECT (adjustment), "value_changed",
		 adjustment_changed_cb, adjustment);

	adjustments [RESIDENT] = GTK_ADJUSTMENT (adjustment);

	label = gtk_label_new (_("Minimum Shared Memory Size"));
	gtk_misc_set_alignment (GTK_MISC (label), 1, 0.5);
	gtk_table_attach_defaults (GTK_TABLE (table), label, 0, 1, 1, 2);

	spin = gtk_spin_button_new (NULL, 1, 0);
	adjustment = gtk_adjustment_new (0, 0, INT_MAX, 1, 256, 256);
	gtk_spin_button_set_adjustment
		(GTK_SPIN_BUTTON (spin), GTK_ADJUSTMENT (adjustment));
	gtk_spin_button_set_value
		(GTK_SPIN_BUTTON (spin),
		 gtop_temp_properties.memusage_thresholds [SHARED]);
	gtk_table_attach_defaults (GTK_TABLE (table), spin, 1, 2, 1, 2);

	gtk_signal_connect
		(GTK_OBJECT (adjustment), "value_changed",
		 adjustment_changed_cb, adjustment);

	adjustments [SHARED] = GTK_ADJUSTMENT (adjustment);

	label = gtk_label_new (_("Minimum Total Process Size"));
	gtk_misc_set_alignment (GTK_MISC (label), 1, 0.5);
	gtk_table_attach_defaults (GTK_TABLE (table), label, 0, 1, 2, 3);

	spin = gtk_spin_button_new (NULL, 1, 0);
	adjustment = gtk_adjustment_new (0, 0, INT_MAX, 1, 256, 256);
	gtk_spin_button_set_adjustment
		(GTK_SPIN_BUTTON (spin), GTK_ADJUSTMENT (adjustment));
	gtk_spin_button_set_value
		(GTK_SPIN_BUTTON (spin),
		 gtop_temp_properties.memusage_thresholds [SIZE]);
	gtk_table_attach_defaults (GTK_TABLE (table), spin, 1, 2, 2, 3);

	gtk_signal_connect
		(GTK_OBJECT (adjustment), "value_changed",
		 adjustment_changed_cb, adjustment);

	adjustments [SIZE] = GTK_ADJUSTMENT (adjustment);

	label = gtk_label_new (_("Minimum Virtual Memory Size"));
	gtk_misc_set_alignment (GTK_MISC (label), 1, 0.5);
	gtk_table_attach_defaults (GTK_TABLE (table), label, 0, 1, 3, 4);

	spin = gtk_spin_button_new (NULL, 1, 0);
	adjustment = gtk_adjustment_new (0, 0, INT_MAX, 1, 256, 256);
	gtk_spin_button_set_adjustment
		(GTK_SPIN_BUTTON (spin), GTK_ADJUSTMENT (adjustment));
	gtk_spin_button_set_value
		(GTK_SPIN_BUTTON (spin),
		 gtop_temp_properties.memusage_thresholds [VIRTUAL]);
	gtk_table_attach_defaults (GTK_TABLE (table), spin, 1, 2, 3, 4);

	gtk_signal_connect
		(GTK_OBJECT (adjustment), "value_changed",
		 adjustment_changed_cb, adjustment);

	adjustments [VIRTUAL] = GTK_ADJUSTMENT (adjustment);

	label = gtk_label_new (_("Minimum Swapped Memory Size"));
	gtk_misc_set_alignment (GTK_MISC (label), 1, 0.5);
	gtk_table_attach_defaults (GTK_TABLE (table), label, 0, 1, 4, 5);

	spin = gtk_spin_button_new (NULL, 1, 0);
	adjustment = gtk_adjustment_new (0, 0, INT_MAX, 1, 256, 256);
	gtk_spin_button_set_adjustment
		(GTK_SPIN_BUTTON (spin), GTK_ADJUSTMENT (adjustment));
	gtk_spin_button_set_value
		(GTK_SPIN_BUTTON (spin),
		 gtop_temp_properties.memusage_thresholds [SWAP]);
	gtk_table_attach_defaults (GTK_TABLE (table), spin, 1, 2, 4, 5);

	gtk_signal_connect
		(GTK_OBJECT (adjustment), "value_changed",
		 adjustment_changed_cb, adjustment);

	adjustments [SWAP] = GTK_ADJUSTMENT (adjustment);

	gtk_container_add (GTK_CONTAINER (frame), table);
	gtk_box_pack_start (GTK_BOX (vb), frame, FALSE, TRUE, 0);

	gtk_notebook_append_page
	  (GTK_NOTEBOOK (GNOME_PROPERTY_BOX (win)->notebook),
	   vb, gtk_label_new (_("Memory usage")));
}

static void
memusage_properties_load (GTopPropertiesHook *hook)
{
	gtop_properties.proc_select =
		gnome_config_get_int ("gtop/memusage/proc_select=0");

	gtop_properties.memusage_thresholds [RESIDENT] =
		gnome_config_get_int ("gtop/memusage/min_resident=1024");

	gtop_properties.memusage_thresholds [SHARED] =
		gnome_config_get_int ("gtop/memusage/min_shared=1024");

	gtop_properties.memusage_thresholds [SIZE] =
		gnome_config_get_int ("gtop/memusage/min_size=1024");

	gtop_properties.memusage_thresholds [VIRTUAL] =
		gnome_config_get_int ("gtop/memusage/min_virtual=4096");

	gtop_properties.memusage_thresholds [SWAP] =
		gnome_config_get_int ("gtop/memusage/min_swap=1024");
}

static void
memusage_properties_save (GTopPropertiesHook *hook)
{
	gnome_config_set_int
		("gtop/memusage/proc_select",
		 gtop_properties.proc_select);

	gnome_config_set_int
		("gtop/memusage/min_resident",
		 gtop_properties.memusage_thresholds [RESIDENT]);

	gnome_config_set_int
		("gtop/memusage/min_shared",
		 gtop_properties.memusage_thresholds [SHARED]);

	gnome_config_set_int
		("gtop/memusage/min_size",
		 gtop_properties.memusage_thresholds [SIZE]);

	gnome_config_set_int
		("gtop/memusage/min_virtual",
		 gtop_properties.memusage_thresholds [VIRTUAL]);

	gnome_config_set_int
		("gtop/memusage/min_swap",
		 gtop_properties.memusage_thresholds [SWAP]);
}
