/*
 * bonobo-moniker-http.c: HTTP based Moniker
 *
 * Copyright (c) 2000 Helix Code, Inc.
 *
 * Author:
 *   Joe Shaw (joe@helixcode.com)
 *
 */

#include <config.h>
#include <gnome.h>
#include <liboaf/liboaf.h>
#include <bonobo/bonobo.h>

#include "bonobo-moniker-http.h"

#define PREFIX_LEN (sizeof ("http:") - 1)

static int running_objects = 0;
static BonoboGenericFactory *http_factory = NULL;

static BonoboMonikerClass *bonobo_moniker_http_parent_class;

static Bonobo_Moniker 
http_parse_display_name(BonoboMoniker *moniker,
			Bonobo_Moniker parent,
			const CORBA_char *name,
			CORBA_Environment *ev)
{
	int i;

	g_return_val_if_fail(moniker != NULL, CORBA_OBJECT_NIL);
	g_return_val_if_fail(strlen(name) >= PREFIX_LEN, CORBA_OBJECT_NIL);

	bonobo_moniker_set_parent(moniker, parent, ev);

	i = bonobo_moniker_util_seek_std_separator(name, 0);

	bonobo_moniker_set_name(moniker, name, i);

	/* This is a hack because we don't want the item moniker to be
	   launched. */
	if (name[i] == '!')
		i++;

	if (name[i])
		return bonobo_moniker_util_new_from_name_full(
			bonobo_object_corba_objref(BONOBO_OBJECT(moniker)),
			&name[i], ev);
	else
		return bonobo_object_dup_ref(
			bonobo_object_corba_objref(BONOBO_OBJECT(moniker)), 
			ev);
} /* http_parse_display_name */

static Bonobo_Unknown
http_resolve(BonoboMoniker *moniker,
	     const Bonobo_ResolveOptions *options,
	     const CORBA_char *requested_interface,
	     CORBA_Environment *ev)
{
	const char *url = bonobo_moniker_get_name(moniker, PREFIX_LEN);
	char *real_url;

	/* because resolving the moniker drops the "http:" */
	real_url = g_strconcat("http:", url, NULL);

	if (strcmp(requested_interface, "IDL:Bonobo/Control:1.0") == 0) {
		BonoboObjectClient *client;
		Bonobo_Unknown object;

		client = bonobo_object_activate(
			"OAFIID:control:ebrowser:2b45f4d6-dc8d-49ad-9435-5af899b9ee4c", 0);

		if (!client) {
			/* FIXME: Set a InterfaceNotFound exception here? */
			return CORBA_OBJECT_NIL;
		}

		object = bonobo_object_corba_objref(BONOBO_OBJECT(client));
			
		if (ev->_major != CORBA_NO_EXCEPTION)
			return CORBA_OBJECT_NIL;

		if (object == CORBA_OBJECT_NIL) {
			g_warning("Can't find object satisfying requirements");
			CORBA_exception_set(
				ev, CORBA_USER_EXCEPTION,
				ex_Bonobo_Moniker_InterfaceNotFound, NULL);
			return CORBA_OBJECT_NIL;
		}

		return bonobo_moniker_util_qi_return(
			object, requested_interface, ev);
	}
	else if (strcmp(requested_interface, "IDL:Bonobo/Stream:1.0") == 0) {
		BonoboStream *stream;

		stream = bonobo_stream_open(
			"http", real_url, Bonobo_Storage_READ, 0644);

		if (!stream) {
			g_warning("Failed to open stream '%s'", real_url);
			g_free(real_url);
			CORBA_exception_set(
				ev, CORBA_USER_EXCEPTION,
				ex_Bonobo_Moniker_InterfaceNotFound, NULL);

			return CORBA_OBJECT_NIL;
		}

		g_free(real_url);
		return CORBA_Object_duplicate(
			bonobo_object_corba_objref(BONOBO_OBJECT(stream)), ev);
	}

	return CORBA_OBJECT_NIL;
} /* http_resolve */

static void
bonobo_moniker_http_class_init(BonoboMonikerHTTPClass *klass)
{
	BonoboMonikerClass *mclass = (BonoboMonikerClass *) klass;

	bonobo_moniker_http_parent_class = gtk_type_class(
		bonobo_moniker_get_type());

	mclass->parse_display_name = http_parse_display_name;
	mclass->resolve = http_resolve;
} /* bonobo_moniker_http_class_init */

GtkType
bonobo_moniker_http_get_type(void)
{
	static GtkType type = 0;

	if (!type) {
		GtkTypeInfo info = {
			"BonoboMonikerHTTP",
			sizeof(BonoboMonikerHTTP),
			sizeof(BonoboMonikerHTTPClass),
			(GtkClassInitFunc) bonobo_moniker_http_class_init,
			(GtkObjectInitFunc) NULL,
			NULL, /* reserved 1 */
			NULL, /* reserved 2 */
			(GtkClassInitFunc) NULL
		};

		type = gtk_type_unique(bonobo_moniker_get_type(), &info);
	}

	return type;
} /* bonobo_moniker_http_get_type */

static BonoboMonikerHTTP *
bonobo_moniker_http_new(void)
{
	BonoboMoniker *moniker;
	Bonobo_Moniker corba_moniker;

	moniker = gtk_type_new(bonobo_moniker_http_get_type());
	corba_moniker = bonobo_moniker_corba_object_create(
		BONOBO_OBJECT(moniker));

	if (corba_moniker == CORBA_OBJECT_NIL) {
		bonobo_object_unref(BONOBO_OBJECT(moniker));
		return NULL;
	}

	if (!bonobo_object_construct(BONOBO_OBJECT(moniker), corba_moniker)) {
		bonobo_object_unref(BONOBO_OBJECT(moniker));
		return NULL;
	}

	return BONOBO_MONIKER_HTTP(moniker);
} /* bonobo_moniker_http_new */

static void
http_destroy_cb(BonoboMoniker *moniker)
{
	running_objects--;

	if (running_objects > 0)
		return;

	bonobo_object_unref(BONOBO_OBJECT(http_factory));
	gtk_main_quit();
} /* http_destroy_cb */

static BonoboObject *
bonobo_moniker_http_factory(BonoboGenericFactory *this, void *closure)
{
	BonoboMonikerHTTP *http_moniker = bonobo_moniker_http_new();

	gtk_signal_connect(
		GTK_OBJECT(http_moniker), "destroy",
		GTK_SIGNAL_FUNC(http_destroy_cb), NULL);

	running_objects++;

	return BONOBO_OBJECT(http_moniker);
} /* bonobo_moniker_http_factory */

int
main(int argc, char *argv[])
{
	CORBA_Environment ev;
	CORBA_ORB orb = CORBA_OBJECT_NIL;

	CORBA_exception_init(&ev);

	gnome_init_with_popt_table(
		"http-moniker", "0.0", argc, argv, oaf_popt_options, 0, NULL);
	orb = oaf_init(argc, argv);

	if (!bonobo_init(orb, CORBA_OBJECT_NIL, CORBA_OBJECT_NIL))
		g_error(_("Could not initialize Bonobo"));

	http_factory = bonobo_generic_factory_new(
		"OAFIID:HTTPMonikerFactory:7ab32710-38f0-472d-a720-965feed9da4f",
		bonobo_moniker_http_factory, NULL);

	bonobo_main();

	CORBA_exception_free(&ev);

	return 0;
} /* main */
