/* $Id: guppi-scm-stream.c,v 1.2 2000/01/25 15:49:06 trow Exp $ */

/*
 * guppi-scm-stream.c
 *
 * Copyright (C) 1999, 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org>
 * and Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "guppi-scm-stream.h"

static long stream_type_tag;

#define SCM_TO_STREAM(x) ((GuppiStream*)SCM_CDR(x))
#define SCM_STREAM_P(x) (SCM_NIMP(x) && SCM_CAR(x) == stream_type_tag)

gboolean
scm_stream_p(SCM x)
{
  return SCM_STREAM_P(x);
}

SCM
stream2scm(GuppiStream* x)
{
  SCM smob;

  SCM_DEFER_INTS;
  SCM_NEWCELL(smob);
  SCM_SETCAR(smob, stream_type_tag);
  SCM_SETCDR(smob, x);
  gtk_object_ref(GTK_OBJECT(x));
  SCM_ALLOW_INTS;

  return smob;
}

GuppiStream*
scm2stream(SCM x)
{
  return SCM_STREAM_P(x) ? SCM_TO_STREAM(x) : NULL;
}

static SCM
mark_stream(SCM x)
{
  return SCM_BOOL_F;
}

static scm_sizet
free_stream(SCM x)
{
  GuppiStream* sb = SCM_TO_STREAM(x);

  SCM_DEFER_INTS;
  gtk_object_unref(GTK_OBJECT(sb));
  SCM_ALLOW_INTS;

  return 0;
}

static int
print_stream(SCM x, SCM port, scm_print_state* state)
{
  scm_puts("<GuppiStream>", port);
  return 1;
}

/****************************************************************************/

GUPPI_PROC(sbp, "stream?",
	   1,0,0, (SCM x))
{
  return gh_bool2scm(SCM_STREAM_P(x));
}

GUPPI_PROC(sb_open, "stream-open-file",
	   1,0,0, (SCM x))
{
  GuppiStream* sb;
  gchar* fn;
  gint len;
  SCM ssb;

  SCM_ASSERT(gh_string_p(x), x, SCM_ARG1, str_sb_open);

  fn = gh_scm2newstr(x, &len);
  sb = guppi_stream_open_file(fn);
  g_free(fn);
  if (sb != NULL) {
    ssb = stream2scm(sb);
    return ssb;
  }
  
  return SCM_BOOL_F;
}

GUPPI_PROC(sb_src, "stream-source",
	   1,0,0, (SCM ssb))
{
  SCM_ASSERT(SCM_STREAM_P(ssb), ssb, SCM_ARG1, str_sb_open);
  return gh_str02scm((gchar*)guppi_stream_source(SCM_TO_STREAM(ssb)));
}

GUPPI_PROC(sb_nl, "stream-number-of-lines",
	   1,0,0, (SCM ssb))
{
  SCM_ASSERT(SCM_STREAM_P(ssb), ssb, SCM_ARG1, str_sb_nl);
  return gh_int2scm(guppi_stream_number_of_lines(SCM_TO_STREAM(ssb)));
}

GUPPI_PROC(sb_enl, "stream-estimated-number-of-lines",
	   1,0,0, (SCM ssb))
{
  SCM_ASSERT(SCM_STREAM_P(ssb), ssb, SCM_ARG1, str_sb_enl);
  return gh_int2scm(guppi_stream_estimated_number_of_lines(SCM_TO_STREAM(ssb)));
}


GUPPI_PROC(sb_get, "stream-get-line",
	   2,0,0, (SCM ssb, SCM si))
{
  GuppiStream* sb;
  gint i;
  const gchar* line;

  SCM_ASSERT(SCM_STREAM_P(ssb), ssb, SCM_ARG1, str_sb_get);
  SCM_ASSERT(gh_exact_p(si), si, SCM_ARG2, str_sb_get);

  i = gh_scm2int(si);
  SCM_ASSERT(i >= 0, si, SCM_OUTOFRANGE, str_sb_get);

  sb = SCM_TO_STREAM(ssb);
  line = guppi_stream_get_line(sb, i);
  if (line == NULL) g_message("Null line %d", i);
  SCM_ASSERT(line != NULL, si, SCM_OUTOFRANGE, str_sb_get);

  return gh_str02scm((gchar*)line);
}

GUPPI_PROC(sb_mget, "stream-get-marked-line",
	   2,0,0, (SCM ssb, SCM si))
{
  GuppiStream* sb;
  gint i;
  const gchar* line;

  SCM_ASSERT(SCM_STREAM_P(ssb), ssb, SCM_ARG1, str_sb_mget);
  SCM_ASSERT(gh_exact_p(si), si, SCM_ARG2, str_sb_mget);

  i = gh_scm2int(si);
  SCM_ASSERT(i >= 0, si, SCM_OUTOFRANGE, str_sb_mget);

  sb = SCM_TO_STREAM(ssb);
  line = guppi_stream_get_marked_line(sb, i);
  if (line == NULL) g_message("Null line %d", i);
  SCM_ASSERT(line != NULL, si, SCM_OUTOFRANGE, str_sb_mget);

  return gh_str02scm((gchar*)line);
}

GUPPI_PROC(sb_sani, "stream-get-sanitized-line",
	   2,0,0, (SCM ssb, SCM si))
{
  GuppiStream* sb;
  gint i;
  const gchar* line;
  SCM sline;

  SCM_ASSERT(SCM_STREAM_P(ssb), ssb, SCM_ARG1, str_sb_sani);
  SCM_ASSERT(gh_exact_p(si), si, SCM_ARG2, str_sb_sani);

  i = gh_scm2int(si);
  SCM_ASSERT(i >= 0, si, SCM_OUTOFRANGE, str_sb_sani);

  sb = SCM_TO_STREAM(ssb);
  line = guppi_stream_get_sanitized_line(sb, i);
  if (line == NULL) g_message("Null line %d", i);
  SCM_ASSERT(line != NULL, si, SCM_OUTOFRANGE, str_sb_sani);

  sline = gh_str02scm((gchar*)line);

  return sline;
}

/****************************************************************************/

void
guppi_scm_stream_init(void)
{
  static gboolean init = FALSE;
  static struct scm_smobfuns sb_fns = {
    mark_stream, free_stream, print_stream, NULL
  };

  g_return_if_fail(!init);
  init = TRUE;

  stream_type_tag = scm_newsmob(&sb_fns);

#include "guppi-scm-stream.x"

}


/* $Id: guppi-scm-stream.c,v 1.2 2000/01/25 15:49:06 trow Exp $ */
