/* $Id: guppi-plot-plug-in.c,v 1.2 2000/01/19 19:46:10 trow Exp $ */

/*
 * guppi-plot-plug-in.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org>
 * and Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include "gmodule.h"
#include "guppi-plot-plug-in.h"

static GHashTable* plug_in_table = NULL;

static gint
version_compare(const GuppiPlotPlugInInfo* a, const GuppiPlotPlugInInfo* b)
{
  if (a->major_version < b->major_version)
    return -1;
  else if (a->major_version > b->major_version)
    return +1;
  else if (a->minor_version < b->minor_version)
    return -1;
  else if (a->minor_version > b->minor_version)
    return +1;
  else if (a->micro_version < b->micro_version)
    return -1;
  else if (a->micro_version > b->micro_version)
    return +1;
  else
    return 0;
}

static void
register_plug_in_info(GuppiPlotPlugInInfo* info)
{
  gpointer data;
  GuppiPlotPlugInInfo* ai;
  gint rv;

  g_return_if_fail(info != NULL);

  g_return_if_fail(info->external_name != NULL);
  g_return_if_fail(info->internal_name != NULL);
  g_return_if_fail(info->element_constructor != NULL);

  if (plug_in_table == NULL) 
    plug_in_table = g_hash_table_new(g_str_hash, g_str_equal);

  data = g_hash_table_lookup(plug_in_table, info->internal_name);
  if (data != NULL) {

    ai = (GuppiPlotPlugInInfo*)data;
    rv = version_compare(ai, info);

    if (rv == -1) { 
      /* The already-loaded version is older */
      g_message("Replacing %s %d.%d.%d with %d.%d.%d",
		info->internal_name,
		info->major_version, info->micro_version, info->micro_version,
		ai->major_version, ai->micro_version, ai->micro_version);

      g_hash_table_remove(plug_in_table, info->internal_name);
      
    } else {
      /* The already-loaded version is newer or the same */

      g_message("Skipping %s %d.%d.%d",
		info->internal_name,
		info->major_version, info->micro_version, info->micro_version);
      return;
    }
  }

  g_hash_table_insert(plug_in_table, (gchar*)info->internal_name, info);
}

static GuppiPlotPlugInInfo*
lookup_plug_in_info(const gchar* name)
{
  gpointer data;

  g_return_val_if_fail(name != NULL, NULL);

  data = g_hash_table_lookup(plug_in_table, name);
  if (data == NULL)
    g_warning("Failed plot plug-in lookup: \"%s\"", name);

  return (GuppiPlotPlugInInfo*)data;
}

GuppiPlotElement*
guppi_plot_element_new(const gchar* name)
{
  GuppiPlotPlugInInfo* info;

  g_return_val_if_fail(name != NULL, NULL);

  info = lookup_plug_in_info(name);
  if (info == NULL)
    return NULL;

  return (info->element_constructor)();
}

void
guppi_load_plot_plug_in(const gchar* path)
{
  GModule* module;
  gboolean found_symbol;
  gpointer plot_plug_in = NULL;
  GuppiPlotPlugInInfo* info;

  g_return_if_fail(path != NULL);

  module = g_module_open(path, G_MODULE_BIND_LAZY);
  if (module == NULL) {
    g_warning("Attempt to open plug-in %s failed: %s",
	      path, g_module_error());
    return;
  }

  found_symbol = g_module_symbol(module,"plot_plug_in",&plot_plug_in);
  if (! found_symbol) {
    g_warning("Can't find symbol plot_plug_in in %s", path);
    g_module_close(module);
    return;
  }

  if (plot_plug_in == NULL) {
    g_warning("In %s, the symbol plot_plug_in is NULL", path);
    g_module_close(module);
    return;
  }

  info = ((GuppiPlotPlugInInfo* (*)(void))plot_plug_in)();
  if (info == NULL) {
    g_warning("In %s, plot_plug_in() returned NULL", path);
    g_module_close(module);
    return;
  }

  if (info->magic_number != GUPPI_PLOT_PLUG_IN_MAGIC_NUMBER) {
    g_warning("In %s, plot_plug_in() returned a structure with a bad magic number.", path);
    g_module_close(module);
    return;
  }

  info->load_path = g_strdup(path);
  info->reserved = module;

  register_plug_in_info(info);
  g_message("Successfully loaded plug-in: %s", info->external_name);
}

void
guppi_load_plot_plug_in_dir(const gchar* path, gboolean recursive)
{
  DIR* dir;
  struct dirent* dirent;
  struct stat buf;
  gchar str[1024];

  dir = opendir(path);
  if (dir == NULL) {
    g_message("Couldn't load plug-ins from %s", path);
    return;
  }

  errno = 0;
  while ((dirent = readdir(dir)) != NULL) {

    if (recursive &&
	strcmp(dirent->d_name, ".") && 
	strcmp(dirent->d_name, "..")) {
      g_snprintf(str, 1023, "%s/%s", path, dirent->d_name);
      if (stat(str, &buf) < 0) {
	g_message("stat() on %s failed", str);
      } else if (S_ISDIR(buf.st_mode)) 
	guppi_load_plot_plug_in_dir(str, recursive);
    }

    if (!strncmp(dirent->d_name, "lib", 3) &&
	!strcmp(g_extension_pointer(dirent->d_name), "so")) {
      g_snprintf(str, 1023, "%s/%s", path, dirent->d_name);
      guppi_load_plot_plug_in(str);
    }

    errno = 0;
  }
  if (errno) {
    g_message("An error occured while reading %s", path);
  }

  closedir(dir);
}

void
guppi_load_all_plug_ins(void)
{
  guppi_load_plot_plug_in_dir("../../plug-ins", TRUE);
#ifdef GUPPI_PLUGIN_DIR
  guppi_load_plot_plug_in_dir(GUPPI_PLUGIN_DIR, TRUE);
#endif
}

/*****************************************************************************/

static void
add_credit_box(gpointer key, gpointer hash_data, gpointer user_data)
{
  GtkWidget* label;
  GuppiPlotPlugInInfo* info;
  GtkBox* box;
  gchar* str;

  g_return_if_fail(hash_data != NULL);
  g_return_if_fail(user_data != NULL);

  info = (GuppiPlotPlugInInfo*)hash_data;
  box = GTK_BOX(user_data);

  str = g_strdup_printf("%s %d.%d.%d:\n%s\n%s\n%s",
			info->internal_name,
			info->major_version,
			info->minor_version,
			info->micro_version,
			info->load_path,
			info->copyright,
			info->authors);

  label = gtk_label_new(str);
  gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
  gtk_box_pack_start(box, label, TRUE, TRUE, GNOME_PAD_SMALL);
  g_free(str);
}

GtkWidget*
guppi_plot_plug_in_credits_new(void)
{
  GtkWidget* w = gtk_vbox_new(FALSE, GNOME_PAD_SMALL);

  if (plug_in_table == NULL || g_hash_table_size(plug_in_table) == 0) {
    gtk_container_add(GTK_CONTAINER(w),
		      gtk_label_new(_("No plug-ins loaded.")));
  } else {
    g_hash_table_foreach(plug_in_table, add_credit_box, w);
  }

  return w;
}


/* $Id: guppi-plot-plug-in.c,v 1.2 2000/01/19 19:46:10 trow Exp $ */
