/* This is -*- C -*- */
/* $Id: guppi-string-data.c,v 1.3 1999/12/07 17:05:04 trow Exp $ */

/*
 * guppi-string-data.c
 *
 * Copyright (C) 1999 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@emccta.com> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <stdlib.h>
#include <string.h>
#include "guppi-scalar-data.h"
#include "guppi-boolean-data.h"
#include "guppi-categorical-data.h"
#include "guppi-string-data.h"


static GtkObjectClass* parent_class = NULL;

enum {
  ARG_0
};

static void
guppi_string_data_get_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_string_data_set_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_string_data_destroy(GtkObject* obj)
{
  GuppiStringData* sd = GUPPI_STRING_DATA(obj);
  gsize i;

  for(i=0; i<sd->size; ++i)
    g_free(sd->data[i]);

  g_free(sd->data);
  sd->data = NULL;
  
  g_free(sd->sorted_data);
  sd->sorted_data = NULL;

  if (parent_class->destroy)
    parent_class->destroy(obj);
}

static void
guppi_string_data_finalize(GtkObject* obj)
{
  if (parent_class->finalize)
    parent_class->finalize(obj);
}

/*****************************************************************************/

/* GuppiData virtual functions */

static gboolean
guppi_string_data_string_validate(const GuppiData* data, const gchar* sbuf,
				  gchar* errstr, gsize errstr_len)
{
  return TRUE;
}

static void
guppi_string_data_string_get(const GuppiData* data, dindex_t i,
			     gchar* sbuf, gsize maxlen)
{
  const GuppiStringData* sd = GUPPI_STRING_DATA(data);
  if (maxlen > 0) {
    strncpy(sbuf, sd->data[i], maxlen-1);
    sbuf[maxlen-1] = '\0';
  }
}

static void
guppi_string_data_string_set(GuppiData* data, dindex_t i, const gchar* sbuf)
{
  const GuppiStringData* sd = GUPPI_STRING_DATA(data);
  g_free(sd->data[i]);
  sd->data[i] = g_strdup(sbuf);
}

static void
guppi_string_data_string_add(GuppiData* data, const gchar* sbuf)
{
  guppi_string_data_add(GUPPI_STRING_DATA(data), sbuf);
}

static void
guppi_string_data_string_insert(GuppiData* data, dindex_t i, const gchar* sbuf)
{
  guppi_string_data_insert(GUPPI_STRING_DATA(data), i, sbuf);
}

static void
guppi_string_data_string_delete(GuppiData* data, dindex_t i)
{
  guppi_string_data_delete(GUPPI_STRING_DATA(data), i);
}

static GuppiData*
guppi_string_data_copy(const GuppiData* data)
{
  const GuppiStringData* sd = GUPPI_STRING_DATA(data);
  GuppiStringData* copy;
  dindex_t i;

  copy = GUPPI_STRING_DATA(guppi_string_data_new());

  /* This could be optimized */
  for(i=guppi_data_min_index(data); i<=guppi_data_max_index(data); ++i) {
    guppi_string_data_add(copy, guppi_string_data_get(sd, i));
  }
  guppi_data_shift_indices(GUPPI_DATA(copy),
			   guppi_data_min_index(data));

  return GUPPI_DATA(copy);
}

/*****************************************************************************/

static void
guppi_string_data_class_init(GuppiStringDataClass* klass)
{
  GtkObjectClass* object_class = (GtkObjectClass*)klass;

  parent_class = gtk_type_class(GUPPI_TYPE_DATA);

  klass->parent_class.type_name = _("string");

  klass->parent_class.validate = guppi_string_data_string_validate;
  klass->parent_class.get = guppi_string_data_string_get;
  klass->parent_class.set = guppi_string_data_string_set;  
  klass->parent_class.add = guppi_string_data_string_add;
  klass->parent_class.insert = guppi_string_data_string_insert;
  klass->parent_class.del = guppi_string_data_string_delete;
  klass->parent_class.copy = guppi_string_data_copy;

  object_class->get_arg = guppi_string_data_get_arg;
  object_class->set_arg = guppi_string_data_set_arg;
  object_class->destroy = guppi_string_data_destroy;
  object_class->finalize = guppi_string_data_finalize;

}

static void
guppi_string_data_init(GuppiStringData* obj)
{
  obj->data = NULL;
  obj->size = 0;
  obj->poolsize = 0;
  obj->sorted_data = NULL;
  obj->potential_type = 0;
  obj->potential_empty_override = NULL;
  obj->potential_invalid_override = NULL;
}

GtkType
guppi_string_data_get_type(void)
{
  static GtkType guppi_string_data_type = 0;
  if (!guppi_string_data_type) {
    static const GtkTypeInfo guppi_string_data_info = {
      "GuppiStringData",
      sizeof(GuppiStringData),
      sizeof(GuppiStringDataClass),
      (GtkClassInitFunc)guppi_string_data_class_init,
      (GtkObjectInitFunc)guppi_string_data_init,
      NULL, NULL, (GtkClassInitFunc)NULL
    };
    guppi_string_data_type = gtk_type_unique(GUPPI_TYPE_DATA, &guppi_string_data_info);
  }
  return guppi_string_data_type;
}

GuppiData*
guppi_string_data_new(void)
{
  return GUPPI_DATA(gtk_type_new(guppi_string_data_get_type()));
}

/***************************************************************************/

static void
guppi_string_data_grow(GuppiStringData* sd, gsize newsize)
{
  gchar** newdata;
  g_return_if_fail(sd != NULL);

  if (newsize  <= sd->poolsize)
    return;

  newdata = g_new0(gchar*, newsize);
  memcpy(newdata, sd->data, sizeof(gchar*) * sd->size);
  g_free(sd->data);
  sd->data = newdata;
  sd->poolsize = newsize;
}

const gchar*
guppi_string_data_get(const GuppiStringData* sd, dindex_t i)
{
  g_return_val_if_fail(sd != NULL, NULL);
  g_return_val_if_fail(guppi_data_in_bounds(GUPPI_DATA(sd), i), NULL);

  return sd->data[i - guppi_data_min_index(GUPPI_DATA(sd))];
}

void
guppi_string_data_set_nc(GuppiStringData* sd, dindex_t i, gchar* sbuf)
{
  gchar* old_value;
  g_return_if_fail(sd != NULL);
  g_return_if_fail(guppi_data_in_bounds(GUPPI_DATA(sd), i));
  g_return_if_fail(sbuf != NULL);

  old_value = sd->data[i - sd->base.min_index];
  sd->data[i - sd->base.min_index] = sbuf;

  g_free(sd->sorted_data);
  sd->sorted_data = NULL;

  guppi_data_touch_one(GUPPI_DATA(sd), i, old_value);
  g_free(old_value);
}

void
guppi_string_data_set(GuppiStringData* sd, dindex_t i, const gchar* sbuf)
{
  guppi_string_data_set_nc(sd, i, g_strdup(sbuf));
}

void
guppi_string_data_setn(GuppiStringData* sd, dindex_t i,
		       const gchar* sbuf, guint size)
{
  guppi_string_data_set_nc(sd, i, g_strndup(sbuf, size));
}

void
guppi_string_data_add_nc(GuppiStringData* sd, gchar* sbuf)
{
  g_return_if_fail(sd != NULL);
  g_return_if_fail(sbuf != NULL);

  if (sd->size == sd->poolsize)
    guppi_string_data_grow(sd, sd->size ? 2*sd->size : 128);

  sd->data[sd->size] = sbuf;

  ++sd->size;
  ++sd->base.max_index;

  g_free(sd->sorted_data);
  sd->sorted_data = NULL;
  
  guppi_data_touch_add(GUPPI_DATA(sd),
		       sd->base.max_index,
		       sd->data[sd->size-1]);
}

void
guppi_string_data_add(GuppiStringData* sd, const gchar* sbuf)
{
  guppi_string_data_add_nc(sd, g_strdup(sbuf));
}

void
guppi_string_data_addn(GuppiStringData* sd, const gchar* sbuf, guint size)
{
  guppi_string_data_add_nc(sd, g_strndup(sbuf, size));
}

void
guppi_string_data_insert_nc(GuppiStringData* sd, dindex_t i, gchar* sbuf)
{
  gsize j;

  g_return_if_fail(sd != NULL);
  g_return_if_fail(guppi_data_in_bounds(GUPPI_DATA(sd), i));
  g_return_if_fail(sbuf != NULL);

  if (sd->size == sd->poolsize)
    guppi_string_data_grow(sd, sd->size ? 2*sd->size : 128);

  for(j=sd->size-1; j >= (i-sd->base.min_index); --j) 
    sd->data[j+1] = sd->data[j];

  sd->data[i-sd->base.min_index] = sbuf;

  ++sd->size;
  ++sd->base.max_index;

  g_free(sd->sorted_data);
  sd->sorted_data = NULL;

  guppi_data_touch_add(GUPPI_DATA(sd),
		       i,
		       sd->data[i-sd->base.min_index]);
}

void
guppi_string_data_insert(GuppiStringData* sd, dindex_t i, const gchar* sbuf)
{
  guppi_string_data_insert_nc(sd, i, g_strdup(sbuf));
}

void
guppi_string_data_insertn(GuppiStringData* sd, dindex_t i,
			  const gchar* sbuf, guint size)
{
  guppi_string_data_insert_nc(sd, i, g_strndup(sbuf, size));
}

void
guppi_string_data_delete(GuppiStringData* sd, dindex_t i)
{
  gsize j;
  g_return_if_fail(sd != NULL);
  g_return_if_fail(guppi_data_in_bounds(GUPPI_DATA(sd), i));

  g_free(sd->data[i - sd->base.min_index]);

  for(j=i+1-sd->base.min_index; j<sd->size; ++j)
    sd->data[j-1] = sd->data[j];

  --sd->size;
  --sd->base.max_index;

  g_free(sd->sorted_data);
  sd->sorted_data = NULL;

  guppi_data_touch(GUPPI_DATA(sd));
}

/***************************************************************************/

static gint
guppi_string_data_string_index_pair_compare(const void* a, const void* b)
{
  return strcmp(((GuppiStringIndexPair*)a)->str,
		((GuppiStringIndexPair*)b)->str);
}

static void
guppi_string_data_build_sorted(GuppiStringData* sd)
{
  gsize i;
  g_return_if_fail(sd != NULL);

  if (sd->sorted_data != NULL)
    return;

  if (sd->size == 0) {
    sd->sorted_data = NULL;
    return;
  }

  sd->sorted_data = g_new(GuppiStringIndexPair, sd->size);
  for (i=0; i<sd->size; ++i) {
    sd->sorted_data[i].str = sd->data[i];
    sd->sorted_data[i].index = i;
  }
  qsort(sd->sorted_data, sd->size, sizeof(GuppiStringIndexPair),
	guppi_string_data_string_index_pair_compare);
}

dindex_t
guppi_string_data_lookup(const GuppiStringData* sd, const gchar* sbuf)
{
  guint a,b,i;
  dindex_t not_found;
  gint retval;
  gint parity=0;

  g_return_val_if_fail(sd != NULL, 0);
  g_return_val_if_fail(sbuf != NULL, guppi_data_max_index(GUPPI_DATA(sd))+1);

  not_found = guppi_data_max_index(GUPPI_DATA(sd))+1;

  if (sd->size == 0)
    return not_found;

  /* Sometimes we just can't help but cast out const, no matter how
     abusive it may be... */
  if (sd->sorted_data == NULL)
    guppi_string_data_build_sorted((GuppiStringData*)sd);

  /* An ugly binary search */

  /* In case of multiple matches, we return the one with the minimal
     dindex_t value */
  a=0; b=sd->size-1;
  while (1) {

    i = (a+b+parity)/2;
    parity = 1-parity;

    retval = strcmp(sbuf, sd->sorted_data[i].str);
      
    if (i == 0) {
      if (retval < 0)
	return not_found;
      else if (retval == 0)
	return sd->sorted_data[0].index;
    }

    if (i == sd->size-1 && retval > 0) 
      return not_found;

    if (i+1 >= sd->size)
      break;
    else if (retval <= 0) {
      b = i;
      i = (a+i)/2;
    } else if (strcmp(sbuf, sd->sorted_data[i+1].str) > 0) {
      a = i;
      i = (b+i)/2;
    } else 
      break;
  }
  ++i;
    
  
  if (i < sd->size && strcmp(sd->sorted_data[i].str, sbuf) == 0)
    return sd->sorted_data[i].index;
  else
    return not_found;
}

gboolean
guppi_string_data_contains(const GuppiStringData* sd, const gchar* sbuf)
{
  return guppi_data_in_bounds(GUPPI_DATA(sd),
			      guppi_string_data_lookup(sd, sbuf));
}

guint
guppi_string_data_distinct_values(const GuppiStringData* sd)
{
  guint count=1;
  gint i;

  g_return_val_if_fail(sd != NULL, 0);

  if (sd->size < 2)
    return sd->size;

  if (sd->sorted_data == NULL)
    guppi_string_data_build_sorted((GuppiStringData*)sd);

  for(i=1; i<sd->size; ++i) {
    if (strcmp(sd->sorted_data[i].str, sd->sorted_data[i-1].str) != 0)
      ++count;
  }

  return count;
}

guint
guppi_string_data_empty_count(const GuppiStringData* sd)
{
  guint count=0;
  gint i;

  g_return_val_if_fail(sd != NULL, 0);

  for(i=0; i<sd->size; ++i)
    if (sd->data[i] == NULL || *(sd->data[i]) == '\0')
      ++count;

  return count;
}

/* $Id: guppi-string-data.c,v 1.3 1999/12/07 17:05:04 trow Exp $ */
