/* $Id: guppi-scm-data.c,v 1.2 2000/01/17 06:44:17 trow Exp $ */

/*
 * guppi-scm-data.c
 *
 * Copyright (C) 1999,2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org>
 * and Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "guppi-scm-dataset.h"
#include "guppi-scm-data.h"
#include "guppi-scalar-data.h"
#include "guppi-boolean-data.h"
#include "guppi-string-data.h"
#include "guppi-categorical-data.h"

GtkType
scm_symbol2type(SCM x)
{
  gchar* symbol;
  gint len;
  GtkType t = 0;

  if (!gh_symbol_p(x))
    return 0;

  symbol = gh_symbol2newstr(x, &len);

  if (strcmp(symbol, "scalar") == 0) 
    t = GUPPI_TYPE_SCALAR_DATA;
  else if (strcmp(symbol, "boolean") == 0)
    t = GUPPI_TYPE_BOOLEAN_DATA;
  else if (strcmp(symbol, "string") == 0)
    t = GUPPI_TYPE_STRING_DATA;
  else if (strcmp(symbol, "categorical") == 0)
    t = GUPPI_TYPE_CATEGORICAL_DATA;

  g_free(symbol);

  return t;
}

SCM
scm_type2symbol(GtkType t)
{
  if (t == GUPPI_TYPE_SCALAR_DATA)
    return gh_symbol2scm("scalar");
  else if (t == GUPPI_TYPE_BOOLEAN_DATA)
    return gh_symbol2scm("boolean");
  else if (t == GUPPI_TYPE_STRING_DATA)
    return gh_symbol2scm("string");
  else if (t == GUPPI_TYPE_CATEGORICAL_DATA)
    return gh_symbol2scm("categorical");
  else
    return gh_symbol2scm("unknown-type");
}

/*****************************************************************************/

static long data_type_tag;

#define SCM_TO_DATA(x) (GUPPI_DATA(SCM_CDR(x)))
#define SCM_DATA_P(x) (SCM_NIMP(x) && SCM_CAR(x) == data_type_tag)



gboolean
scm_data_p(SCM x)
{
  return SCM_DATA_P(x);
}

SCM
data2scm(GuppiData* x)
{
  SCM smob;

  SCM_DEFER_INTS;
  SCM_NEWCELL(smob);
  SCM_SETCAR(smob, data_type_tag);
  SCM_SETCDR(smob, x);
  gtk_object_ref(GTK_OBJECT(x));
  SCM_ALLOW_INTS;

  return smob;
}

GuppiData*
scm2data(SCM x)
{
  return SCM_DATA_P(x) ? SCM_TO_DATA(x) : NULL;
}

static SCM
mark_data(SCM x)
{
  return SCM_BOOL_F;
}

static scm_sizet
free_data(SCM x)
{
  GuppiData* d = SCM_TO_DATA(x);

  SCM_DEFER_INTS;
  gtk_object_unref(GTK_OBJECT(d));
  SCM_ALLOW_INTS;

  return 0;
}

static int
print_data(SCM x, SCM port, scm_print_state* state)
{
  static gchar buffer[256];
  GuppiData* d;

  d = SCM_TO_DATA(x);

  g_snprintf(buffer, 256,
	     "<GuppiData \"%s\" type=%s size=%d",
	     guppi_data_label(d),
	     guppi_data_type_name(d),
	     guppi_data_size(d));
  scm_puts(buffer, port);

  if (guppi_data_size(d) > 0) {
    g_snprintf(buffer, 256,
	       " range=%d:%d",
	       guppi_data_min_index(d),
	       guppi_data_max_index(d));
    scm_puts(buffer, port);
  }

  scm_puts(">", port);

  return 1;
}

/*****************************************************************************/

GUPPI_PROC(datap, "data?",
	   1,0,0, (SCM x))
{
  return gh_bool2scm(SCM_DATA_P(x));
}

GUPPI_PROC(guppi_data_min_index, "data-min-index",
	   1,0,0, (SCM x))
{
  SCM_ASSERT(SCM_DATA_P(x), x, SCM_ARG1, str_guppi_data_min_index);

  return gh_int2scm(guppi_data_min_index(SCM_TO_DATA(x)));
}

GUPPI_PROC(guppi_data_max_index, "data-max-index",
	   1,0,0, (SCM x))
{
  SCM_ASSERT(SCM_DATA_P(x), x, SCM_ARG1, str_guppi_data_max_index);

  return gh_int2scm(guppi_data_max_index(SCM_TO_DATA(x)));
}

GUPPI_PROC(guppi_data_size, "data-size",
	   1,0,0, (SCM x))
{
  SCM_ASSERT(SCM_DATA_P(x), x, SCM_ARG1, str_guppi_data_size);

  return gh_int2scm(guppi_data_size(SCM_TO_DATA(x)));
}

GUPPI_PROC(guppi_data_in_bounds, "data-in-bounds?",
	   2,0,0, (SCM x, SCM i))
{
  SCM_ASSERT(SCM_DATA_P(x), x, SCM_ARG1, str_guppi_data_in_bounds);
  SCM_ASSERT(gh_exact_p(i), i, SCM_ARG2, str_guppi_data_in_bounds);

  return gh_bool2scm(guppi_data_in_bounds(SCM_TO_DATA(x),
					  gh_scm2int(i)));
}

GUPPI_PROC(guppi_data_shift_indices, "data-shift-indices!",
	   2,0,0, (SCM x, SCM del))
{
  SCM_ASSERT(SCM_DATA_P(x), x, SCM_ARG1, str_guppi_data_shift_indices);
  SCM_ASSERT(gh_exact_p(del), del, SCM_ARG2, str_guppi_data_shift_indices);

  guppi_data_shift_indices(SCM_TO_DATA(x), gh_scm2int(del));

  return x;
}

GUPPI_PROC(guppi_data_label, "data-label",
	   1,0,0, (SCM x))
{
  SCM_ASSERT(SCM_DATA_P(x), x, SCM_ARG1, str_guppi_data_label);

  return gh_str02scm((gchar*)guppi_data_label(SCM_TO_DATA(x)));
}

GUPPI_PROC(guppi_data_set_label, "data-set-label!",
	   2,0,0, (SCM x, SCM label))
{
  gchar* label_str;
  gint len;

  SCM_ASSERT(SCM_DATA_P(x), x, SCM_ARG1, str_guppi_data_set_label);
  SCM_ASSERT(gh_string_p(label), label, SCM_ARG2, str_guppi_data_set_label);

  label_str = gh_scm2newstr(label, &len);
  guppi_data_set_label(SCM_TO_DATA(x), label_str);
  free(label_str);

  return x;
}

GUPPI_PROC(gd_ds, "data-dataset",
	   1,0,0, (SCM sdata))
{
  GuppiDataset* ds;

  SCM_ASSERT(SCM_DATA_P(sdata), sdata, SCM_ARG1, str_gd_ds);
  
  ds = guppi_data_dataset(scm2data(sdata));
  return ds != NULL ? dataset2scm(ds) : SCM_BOOL_F;
}

GUPPI_PROC(gd_set_ds, "data-set-dataset!",
	   2,0,0, (SCM sdata, SCM sds))
{
  GuppiData* d;

  SCM_ASSERT(SCM_DATA_P(sdata), sdata, SCM_ARG1, str_gd_set_ds);
  SCM_ASSERT(scm_dataset_p(sds) || sds == SCM_BOOL_F, sds,
	     SCM_ARG2, str_gd_set_ds);

  d = SCM_TO_DATA(sdata);

  if (sds == SCM_BOOL_F)
    guppi_data_set_dataset(d, NULL);
  else
    guppi_data_set_dataset(d, scm2dataset(sds));

  return sdata;
}

GUPPI_PROC(guppi_data_type, "data-type",
	   1,0,0, (SCM x))
{
  SCM_ASSERT(SCM_DATA_P(x), x, SCM_ARG1, str_guppi_data_type);

  return scm_type2symbol(GTK_OBJECT(SCM_TO_DATA(x))->klass->type);
}

GUPPI_PROC(guppi_data_type_name, "data-type-name",
	   1,0,0, (SCM x))
{
  SCM_ASSERT(SCM_DATA_P(x), x, SCM_ARG1, str_guppi_data_type_name);

  return gh_str02scm((gchar*)guppi_data_type_name(SCM_TO_DATA(x)));
}

/* Virtual functions */

GUPPI_PROC(d_valid, "data-valid?",
	   2,0,0, (SCM sd, SCM stxt))
{
  gchar* txt;
  gint len;
  gboolean b;

  SCM_ASSERT(SCM_DATA_P(sd), sd, SCM_ARG1, str_d_valid);
  SCM_ASSERT(gh_string_p(stxt), stxt, SCM_ARG2, str_d_valid);

  txt = gh_scm2newstr(stxt, &len);

  b = guppi_data_validate(SCM_TO_DATA(sd), txt, NULL, 0);
  g_free(txt);
  
  return gh_bool2scm(b);
}

GUPPI_PROC(d_get, "data-get",
	   2,0,0, (SCM sd, SCM si))
{
  GuppiData* d;
  gint i;
  gchar buffer[512];
  const gint buffer_len = 512;

  SCM_ASSERT(SCM_DATA_P(sd), sd, SCM_ARG1, str_d_get);
  SCM_ASSERT(gh_exact_p(si), si, SCM_ARG2, str_d_get);

  d = SCM_TO_DATA(sd);

  i = gh_scm2int(si);
  SCM_ASSERT(guppi_data_in_bounds(d,i), si, SCM_OUTOFRANGE, str_d_get);
  
  buffer[0] = '\0';
  guppi_data_get(d, i, buffer, buffer_len);

  return gh_str02scm(buffer);
}

GUPPI_PROC(d_set, "data-set!",
	   3,0,0, (SCM sd, SCM si, SCM stxt))
{
  GuppiData* d;
  gint i;
  gchar* txt;
  gint len;

  SCM_ASSERT(SCM_DATA_P(sd), sd, SCM_ARG1, str_d_set);
  SCM_ASSERT(gh_exact_p(si), si, SCM_ARG2, str_d_set);
  SCM_ASSERT(gh_string_p(stxt), stxt, SCM_ARG3, str_d_set);

  d = SCM_TO_DATA(sd);

  i = gh_scm2int(si);
  SCM_ASSERT(guppi_data_in_bounds(d,i), si, SCM_OUTOFRANGE, str_d_set);

  txt = gh_scm2newstr(stxt, &len);
  guppi_data_set(d, i, txt);
  g_free(txt);

  return sd;
}

GUPPI_PROC(d_add, "data-add!",
	   2,0,0, (SCM sd, SCM stxt))
{
  GuppiData* d;
  gchar* txt;
  gint len;

  SCM_ASSERT(SCM_DATA_P(sd), sd, SCM_ARG1, str_d_add);
  SCM_ASSERT(gh_string_p(stxt), stxt, SCM_ARG2, str_d_add);

  d = SCM_TO_DATA(sd);

  txt = gh_scm2newstr(stxt, &len);
  guppi_data_add(d, txt);
  g_free(txt);

  return sd;
}

GUPPI_PROC(d_ins, "data-insert!",
	   3,0,0, (SCM sd, SCM si, SCM stxt))
{
  GuppiData* d;
  gint i;
  gchar* txt;
  gint len;

  SCM_ASSERT(SCM_DATA_P(sd), sd, SCM_ARG1, str_d_ins);
  SCM_ASSERT(gh_exact_p(si), si, SCM_ARG2, str_d_ins);
  SCM_ASSERT(gh_string_p(stxt), stxt, SCM_ARG3, str_d_ins);

  d = SCM_TO_DATA(sd);

  i = gh_scm2int(si);
  SCM_ASSERT(guppi_data_in_bounds(d,i), si, SCM_OUTOFRANGE, str_d_ins);

  txt = gh_scm2newstr(stxt, &len);
  guppi_data_insert(d, i, txt);
  g_free(txt);

  return sd;
}

GUPPI_PROC(d_del, "data-delete!",
	   2,0,0, (SCM sd, SCM si))
{
  GuppiData* d;
  gint i;

  SCM_ASSERT(SCM_DATA_P(sd), sd, SCM_ARG1, str_d_set);
  SCM_ASSERT(gh_exact_p(si), si, SCM_ARG2, str_d_set);

  d = SCM_TO_DATA(sd);

  i = gh_scm2int(si);
  SCM_ASSERT(guppi_data_in_bounds(d,i), si, SCM_OUTOFRANGE, str_d_set);

  guppi_data_delete(d, i);

  return sd;
}

GUPPI_PROC(d_copy, "data-copy",
	   1,0,0, (SCM sd))
{
  SCM_ASSERT(SCM_DATA_P(sd), sd, SCM_ARG1, str_d_copy);

  return data2scm(guppi_data_copy(SCM_TO_DATA(sd)));
}

/*****************************************************************************/

GUPPI_PROC(d_convpot, "data-conversion-potential",
	   2,0,0, (SCM sd, SCM stype))
{
  GtkType t;
  double p;

  SCM_ASSERT(SCM_DATA_P(sd), sd, SCM_ARG1, str_d_convpot);
  SCM_ASSERT(gh_symbol_p(stype), stype, SCM_ARG2, str_d_convpot);

  t = scm_symbol2type(stype);
  SCM_ASSERT(t != 0, stype, SCM_OUTOFRANGE, str_d_convpot);

  p = guppi_data_conversion_potential(SCM_TO_DATA(sd), t);

  return gh_double2scm(p);
}

GUPPI_PROC(d_conv, "data-convert",
	   2,1,0, (SCM sd, SCM stype, SCM sinv))
{
  GtkType t;
  GuppiData* d;
  gchar* inv = NULL;
  gint len;

  SCM_ASSERT(SCM_DATA_P(sd), sd, SCM_ARG1, str_d_conv);
  SCM_ASSERT(gh_symbol_p(stype), stype, SCM_ARG2, str_d_conv);
  SCM_ASSERT(SCM_UNBNDP(sinv) || gh_string_p(sinv), sinv,
	     SCM_ARG3, str_d_conv);

  t = scm_symbol2type(stype);
  SCM_ASSERT(t != 0, stype, SCM_OUTOFRANGE, str_d_conv);

  if (!SCM_UNBNDP(sinv)) 
    inv = gh_scm2newstr(sinv, &len);

  d = guppi_data_convert(SCM_TO_DATA(sd), t, inv);

  g_free(inv);

  return data2scm(d);
}

/*****************************************************************************/

/*
#include <guppi-type-changer.h>
GUPPI_PROC(foobar, "data-change",
	   1,0,0, (SCM x))
{
  GtkType pot[4];
  GtkWidget* win;
  GtkWidget* gtc;

  SCM_ASSERT(SCM_DATA_P(x), x, SCM_ARG1, str_foobar);
  
  pot[0] = GUPPI_TYPE_SCALAR_DATA;
  pot[1] = GUPPI_TYPE_BOOLEAN_DATA;
  pot[2] = GUPPI_TYPE_STRING_DATA;
  pot[3] = 0;

  win = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtc = guppi_type_changer_new(SCM_TO_DATA(x), pot, 0);
  gtk_container_add(GTK_CONTAINER(win), gtc);
  gtk_widget_show_all(win);

  return x;
}
*/

/*****************************************************************************/

void
guppi_scm_data_init(void)
{
  static gboolean initialized = FALSE;
  static struct scm_smobfuns data_fns = {
    mark_data, free_data, print_data, NULL
  };

  g_return_if_fail(!initialized);
  initialized = TRUE;

  data_type_tag = scm_newsmob(&data_fns);

#include "guppi-scm-data.x"

}

/* $Id: guppi-scm-data.c,v 1.2 2000/01/17 06:44:17 trow Exp $ */
