/* $Id: guppi-scalar-data.c,v 1.1.1.1 1999/12/03 07:02:39 trow Exp $ */

/*
 * guppi-scalar-data.c
 *
 * Copyright (C) 1999 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@emccta.com> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "guppi-scalar-data.h"

static GtkObjectClass* parent_class = NULL;

static void
guppi_scalar_data_finalize(GtkObject* obj)
{
  GuppiScalarData* data = GUPPI_SCALAR_DATA(obj);

  g_free(data);

  g_free(data->sip);

  if (parent_class->finalize)
    (*parent_class->finalize)(obj);
}

/*****************************************************************************/

/*
 * GuppiData virtual functions
 */

static double
guppi_scalar_data_string_convert(const gchar* sbuf)
{
  g_return_val_if_fail(sbuf, -666);

  return atof(sbuf);
}

static gboolean
guppi_scalar_data_string_validate(const GuppiData* data, const gchar* sbuf,
				  gchar* errstr, gsize errstr_len)
{
  gchar* endptr;
  gchar* back;

  strtod(sbuf, &endptr);
  if (endptr == sbuf) {
    if (errstr)
      g_snprintf(errstr,errstr_len,"\"%s\" is not a valid scalar value.",sbuf);
    return FALSE;
  }

  back = g_strdup(endptr);
  g_strstrip(back);

  if (strlen(back) != 0) {
    if (errstr)
      g_snprintf(errstr,errstr_len,
		 "\%s\" is not a valid scalar value:\nExtraneous trailing text \"%s\"", sbuf, back);
    g_free(back);
    return FALSE;
  }
  g_free(back);
  return TRUE;
}

/* We assume inputs are OK, since they are checked by guppi_data_get() */
static void
guppi_scalar_data_string_get(const GuppiData* data, dindex_t i,
			     gchar* sbuf, gsize maxlen)
{
  const GuppiScalarData* sd = GUPPI_SCALAR_DATA(data);

  g_snprintf(sbuf, maxlen, "%g", sd->data[i-sd->base.min_index]);
}

static void
guppi_scalar_data_string_set(GuppiData* data, dindex_t i, const gchar* sbuf)
{
  double x;
  GuppiScalarData* sd;

  sd = GUPPI_SCALAR_DATA(data);

  x = guppi_scalar_data_string_convert(sbuf);
  guppi_scalar_data_set(sd, i, x);
}

static void
guppi_scalar_data_string_add(GuppiData* data, const gchar* sbuf)
{
  double x;
  GuppiScalarData* sd;

  sd = GUPPI_SCALAR_DATA(data);
  x = guppi_scalar_data_string_convert(sbuf);
  guppi_scalar_data_add(sd, x);
}

static void
guppi_scalar_data_string_insert(GuppiData* data, dindex_t i, const gchar* sbuf)
{
  double x;
  GuppiScalarData* sd;

  sd = GUPPI_SCALAR_DATA(data);
  x = guppi_scalar_data_string_convert(sbuf);
  guppi_scalar_data_insert(sd, i, x);
}

/* This has nothing to do with strings, other than being part of the
   "string access" section of the API */
static void
guppi_scalar_data_string_delete(GuppiData* data, dindex_t i)
{
  g_return_if_fail(data);
  guppi_scalar_data_delete(GUPPI_SCALAR_DATA(data), i);
}

static GuppiData*
guppi_scalar_data_copy(const GuppiData* data)
{
  GuppiScalarData* sd = GUPPI_SCALAR_DATA(data);
  GuppiScalarData* copy;
  dindex_t i, i0, i1;

  g_return_val_if_fail(sd != NULL, NULL);

  /* This could be a LOT more efficient */
  copy = GUPPI_SCALAR_DATA(guppi_scalar_data_new());
  i0 = guppi_data_min_index(data);
  i1 = guppi_data_max_index(data);
  for(i=i0; i<=i1; ++i)
    guppi_scalar_data_add(copy, guppi_scalar_data_get(sd, i));

  guppi_data_shift_indices(GUPPI_DATA(copy), i0);

  return GUPPI_DATA(copy);
}

/*****************************************************************************/

static void
guppi_scalar_data_class_init(GuppiScalarDataClass* klass)
{
  GtkObjectClass* object_class;

  object_class = (GtkObjectClass*)klass;
  parent_class = gtk_type_class(GUPPI_TYPE_DATA);

  klass->parent_class.type_name = _("scalar");

  klass->parent_class.validate = guppi_scalar_data_string_validate;
  klass->parent_class.get = guppi_scalar_data_string_get;
  klass->parent_class.set = guppi_scalar_data_string_set;  
  klass->parent_class.add = guppi_scalar_data_string_add;
  klass->parent_class.insert = guppi_scalar_data_string_insert;
  klass->parent_class.del = guppi_scalar_data_string_delete;
  klass->parent_class.copy = guppi_scalar_data_copy;

  object_class->finalize = guppi_scalar_data_finalize;
}

static void
guppi_scalar_data_init(GuppiScalarData* data)
{
  data->data = NULL;
  data->size = data->poolsize = 0;
  data->min = data->max = data->sum = 0;
  data->sip = NULL;
}

/*****************************************************************************/

GtkType
guppi_scalar_data_get_type(void)
{
  static GtkType scalar_data_type = 0;

  if (!scalar_data_type) {
    static const GtkTypeInfo scalar_data_info = {
      "GuppiScalarData",
      sizeof(GuppiScalarData),
      sizeof(GuppiScalarDataClass),
      (GtkClassInitFunc)guppi_scalar_data_class_init,
      (GtkObjectInitFunc)guppi_scalar_data_init,
      NULL, NULL,
      (GtkClassInitFunc)NULL
    };
    scalar_data_type = gtk_type_unique(GUPPI_TYPE_DATA, &scalar_data_info);
  }
  
  return scalar_data_type;
}

static void
guppi_scalar_data_recalc_stats(GuppiScalarData* sd)
{
  gsize i=0;
  double x;

  if (sd->data == NULL || sd->size == 0) {
    sd->min = sd->max = sd->sum = 0;
    return;
  }

  sd->min = sd->max = sd->sum = sd->data[0];

  for(i=1; i<sd->size; ++i) {
    x = sd->data[i];
    if (x < sd->min) sd->min = x;
    if (x > sd->max) sd->max = x;
    sd->sum += x;
  }
}

static int
sip_compare(const void* a, const void* b)
{
  const GuppiScalarIndexPair** aa = (const GuppiScalarIndexPair**)a;
  const GuppiScalarIndexPair** bb = (const GuppiScalarIndexPair**)b;

  return
    ((*aa)->scalar > (*bb)->scalar) - ((*aa)->scalar < (*bb)->scalar);
}

static void
guppi_scalar_data_build_sip_array(GuppiScalarData* sd)
{
  gsize i;
  GuppiScalarIndexPair* a;
  g_return_if_fail(sd != NULL);

  if (sd->sip != NULL)
    return;
    
  sd->sip = g_new(GuppiScalarIndexPair*, sd->size);
  for(i=0; i<sd->size; ++i) {
    a = g_new(GuppiScalarIndexPair, 1);
    a->scalar = sd->data[i];
    a->index = (dindex_t)i - sd->base.min_index;
    sd->sip[i] = a;
  }
  qsort(sd->sip, sd->size, sizeof(GuppiScalarIndexPair*), sip_compare);
  
}

static void
guppi_scalar_data_clear_sip_array(GuppiScalarData* sd)
{
  g_free(sd->sip);
  sd->sip = NULL;
}

GuppiData*
guppi_scalar_data_new(void)
{
  return GUPPI_DATA(gtk_type_new(guppi_scalar_data_get_type()));
}

static void
guppi_scalar_data_grow(GuppiScalarData* sd, gsize newsize)
{
  double* newdata;
  g_return_if_fail(sd);

  if (newsize <= sd->poolsize)
    return;

  newdata = g_new(double, newsize);
  memcpy(newdata, sd->data, sizeof(double) * sd->size);
  g_free(sd->data);
  sd->data = newdata;
  sd->poolsize = newsize;
}

double
guppi_scalar_data_get(const GuppiScalarData* sd, dindex_t i)
{
  g_return_val_if_fail(sd != NULL, 0);
  g_return_val_if_fail(guppi_data_in_bounds(GUPPI_DATA(sd), i), 0);

  return sd->data[i - sd->base.min_index];
}

void
guppi_scalar_data_set(GuppiScalarData* sd, dindex_t i, double x)
{
  double old_value;
  dindex_t i0;

  g_return_if_fail(sd != NULL);
  g_return_if_fail(guppi_data_in_bounds(GUPPI_DATA(sd), i));

  i0 = sd->base.min_index;

  old_value = sd->data[i - i0];

  if (old_value != x) {
    sd->data[i - i0] = x;

    sd->sum += x - old_value;

    if (x > sd->max) sd->max = x;
    if (x < sd->min) sd->min = x;
    if (old_value == sd->min || old_value == sd->max)
      guppi_scalar_data_recalc_stats(sd);
    
    guppi_data_touch_one(GUPPI_DATA(sd), i, &old_value);
  }
}

void
guppi_scalar_data_add(GuppiScalarData* sd, double x)
{
  g_return_if_fail(sd != NULL);
  
  if (sd->size == sd->poolsize)
    guppi_scalar_data_grow(sd, sd->size ? 2*sd->size : 128);

  sd->data[sd->size] = x;

  if (sd->size == 0) {
    sd->min = sd->max = x;
  } else {
    if (x > sd->max) sd->max = x;
    if (x < sd->min) sd->min = x;
  }
  sd->sum += x;

  ++sd->size;
  ++sd->base.max_index;

  guppi_scalar_data_clear_sip_array(sd);

  guppi_data_touch_add(GUPPI_DATA(sd), sd->base.max_index-1, &x);
}

void
guppi_scalar_data_add_many(GuppiScalarData* sd, double* m, gsize N)
{
  gsize i;

  g_return_if_fail(sd);
  g_return_if_fail(m);

  if (sd->size + N >= sd->poolsize)
    guppi_scalar_data_grow(sd, sd->size ? 2*(sd->size+N) : MIN(2*N,128));

  /* Once again, we are safe in assuming that the stride of data that
     we've copied and own == sizeof(double) */
  if (sd->size == 0 && N)
    sd->min = sd->max = m[0];

  for(i=0; i<N; ++i) {
    sd->data[sd->size+i] = m[i];
    if (m[i] > sd->max) sd->max = m[i];
    if (m[i] < sd->min) sd->min = m[i];
    sd->sum += m[i];
  }
  sd->size += N;
  sd->base.max_index += N;

  guppi_scalar_data_clear_sip_array(sd);
  
  guppi_data_touch(GUPPI_DATA(sd));
}

void
guppi_scalar_data_insert(GuppiScalarData* sd, dindex_t i, double x)
{
  gint j;

  g_return_if_fail(sd);
  g_return_if_fail(guppi_data_in_bounds(GUPPI_DATA(sd), i));

  if (sd->size == sd->poolsize)
    guppi_scalar_data_grow(sd, sd->size ? 2*sd->size : 128);

  for(j=sd->size-1; j >= (i-sd->base.min_index); --j) 
    sd->data[j+1] = sd->data[j];

  sd->data[i-sd->base.min_index] = x;

  if (x > sd->max) sd->max = x;
  if (x < sd->min) sd->min = x;
  sd->sum += x;

  ++sd->size;
  ++sd->base.max_index;

  guppi_scalar_data_clear_sip_array(sd);

  guppi_data_touch_add(GUPPI_DATA(sd), i, &x);
}

void
guppi_scalar_data_delete(GuppiScalarData* sd, dindex_t i)
{
  gsize j;
  double x;

  g_return_if_fail(sd);
  g_return_if_fail(guppi_data_in_bounds(GUPPI_DATA(sd), i));

  x = sd->data[i-sd->base.min_index];

  /* Shift everything down by one. */
  for(j=i+1-sd->base.min_index; j<sd->size; ++j)
    sd->data[j-1] = sd->data[j];

  --sd->size;
  --sd->base.max_index;

  sd->sum -= x;
  if (sd->min == x || sd->max == x)
    guppi_scalar_data_recalc_stats(sd);

  guppi_scalar_data_clear_sip_array(sd);

  guppi_data_touch(GUPPI_DATA(sd));
}

const double*
guppi_scalar_data_raw_access(const GuppiScalarData* sd)
{
  g_return_val_if_fail(sd, 0);
  return sd->data;
}

double
guppi_scalar_data_min(const GuppiScalarData* sd)
{
  g_return_val_if_fail(sd, 0);
  return sd->min;
}

double
guppi_scalar_data_max(const GuppiScalarData* sd)
{
  g_return_val_if_fail(sd, 0);
  return sd->max;
}

double
guppi_scalar_data_sum(const GuppiScalarData* sd)
{
  g_return_val_if_fail(sd, 0);
  return sd->sum;
}

double
guppi_scalar_data_mean(const GuppiScalarData* sd)
{
  g_return_val_if_fail(sd, 0);
  return sd->size ? sd->sum / sd->size : 0;
}

double
guppi_scalar_data_var(const GuppiScalarData* sd)
{
  double sumsq, m, om, x;
  gsize i;
  
  g_return_val_if_fail(sd != NULL, -1);
  g_return_val_if_fail(sd->size >= 1, -1);

  m = sd->data[0];
  sumsq = 0;
  for (i=1; i<sd->size; ++i) {
    x = sd->data[i];
    om = m;
    m += (x-m)/(i+1);
    sumsq += (x-m)*(x-om);
  }

  return sumsq / sd->size;
}

double
guppi_scalar_data_vars(const GuppiScalarData* sd)
{
  g_return_val_if_fail(sd != NULL, -1);
  g_return_val_if_fail(sd->size >= 2, -1);

  return (sd->size / (double)(sd->size - 1)) * guppi_scalar_data_var(sd);
}

double
guppi_scalar_data_sdev(const GuppiScalarData* sd)
{
  g_return_val_if_fail(sd != NULL, -1);
  g_return_val_if_fail(sd->size >= 1, -1);

  return sqrt(guppi_scalar_data_var(sd));
}

double
guppi_scalar_data_sdevs(const GuppiScalarData* sd)
{
  g_return_val_if_fail(sd != NULL, -1);
  g_return_val_if_fail(sd->size >= 1, -1);

  return sqrt(guppi_scalar_data_vars(sd));
}

gint
guppi_scalar_data_range_query(const GuppiScalarData* sd,
			      double x0, double x1,
			      GuppiBooleanData* bd)
{
  gsize a,b,i;
  gint count=0;
  double t;

  g_return_val_if_fail(sd != NULL, 0);
  g_return_val_if_fail(bd != NULL, 0);
  g_return_val_if_fail(guppi_data_contains_bounds(GUPPI_DATA(sd),
						  GUPPI_DATA(bd)), 0);

  if (sd->size == 0)
    return 0;
		   
  if (x1 < x0) { t=x0; x0=x1; x1=t; }

  if (x1 < sd->min || x0 > sd->max)
    return 0;

  if (sd->sip == NULL)
    guppi_scalar_data_build_sip_array((GuppiScalarData*)sd);

  if (x0 <= sd->min) {
    i=0;
  } else {
    a=0; b=sd->size-1;
    i = sd->size/2;
    while (1) {
      if (i+1 >= sd->size)
	break;
      else if (sd->sip[i]->scalar >= x0) {
	b = i;
	i = (a+i)/2;
      } else if (sd->sip[i+1]->scalar < x0) {
	a = i;
	i = (b+i)/2;
      } else
	break;
    }
    ++i;
  }
  while (i < sd->size && sd->sip[i]->scalar <= x1) {
    guppi_boolean_data_set_true_unsafe(bd, sd->sip[i]->index);
    ++i;
    ++count;
  }

  return count;
}



/* $Id: guppi-scalar-data.c,v 1.1.1.1 1999/12/03 07:02:39 trow Exp $ */
