/* This is -*- C -*- */
/* $Id: guppi-data-catalog-browser.c,v 1.3 2000/01/05 02:10:59 trow Exp $ */

/*
 * guppi-data-catalog-browser.c
 *
 * Copyright (C) 1999 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@emccta.com> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "guppi-scalar-data.h"
#include "guppi-data-browser.h"
#include "guppi-data-catalog-browser.h"

static GtkObjectClass* parent_class = NULL;

enum {
  ARG_0
};

static void
guppi_data_catalog_browser_get_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_data_catalog_browser_set_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_data_catalog_browser_destroy(GtkObject* obj)
{
  if (parent_class->destroy)
    parent_class->destroy(obj);
}

static void
guppi_data_catalog_browser_finalize(GtkObject* obj)
{
  if (parent_class->finalize)
    parent_class->finalize(obj);
}

static void
guppi_data_catalog_browser_class_init(GuppiDataCatalogBrowserClass* klass)
{
  GtkObjectClass* object_class = (GtkObjectClass*)klass;

  parent_class = gtk_type_class(GTK_TYPE_CLIST);

  object_class->get_arg = guppi_data_catalog_browser_get_arg;
  object_class->set_arg = guppi_data_catalog_browser_set_arg;
  object_class->destroy = guppi_data_catalog_browser_destroy;
  object_class->finalize = guppi_data_catalog_browser_finalize;

}

static void
guppi_data_catalog_browser_init(GuppiDataCatalogBrowser* obj)
{
  obj->catalog = NULL;
}

GtkType
guppi_data_catalog_browser_get_type(void)
{
  static GtkType guppi_data_catalog_browser_type = 0;
  if (!guppi_data_catalog_browser_type) {
    static const GtkTypeInfo guppi_data_catalog_browser_info = {
      "GuppiDataCatalogBrowser",
      sizeof(GuppiDataCatalogBrowser),
      sizeof(GuppiDataCatalogBrowserClass),
      (GtkClassInitFunc)guppi_data_catalog_browser_class_init,
      (GtkObjectInitFunc)guppi_data_catalog_browser_init,
      NULL, NULL, (GtkClassInitFunc)NULL
    };
    guppi_data_catalog_browser_type = gtk_type_unique(GTK_TYPE_CLIST, &guppi_data_catalog_browser_info);
  }
  return guppi_data_catalog_browser_type;
}

/****************************************************************************/

/* Popup menu for right-clicking on data catalog window */

static void
popup_xform_ln_cb(GtkWidget* w, gpointer user_data)
{ }

static void
popup_xform_log10_cb(GtkWidget* w, gpointer user_data)
{ }

static void
popup_xform_exp_cb(GtkWidget* w, gpointer user_data)
{ }

static void
popup_browse_cb(GtkWidget* w, gpointer user_data)
{
  GuppiDataCatalogBrowser* browser;
  GuppiData** selection;
  GtkWidget* window;
  GtkWidget* swin;
  GtkWidget* dbrow;

  browser = GUPPI_DATA_CATALOG_BROWSER(user_data);
  selection = guppi_data_catalog_browser_get_selection(browser);

  if (selection == NULL)
    return;

  window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_default_size(GTK_WINDOW(window), 200, 300);

  swin = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(swin),
				 GTK_POLICY_AUTOMATIC,
				 GTK_POLICY_AUTOMATIC);

  dbrow = guppi_data_browser_new(selection);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(swin),
					dbrow);

  gtk_container_add(GTK_CONTAINER(window), swin);

  gtk_widget_show_all(window);

  /* Should handle signals from the window */

  g_free(selection);
}

static void
popup_copy_cb(GtkWidget* w, gpointer user_data)
{ }

static void
popup_delete_cb(GtkWidget* w, gpointer user_data)
{
  GuppiDataCatalogBrowser* browser;
  GuppiData** selection;
  gint i;

  browser = GUPPI_DATA_CATALOG_BROWSER(user_data);
  selection = guppi_data_catalog_browser_get_selection(browser);
  if (selection == NULL)
    return;

  i=0;
  while (selection[i]) {
    guppi_data_catalog_remove(browser->catalog, selection[i]);
    ++i;
  }
  
  g_free(selection);
}


static void
guppi_data_catalog_build_popup(GuppiDataCatalogBrowser* browser)
{
  static GnomeUIInfo transform_popup[] = {
    { GNOME_APP_UI_ITEM, N_("Ln"), N_("Ln"),
      popup_xform_ln_cb, NULL, NULL,
      GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType)0, NULL },
    { GNOME_APP_UI_ITEM, N_("Log10"), N_("Log10"),
      popup_xform_log10_cb, NULL, NULL,
      GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType)0, NULL },
    { GNOME_APP_UI_ITEM, N_("Exp"), N_("Exp"),
      popup_xform_exp_cb, NULL, NULL,
      GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType)0, NULL },
    GNOMEUIINFO_END
  };

  static GnomeUIInfo popup[] = {
    { GNOME_APP_UI_ITEM, N_("Browse"), N_("Browse"),
      popup_browse_cb, NULL, NULL,
      GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType)0, NULL },
    { GNOME_APP_UI_ITEM, N_("Copy"), N_("Copy"),
      popup_copy_cb, NULL, NULL,
      GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType)0, NULL },
    { GNOME_APP_UI_ITEM, N_("Delete"), N_("Delete"),
      popup_delete_cb, NULL, NULL,
      GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType)0, NULL },
    { GNOME_APP_UI_SUBTREE, N_("Transform"), N_("Transform"),
      transform_popup, NULL, NULL,
      GNOME_APP_PIXMAP_NONE, NULL, 0, (GdkModifierType)0, NULL },
    GNOMEUIINFO_END
  };

  GtkWidget* p;

  p = gnome_popup_menu_new(popup);
  gnome_popup_menu_attach(p, GTK_WIDGET(browser), browser);
}

/****************************************************************************/


static void
guppi_data_catalog_browser_init_clist(GuppiDataCatalogBrowser* browser)
{
  gint i=0, count=0;
  const gchar* titles[] = { N_("Dataset"),
			    N_("Label"),
			    N_("Type"),
			    N_("Size"),
			    N_("Span"),
			    NULL };
  GtkCList* clist;

  g_return_if_fail(browser != NULL);
  
  clist = GTK_CLIST(browser);

  while (titles[count] != NULL) ++count;

  gtk_clist_construct(clist, count, NULL);
  gtk_clist_set_selection_mode(clist, GTK_SELECTION_MULTIPLE);

  gtk_clist_freeze(clist);

  while (titles[i] != NULL) {
    gtk_clist_set_column_title(clist, i, titles[i]);
    ++i;
  }
  gtk_clist_column_titles_show(clist);

  gtk_clist_thaw(clist);
}

static void
guppi_data_catalog_browser_build(GuppiDataCatalogBrowser* browser)
{
  GtkCList* clist;
  GList* iter;
  gint i;
  gchar* cols[5];
  const gint buffer_len = 256;
  gchar size_buffer[256], span_buffer[256];
  

  g_return_if_fail(browser != NULL);
  if (browser->catalog == NULL)
    return;

  clist = GTK_CLIST(browser);

  gtk_clist_freeze(clist);

  gtk_clist_clear(clist);

  /* We reach inside the guts of our browser's catalog in an untowards
     way here. */
  iter = browser->catalog->data;
  i=0;
  while (iter != NULL) {
    cols[0] = (gchar*)guppi_dataset_name(guppi_data_dataset(GUPPI_DATA(iter->data)));
    cols[1] = (gchar*)guppi_data_label(GUPPI_DATA(iter->data));
    cols[2] = (gchar*)guppi_data_type_name(GUPPI_DATA(iter->data));

    g_snprintf(size_buffer, buffer_len, "%d",
	       guppi_data_size(GUPPI_DATA(iter->data)));
    cols[3] = size_buffer;

    if (guppi_data_size(GUPPI_DATA(iter->data)) > 0) {
      g_snprintf(span_buffer, buffer_len, "%d : %d",
		 guppi_data_min_index(GUPPI_DATA(iter->data)),
		 guppi_data_max_index(GUPPI_DATA(iter->data)));
      cols[4] = span_buffer;
    } else {
      cols[4] = "";
    }

    gtk_clist_insert(clist, i, cols);
    ++i;
    iter = g_list_next(iter);
  }

  /* Make each column wide enough so that we can see everything. */
  for(i=0; i<5; ++i)
    gtk_clist_set_column_width(clist, i,
			       gtk_clist_optimal_column_width(clist, i));
    
  gtk_clist_thaw(clist);
}


static void
catalog_changed_cb(GuppiDataCatalog* cat, gpointer user_data)
{
  g_return_if_fail(cat != NULL);
  g_return_if_fail(user_data != NULL);
  guppi_data_catalog_browser_build(GUPPI_DATA_CATALOG_BROWSER(user_data));
}

GtkWidget*
guppi_data_catalog_browser_new(GuppiDataCatalog* cat)
{
  GtkObject* obj = GTK_OBJECT(gtk_type_new(GUPPI_TYPE_DATA_CATALOG_BROWSER));
  
  if (cat == NULL)
    cat = guppi_data_catalog();

  GUPPI_DATA_CATALOG_BROWSER(obj)->catalog = cat;

  if (cat != NULL) {
    g_assert(obj != NULL);
    gtk_signal_connect(GTK_OBJECT(cat),
		       "changed",
		       GTK_SIGNAL_FUNC(catalog_changed_cb),
		       obj);
  }

  guppi_data_catalog_build_popup(GUPPI_DATA_CATALOG_BROWSER(obj));
  guppi_data_catalog_browser_init_clist(GUPPI_DATA_CATALOG_BROWSER(obj));
  guppi_data_catalog_browser_build(GUPPI_DATA_CATALOG_BROWSER(obj));

  return GTK_WIDGET(obj);
}

GuppiDataCatalog*
guppi_data_catalog_browser_get_catalog(GuppiDataCatalogBrowser* browser)
{
  g_return_val_if_fail(browser != NULL, NULL);
  
  return browser->catalog;
}

GuppiData**
guppi_data_catalog_browser_get_selection(GuppiDataCatalogBrowser* browser)
{
  GuppiData** all;
  GuppiDataCatalog* catalog;
  GtkCList* clist;
  GList* clist_iter;
  gint i, j, size;
  GuppiData** result = NULL;


  g_return_val_if_fail(browser != NULL, NULL);
  clist = GTK_CLIST(browser);


  /* Make a first pass through and count */
  clist_iter = clist->row_list;
  size = 0;
  while (clist_iter != NULL) {
    if (GTK_CLIST_ROW(clist_iter)->state == GTK_STATE_SELECTED)
      ++size;
    clist_iter = g_list_next(clist_iter);
  }

  if (size == 0)
    return NULL;

  catalog = guppi_data_catalog_browser_get_catalog(browser);
  g_return_val_if_fail(catalog != NULL, NULL);

  all = guppi_data_catalog_get_all(catalog);

  /* If our size > 0, all can't be NULL! */
  g_assert(all != NULL);

  /* Next allocate and populate our vector */
  result = g_new(GuppiData*, size+1);
  clist_iter = clist->row_list;
  i = j = 0;
  while (clist_iter != NULL) {
    if (GTK_CLIST_ROW(clist_iter)->state == GTK_STATE_SELECTED) {
      result[i] = all[j];
      ++i;
    }
    ++j;
    clist_iter = g_list_next(clist_iter);
  }
  result[i] = NULL;

  for(i=0; all[i] != NULL; ++i)
    gtk_object_unref(GTK_OBJECT(all[i]));
  
  g_free(all);
  return result;
}





/* $Id: guppi-data-catalog-browser.c,v 1.3 2000/01/05 02:10:59 trow Exp $ */
