/* $Id: guppi-boolean-data.c,v 1.1.1.1 1999/12/03 07:02:40 trow Exp $ */

/*
 * guppi-boolean-data.c
 *
 * Copyright (C) 1999 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@emccta.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <string.h>
#include "guppi-boolean-data.h"

static GtkObjectClass* parent_class = NULL;

static void
guppi_boolean_data_finalize(GtkObject* obj)
{
  GuppiBooleanData* bd = GUPPI_BOOLEAN_DATA(obj);
  g_free(bd->data);

  if (parent_class->finalize)
    (*parent_class->finalize)(obj);
}

/****************************************************************************/

/*
 * Virtual functions.
 */

/* First entry of each is "preferred form" */
static const gchar* true_forms[] = { "#t", "t", "true", "yes", "y", "1", NULL };
static const gchar* false_forms[] = { "#f", "f", "false", "no", "n", "0", NULL };

static gboolean
guppi_boolean_data_string_convert(const gchar* sbuf)
{
  gint i=0;
  while (true_forms[i]) {
    if (g_strcasecmp(sbuf, true_forms[i]) == 0)
      return TRUE;
    ++i;
  }

  return FALSE;
}

static gboolean
guppi_boolean_data_string_validate(const GuppiData* data, const gchar* sbuf,
				   gchar* errstr, gsize errstr_len)
{
  gint i=0;
  while (true_forms[i]) {
    if (g_strcasecmp(sbuf, true_forms[i]) == 0)
      return TRUE;
    ++i;
  }
  i=0;
  while (false_forms[i]) {
    if (g_strcasecmp(sbuf, false_forms[i]) == 0)
      return TRUE;
    ++i;
  }

  if (errstr) 
    g_snprintf(errstr, errstr_len, "Not recognized as boolean value");

  return FALSE;
}

static void
guppi_boolean_data_string_get(const GuppiData* data, dindex_t i,
			      gchar* sbuf, gsize maxlen)
{
  const gchar** ptr;
  gboolean q;

  q = guppi_boolean_data_get(GUPPI_BOOLEAN_DATA(data), i);
  ptr = q ? true_forms : false_forms;
  g_snprintf(sbuf, maxlen, "%s", ptr[0]);
}

static void
guppi_boolean_data_string_set(GuppiData* data, dindex_t i, const gchar* sbuf)
{
  gboolean q = guppi_boolean_data_string_convert(sbuf);
  guppi_boolean_data_set(GUPPI_BOOLEAN_DATA(data), i, q);
}

static void
guppi_boolean_data_string_add(GuppiData* data, const gchar* sbuf)
{
  gboolean q = guppi_boolean_data_string_convert(sbuf);
  guppi_boolean_data_add(GUPPI_BOOLEAN_DATA(data), q);

}

static void
guppi_boolean_data_string_insert(GuppiData* data, dindex_t i,const gchar* sbuf)
{
  gboolean q = guppi_boolean_data_string_convert(sbuf);
  guppi_boolean_data_insert(GUPPI_BOOLEAN_DATA(data), i, q);

}

static void
guppi_boolean_data_string_delete(GuppiData* data, dindex_t i)
{
  guppi_boolean_data_delete(GUPPI_BOOLEAN_DATA(data), i);

}

static GuppiData*
guppi_boolean_data_copy(const GuppiData* data)
{
  gsize i;
  GuppiBooleanData* bd = GUPPI_BOOLEAN_DATA(data);
  GuppiBooleanData* bd_new =
    GUPPI_BOOLEAN_DATA(guppi_boolean_data_new_aligned(data));

  for(i=0; i<bd->asize; ++i)
    bd_new->data[i] = bd->data[i];

  return GUPPI_DATA(bd_new);
}

/****************************************************************************/

static void
guppi_boolean_data_class_init(GuppiBooleanDataClass* klass)
{
  GtkObjectClass* object_class;

  object_class = (GtkObjectClass*)klass;
  parent_class = gtk_type_class(GUPPI_TYPE_DATA);

  klass->parent_class.type_name = _("boolean");

  klass->parent_class.validate = guppi_boolean_data_string_validate;
  klass->parent_class.get = guppi_boolean_data_string_get;
  klass->parent_class.set = guppi_boolean_data_string_set;
  klass->parent_class.add = guppi_boolean_data_string_add;
  klass->parent_class.insert = guppi_boolean_data_string_insert;
  klass->parent_class.del = guppi_boolean_data_string_delete;
  klass->parent_class.copy = guppi_boolean_data_copy;

  object_class->finalize = guppi_boolean_data_finalize;
}

static void
guppi_boolean_data_init(GuppiBooleanData* bd)
{
  bd->data = NULL;
  bd->asize = 0;
}

GtkType
guppi_boolean_data_get_type(void)
{
  static GtkType boolean_data_type = 0;

  if (!boolean_data_type) {
    static const GtkTypeInfo boolean_data_info = {
      "GuppiBooleanData",
      sizeof(GuppiBooleanData),
      sizeof(GuppiBooleanDataClass),
      (GtkClassInitFunc)guppi_boolean_data_class_init,
      (GtkObjectInitFunc)guppi_boolean_data_init,
      NULL, NULL,
      (GtkClassInitFunc)NULL
    };
    boolean_data_type = gtk_type_unique(GUPPI_TYPE_DATA, &boolean_data_info);
  }

  return boolean_data_type;
}

/*****************************************************************************/

GuppiData*
guppi_boolean_data_new(void)
{
  return GUPPI_DATA(gtk_type_new(guppi_boolean_data_get_type()));
}

GuppiData*
guppi_boolean_data_new_by_bounds(dindex_t min, dindex_t max)
{
  GuppiBooleanData* bd = GUPPI_BOOLEAN_DATA(guppi_boolean_data_new());

  bd->base.min_index = min;
  bd->base.max_index = max;
  bd->asize = ((max-min+1) >> 5) + 1;
  bd->data = g_new0(guint32, bd->asize);

  return GUPPI_DATA(bd);
}

GuppiData*
guppi_boolean_data_new_aligned(const GuppiData* d)
{
  g_return_val_if_fail(d != NULL, NULL);

  return guppi_boolean_data_new_by_bounds(guppi_data_min_index(d),
					  guppi_data_max_index(d));
}

void
guppi_boolean_data_clear(GuppiBooleanData* bd)
{
  dindex_t i;
  g_return_if_fail(bd != NULL);
  for (i=0; i<bd->asize; ++i)
    bd->data[i] = 0;
}

gboolean
guppi_boolean_data_get(const GuppiBooleanData* bd, dindex_t i)
{
  g_return_val_if_fail(bd != NULL, FALSE);
  g_return_val_if_fail(i >= bd->base.min_index, FALSE);
  g_return_val_if_fail(i <= bd->base.max_index, FALSE);

  i -= bd->base.min_index;
  return bd->data[i >> 5] & (1 << (i & 31));
}

void
guppi_boolean_data_set(GuppiBooleanData* bd, dindex_t i, gboolean q)
{
  g_return_if_fail(bd != NULL);
  g_return_if_fail(i >= bd->base.min_index);
  g_return_if_fail(i <= bd->base.max_index);

  i -= bd->base.min_index;
  if (q)
    bd->data[i >> 5] |= (1 << (i & 31));
  else
    bd->data[i >> 5] &= ~(1 << (i & 31));
}

static void
guppi_boolean_data_grow(GuppiBooleanData* bd, gsize capacity)
{
  guint32* newdata;
  gsize i;

  g_return_if_fail(bd != NULL);

  capacity = capacity >> 5;

  if (capacity <= bd->capacity)
    return;

  newdata = g_new0(guint32, capacity);
  for(i=0; i<bd->asize; ++i)
    newdata[i] = bd->data[i];
  
  g_free(bd->data);
  bd->data = newdata;
  bd->capacity = capacity;
}

static void
guppi_boolean_data_grow_to_accomodate(GuppiBooleanData* bd, gsize N)
{
  gsize c;

  g_return_if_fail(bd);

  c = bd->capacity;
  if (c == 0)
    c = 1024;
  while (((bd->base.max_index - bd->base.min_index + 1 + N) >> 5) > c)
    c *= 2;

  guppi_boolean_data_grow(bd, c<<5);
}

void
guppi_boolean_data_add(GuppiBooleanData* bd, gboolean q)
{
  g_return_if_fail(bd);

  guppi_boolean_data_grow_to_accomodate(bd, 1);
  ++bd->base.max_index;
  bd->asize = ((bd->base.max_index-bd->base.min_index+1)>>5)+1;
  guppi_boolean_data_set(bd, bd->base.max_index, q);
}

void
guppi_boolean_data_insert(GuppiBooleanData* bd, dindex_t i, gboolean q)
{
  gsize j,k;
  gboolean carry;
  guint32 r;
  const guint32 topbit = 1<<32;

  g_return_if_fail(bd);
  g_return_if_fail(bd->base.min_index <= i);
  g_return_if_fail(i <= bd->base.max_index);

  guppi_boolean_data_grow_to_accomodate(bd, 1);
  ++bd->base.max_index;
  bd->asize = ((bd->base.max_index-bd->base.min_index+1)>>5)+1;

  i -= bd->base.min_index;

  j = i >> 5;
  k = i & 31;
  r = bd->data[j];
  carry = r & topbit;

  bd->data[j++] = ((r & ~((1<<k)-1)) << 1) | (r & ((1<<k)-1));

  while (j < bd->asize) {
    r = bd->data[j] << 1;
    if (carry)
      r = r | 1;
    carry = bd->data[j] & topbit;
    bd->data[j] = r;
    ++j;
  }

  guppi_boolean_data_set(bd, i, q);
}

void
guppi_boolean_data_delete(GuppiBooleanData* bd, dindex_t i)
{
  gsize j,k;
  guint32 r;
  const guint32 topbit = 1<<32;

  g_return_if_fail(bd);
  g_return_if_fail(bd->base.min_index <= i);
  g_return_if_fail(i <= bd->base.max_index);

  i -= bd->base.min_index;

  j = i >> 5;
  k = i & 31;
  r = bd->data[j];

  bd->data[j++] = ((r & ~((1<<(k+1))-1)) >> 1) | (r & ((1<<k)-1));
  
  while (j < bd->asize) {
    r = bd->data[j];
    if (r & 1)
      bd->data[j-1] |= topbit;
    bd->data[j] = r >> 1;
    ++j;
  }

  --bd->base.max_index;
  bd->asize = ((bd->base.max_index-bd->base.min_index+1)>>5)+1;
}

void
guppi_boolean_data_logical_and(GuppiBooleanData* bd, const GuppiBooleanData* a)
{
  dindex_t i, i0, i1;
  gboolean q;
  gsize j;

  g_return_if_fail(bd != NULL);
  g_return_if_fail(a != NULL);

  /* We can optimize the easy case. */
  if (bd->base.min_index == a->base.min_index && 
      bd->base.max_index == a->base.max_index) {
    for(j=0; j<bd->asize; ++j)
      bd->data[j] &= a->data[j];
    return;
  }

  i0 = MAX(bd->base.min_index, a->base.min_index);
  i1 = MIN(bd->base.max_index, a->base.max_index);

  for(i=i0; i<=i1; ++i) {
    q = guppi_boolean_data_get(bd, i);
    if (q && !guppi_boolean_data_get(a, i))
      guppi_boolean_data_set(bd, i, FALSE);
  }
}

void
guppi_boolean_data_logical_or(GuppiBooleanData* bd, const GuppiBooleanData* a)
{
  dindex_t i, i0, i1;
  gboolean q;
  gsize j;

  g_return_if_fail(bd != NULL);
  g_return_if_fail(a != NULL);

  /* We can optimize the easy case. */
  if (bd->base.min_index == a->base.min_index && 
      bd->base.max_index == a->base.max_index) {
    for(j=0; j<bd->asize; ++j)
      bd->data[j] |= a->data[j];
    return;
  }

  i0 = MAX(bd->base.min_index, a->base.min_index);
  i1 = MIN(bd->base.max_index, a->base.max_index);

  for(i=i0; i<=i1; ++i) {
    q = guppi_boolean_data_get(bd, i);
    if (!q && guppi_boolean_data_get(a, i))
      guppi_boolean_data_set(bd, i, TRUE);
  }
}

void
guppi_boolean_data_logical_xor(GuppiBooleanData* bd, const GuppiBooleanData* a)
{
  dindex_t i, i0, i1;
  gboolean q, p;
  gsize j;

  g_return_if_fail(bd != NULL);
  g_return_if_fail(a != NULL);

  /* We can optimize the easy case. */
  if (bd->base.min_index == a->base.min_index && 
      bd->base.max_index == a->base.max_index) {
    for(j=0; j<bd->asize; ++j)
      bd->data[j] ^= a->data[j];
  }

  i0 = MAX(bd->base.min_index, a->base.min_index);
  i1 = MIN(bd->base.max_index, a->base.max_index);

  for(i=i0; i<=i1; ++i) {
    q = guppi_boolean_data_get(bd, i);
    p = guppi_boolean_data_get(a, i);
    if ((p ^ q) != q)
      guppi_boolean_data_set(bd, i, p ^ q);
  }
}

void
guppi_boolean_data_logical_not(GuppiBooleanData* bd)
{
  dindex_t i0=bd->base.min_index, i1=bd->base.max_index;
  gsize j=0;
  
  while (i1-i0>=32) {
    bd->data[j] = ~ bd->data[j];
    ++j;
    i0 += 32;
  }
  bd->data[j] = bd->data[j] ^ (~ ((1<<(i1-i0+1))-1));
}

void
guppi_boolean_data_replace(GuppiBooleanData* dest, const GuppiBooleanData* src)
{
  g_return_if_fail(dest != NULL);
  g_return_if_fail(src != NULL);

  if (dest == src)
    return;

  if (dest->base.min_index != src->base.min_index ||
      dest->base.max_index != src->base.max_index) 
    g_error("Non-aligned boolean copy not implemented.");

  memcpy(dest->data, src->data, src->asize * sizeof(guint32));
}


dindex_t 
guppi_boolean_data_first_true(const GuppiBooleanData* bd)
{
  dindex_t i=0;
  guint32 q;

  g_return_val_if_fail(bd != NULL, 0);

  while (i < bd->asize && bd->data[i] == 0) 
    ++i;
  if (i >= bd->asize)
    return bd->base.max_index+1;
  q = bd->data[i];
  i = i << 5;
  while (!(q&1)) {
    q = q>>1;
    ++i;
  }
  return i+bd->base.min_index;
}

dindex_t
guppi_boolean_data_next_true(const GuppiBooleanData* bd, dindex_t i)
{
  dindex_t j,k;
  guint32 q;
  dindex_t i0, i1;

  g_return_val_if_fail(bd != NULL, 0);

  i1 = bd->base.max_index;

  if (i >= i1)
    return 1+i;

  i0 = bd->base.min_index;

  i -= (i0 - 1);
  j = i >> 5;
  k = i & 31;
  if (k) {
    q = bd->data[j] >> k;
    if (q) {
      while (!(q&1)) {
	q = q>>1;
	++i;
      }
      return i + i0;
    } else {
      ++j;
    }
  }

  while (j < bd->asize && bd->data[j] == 0)
    ++j;
  if (j >= bd->asize)
    return i1+1;
  q = bd->data[j];
  i = j<<5;
  while (!(q&1)) {
    q = q>>1;
    ++i;
  }
  return i + i0;
}

guint
guppi_boolean_data_true_count(const GuppiBooleanData* bd)
{
  static guint8* bitcount = NULL;
  int q,i,count;

  if (bitcount == NULL) {
    bitcount = g_new(guint8, 256);
    for(i=0; i<256; ++i) {
      count=0;
      q=i;
      while (q) {
	if (q&1)
	  ++count;
	q = q>>1;
      }
      bitcount[i] = count;
    }
  }

  count=0;
  for(i=0; i<bd->asize; ++i) {
    q = bd->data[i];
    count += bitcount[q & 0xff];
    q = q>>8;
    count += bitcount[q & 0xff];
    q = q>>8;
    count += bitcount[q & 0xff];
    q = q>>8;
    count += bitcount[q & 0xff];
  }
  return (guint)count;
}



/* $Id: guppi-boolean-data.c,v 1.1.1.1 1999/12/03 07:02:40 trow Exp $ */
