/* $Id: file-open.c,v 1.2 2000/01/20 18:50:11 trow Exp $ */

/*
 * file-open.c
 *
 * Copyright (C) 1999 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org>
 * and Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <gnome.h>

#include <guppi-data-catalog.h>
#include <guppi-stream.h>
#include <guppi-delimited-importer.h>
#include <guppi-delimited-preview.h>
#include <guppi-type-changer.h>
#include <guppi-type-dialog.h>
#include "file-open.h"

void
file_open(void)
{
  static void on_file_ok_clicked(GtkWidget*, gpointer);
  GtkFileSelection* file_selector;

  file_selector = GTK_FILE_SELECTION(gtk_file_selection_new(_("Select Import File")));

  gtk_signal_connect(GTK_OBJECT(file_selector->ok_button),
		     "clicked",
                     GTK_SIGNAL_FUNC(on_file_ok_clicked),
		     file_selector);

  gtk_signal_connect_object(GTK_OBJECT(file_selector->cancel_button),
			    "clicked",
			    GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    GTK_OBJECT(file_selector));
  
  gtk_widget_show_all(GTK_WIDGET(file_selector));
}

static void
on_file_ok_clicked(GtkWidget* button,
		   gpointer user_data)
{

  GtkFileSelection* fs = GTK_FILE_SELECTION(user_data);
  gchar* filename;
  gchar buffer[256];
  gint buffer_len = 256;

  GuppiStream* stream;
  GuppiImporter* delim;

  GtkWidget* dialog;
  GtkWidget* preview;
  GtkWidget* ok_button;
  GtkWidget* cancel_button;

  static void on_delim_preview_ok_clicked(GtkWidget*, gpointer);

  g_return_if_fail(fs != NULL);

  filename = gtk_file_selection_get_filename(fs);
  stream = guppi_stream_open_file(filename);

  if (stream == NULL) {
    g_snprintf(buffer, buffer_len,
	       _("Couldn't open file \"%s\""), filename);
    gtk_widget_show_all(gnome_error_dialog(buffer));
    return;
  }

  gtk_widget_destroy(GTK_WIDGET(fs));

  /* Construct & initialize our importer */
  delim = guppi_delimited_importer_new();
  guppi_importer_set_stream(delim, stream);

  gtk_object_unref(GTK_OBJECT(stream));
  stream=NULL;

  guppi_importer_intuit(delim);

  /* Construct our preview & a dialog to hold it */
  preview = guppi_delimited_preview_new(GUPPI_DELIMITED_IMPORTER(delim));
  gtk_object_sink(GTK_OBJECT(delim));


  dialog = gtk_dialog_new();
  gtk_object_set_data(GTK_OBJECT(preview), "dialog", dialog);

  ok_button = gnome_stock_button(GNOME_STOCK_BUTTON_OK);
  cancel_button = gnome_stock_button(GNOME_STOCK_BUTTON_CANCEL);
  
  gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), preview);

  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area),
		     ok_button, TRUE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->action_area),
		     cancel_button, TRUE, FALSE, 0);
  

  gtk_signal_connect(GTK_OBJECT(ok_button),
		     "clicked",
		     GTK_SIGNAL_FUNC(on_delim_preview_ok_clicked),
		     preview);

  gtk_signal_connect_object(GTK_OBJECT(cancel_button),
			    "clicked",
			    GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    GTK_OBJECT(dialog));

  gtk_widget_show_all(dialog);
}

static void
on_delim_preview_ok_clicked(GtkWidget* button,
			    gpointer user_data)
{
  GuppiDelimitedPreview* gdp;
  GuppiDelimitedImporter* gdi;
  GtkWidget* olddialog;
  GuppiData** datav;
  static void on_type_dialog_ok_clicked(GtkWidget*, gpointer);
  GuppiTypeDialog* dialog;

  gdp = GUPPI_DELIMITED_PREVIEW(user_data);
  g_return_if_fail(gdp != NULL);

  gdi = gdp->importer;
  gtk_object_ref(GTK_OBJECT(gdi));

  /* 
     We have to make sure that the DelimitedPreview is destroyed
     before we start importing --- otherwise, the StreamPreview will
     get a bunch of signals when we are loading the data and, while
     trying to re-render stuff, will screw up the stream's sequential
     mode.
     
     Passing it in object data is a bit hackish, but oh well...
  */

  olddialog = GTK_WIDGET(gtk_object_get_data(GTK_OBJECT(gdp), "dialog"));
  gtk_widget_destroy(olddialog);

  /* Actually import our data, and extract the data from our importer */
  guppi_importer_import(GUPPI_IMPORTER(gdi));
  datav = guppi_importer_data(GUPPI_IMPORTER(gdi));
  gtk_object_unref(GTK_OBJECT(gdi));

  /* Now build a dialog containing type-changers for this data */
  dialog = GUPPI_TYPE_DIALOG(guppi_type_dialog_new(datav));

  gtk_signal_connect(GTK_OBJECT(dialog->ok_button),
		     "clicked",
		     GTK_SIGNAL_FUNC(on_type_dialog_ok_clicked),
		     dialog);

  gtk_widget_show_all(GTK_WIDGET(dialog));
}

static void
on_type_dialog_ok_clicked(GtkWidget* button, gpointer user_data)
{
  GuppiTypeDialog* gtd;
  gint i;
  GuppiData* data;

  gtd = GUPPI_TYPE_DIALOG(user_data);
  g_return_if_fail(gtd != NULL);

  for (i=0; gtd->changers[i] != NULL; ++i) {
    if (!gtd->changers[i]->ignore) {
      data = guppi_type_changer_get_converted_data(gtd->changers[i]);
      guppi_data_catalog_add(NULL, data);
    }
  }

  gtk_widget_destroy(GTK_WIDGET(gtd));
}


/* $Id: file-open.c,v 1.2 2000/01/20 18:50:11 trow Exp $ */
