/* E-Mount v0.3 - 10/26/99
 *
 * Written by Daniel Erat, a.k.a. hunchback
 * erat@cats.ucsc.edu, danerat@mindspring.com
 * http://www2.ucsc.edu/~erat/
 * You can also find me on efnet #e as hunchback.
 *
 * The mount point-checking stuff is taken from wmmount, written by Sam
 * Hawker.  There's probably a ton of stuff that's broken here.  If it
 * doesn't work on your system, fix it -- that's what source code is
 * for. :) */

#include <sys/stat.h>
#include <sys/vfs.h>
#include <mntent.h>
#include <epplet.h>

#define MOUNTCMD "/bin/mount"
#define UMOUNTCMD "/bin/umount"

// structures
typedef struct listitem
{
	char *mountpoint;
	struct listitem *prev;
	struct listitem *next;
} listitem;

// global variables
Epplet_gadget mountpointlabel;
Epplet_gadget mountbutton;
Epplet_gadget capacitybar;
int mountbuttonvalue = 0;
int capacitybarvalue = 50;

// functions
listitem *readfstab (char *fstab_file)
{
	FILE *fstab;
	char fstabline[256];
	char mountpoint[256];
	char options[256];
	listitem *currentlistitem = NULL;
	listitem *firstlistitem = NULL;
	int i, j;

	fstab = fopen (fstab_file, "r");
	if (fstab == NULL)
	{
		fprintf (stderr, "error opening fstab file %s -- aborting\n", fstab_file);
		exit (1);
	}
	// read each line of the fstab file
	while (fgets (fstabline, 256, fstab))
	{
		// move to the first argument
		for (i = 0; fstabline[i] == ' ' || fstabline[i] == '\t'; i++);
		// skip comments
		if (fstabline[i] != '#' && fstabline[i] != '\n')
		{
			// skip the first argument
			for (; fstabline[i] != ' ' && fstabline[i] != '\t'; i++);
			for (; fstabline[i] == ' ' || fstabline[i] == '\t'; i++);
			// read the second argument (mount point)
			for (j = 0; fstabline[i] != ' ' && fstabline[i] != '\t' &&
				j <= 254; i++)
			{
				mountpoint[j] = fstabline[i];
				j++;
			}
			mountpoint[j] = '\0';
			// if we've reached the end of our buffer, we need to skip the
			// rest of the argument
			if (j == 255)
				for (; fstabline[i] != ' ' && fstabline[i] != '\t'; i++);
			// skip the third argument
			for (; fstabline[i] == ' ' || fstabline[i] == '\t'; i++);
			for (; fstabline[i] != ' ' && fstabline[i] != '\t'; i++);
			for (; fstabline[i] == ' ' || fstabline[i] == '\t'; i++);
			// read the fourth argument (options)
			for (j = 0; fstabline[i] != ' ' && fstabline[i] != '\t' && j <= 254;
				i++)
			{
				options[j] = fstabline[i];
				j++;
			}
			// if the mount point is user-mountable, add it to the list
			if (strstr (options, "user"))
			{
				if (currentlistitem != NULL)
				{
					currentlistitem->next = malloc (sizeof (listitem));
					currentlistitem->next->prev = currentlistitem;
					currentlistitem = currentlistitem->next;
				}
				else
					currentlistitem = firstlistitem =
						malloc (sizeof (listitem));

				currentlistitem->mountpoint = strdup (mountpoint);
			}
		}
	}
	fclose (fstab);

	if (firstlistitem == NULL)
	{
		fprintf (stderr, "no mounts containing the \"user\" option found in %s -- aborting\n", fstab_file);
		exit (1);
	}

	firstlistitem->prev = currentlistitem;
	currentlistitem->next = firstlistitem;
	return (firstlistitem);
}

void updatefsinfo (char *mountpoint, int changelabel)
{
	dev_t dev1, dev2;
	struct stat st;
	struct statfs sfs;
	int usage, mounted;
	char *buf;

	mounted = 0;
	buf = strdup (mountpoint);
	stat (buf, &st);
	dev1 = st.st_dev;
	if (buf[strlen(buf)-1]=='/')
		buf[strlen(buf)-1] = '\0';
	*(strrchr(buf, '/')+1)='\0';
	stat (buf, &st);
	dev2 = st.st_dev;
	if (memcmp(&dev1, &dev2, sizeof(dev_t)) != 0)
		mounted = 1;
	free (buf);
	
	statfs (mountpoint, &sfs);
	usage = (sfs.f_blocks - sfs.f_bavail) * 100 / sfs.f_blocks;

	if (changelabel)
		Epplet_change_label (mountpointlabel, mountpoint);

	if (mounted)
		capacitybarvalue = usage;
	else
		capacitybarvalue = 0;
	mountbuttonvalue = mounted;

	Epplet_gadget_data_changed (mountbutton);
	Epplet_gadget_data_changed (capacitybar);
}
	
void moveprevpoint (void *data)
{
	listitem **pointertocurrentlistitem = data;

	*pointertocurrentlistitem = (*pointertocurrentlistitem)->prev;
	updatefsinfo ((*pointertocurrentlistitem)->mountpoint, 1);

}

void movenextpoint (void *data)
{
	listitem **pointertocurrentlistitem = data;

	*pointertocurrentlistitem = (*pointertocurrentlistitem)->next;
	updatefsinfo ((*pointertocurrentlistitem)->mountpoint, 1);

}

void closeclick (void *data)
{
	Epplet_unremember ();
	Esync ();

	exit (0);
}

void togglemount (void *data)
{
	listitem **pointertocurrentlistitem = data;
	char *fullstring;
	int returncode;
	int changelabel = 1;

	if (mountbuttonvalue)
	{
		fullstring = calloc (strlen (MOUNTCMD) +
			strlen ((*pointertocurrentlistitem)->mountpoint) + 2,
			sizeof (char));
		strcpy (fullstring, MOUNTCMD);
		strcat ((strcat (fullstring, " ")),
			(*pointertocurrentlistitem)->mountpoint);
	}
	else
	{
		fullstring = calloc (strlen (UMOUNTCMD) +
			strlen ((*pointertocurrentlistitem)->mountpoint) + 2,
			sizeof (char));
		strcpy (fullstring, UMOUNTCMD);
		strcat ((strcat (fullstring, " ")),
			(*pointertocurrentlistitem)->mountpoint);
	}

	returncode = system (fullstring);
	printf ("%d\n", returncode);

	switch (returncode)
	{
		case 127:
			Epplet_change_label (mountpointlabel, "shell error");
			changelabel = 0;
			break;
		case -1:
			Epplet_change_label (mountpointlabel, "misc error");
			changelabel = 0;
			break;
		case 8192:
			Epplet_change_label (mountpointlabel, "mount error");
			changelabel = 0;
			break;
		case 512:
			Epplet_change_label (mountpointlabel, "unmount error");
			changelabel = 0;
			break;
		default:
	}

	updatefsinfo ((*pointertocurrentlistitem)->mountpoint, changelabel);
}

int main (int argc, char **argv)
{
	Epplet_gadget leftarrow;
	Epplet_gadget rightarrow;
	Epplet_gadget closebutton;
	int width = 4;
	char *fstab_file = NULL;
	int i = 1;
	listitem *firstlistitem = NULL;
	listitem *currentlistitem = NULL;;

	while (i < argc)
	{
		if (argv[i][0] == '-')
		{
			switch (argv[i][1])
			{
				case 'f':
					if (fstab_file == NULL)
					{
						if (i+1 < argc)
							fstab_file = strdup (argv[i+1]);
						else
							fprintf (stderr, "missing fstab file ignored -- using /etc/fstab\n");
					}
					else
						fprintf (stderr, "redefinition of fstab file ignored -- using %s\n", fstab_file);
					i += 2;
					break;
				case 'w':
					width = atoi (argv[i+1]);
					if (width < 4)
					{
						fprintf (stderr, "invalid width of %d -- using 4\n",
							width);
						width = 4;
					}
					i += 2;
					break;
				case 'h':
					printf ("usage: E-Mount.epplet -f <file> -w <width> -h\n");
					printf ("  -f         defines fstab file [default = /etc/fstab]\n");
					printf ("  -w         defines width of window, in units of 16 pixels [default = 4]\n");
					printf ("  -h         displays this help screen and exits\n");
					exit (0);
					break;
				default:
					fprintf (stderr, "invalid option %c -- use -h for help\n",
						argv[i][1]);
					exit (1);
			}
		}
		else
			i++;

	}

	if (fstab_file == NULL)
		fstab_file = strdup ("/etc/fstab");

	// read the fstab file
	firstlistitem = readfstab (fstab_file);
	free (fstab_file);
	fstab_file = NULL;
	
	currentlistitem = firstlistitem;

	atexit (Epplet_cleanup);
	Epplet_Init ("E-Mount", "0.3", "Enlightenment Drive-Mounting Epplet",
		width, 2, argc, argv, 0);

	Epplet_gadget_show (mountpointlabel = Epplet_create_label
		(2, 2, currentlistitem->mountpoint, 0));
	Epplet_gadget_show (capacitybar = Epplet_create_hbar (2, 11,
		width * 16 - 17, 7, 0, &capacitybarvalue));
	Epplet_gadget_show (closebutton = Epplet_create_button (NULL, NULL,
		width * 16 - 14, 2, 0, 0, "CLOSE", 0, NULL, closeclick, NULL));
	Epplet_gadget_show (leftarrow = Epplet_create_button (NULL, NULL,
		2, 18, 0, 0, "ARROW_LEFT", 0, NULL, moveprevpoint,
		&currentlistitem));
	Epplet_gadget_show (rightarrow = Epplet_create_button (NULL, NULL,
		width * 16 - 14, 18, 0, 0, "ARROW_RIGHT", 0, NULL, movenextpoint,
		&currentlistitem));
	Epplet_gadget_show (mountbutton = Epplet_create_togglebutton("Mount",
		NULL, 14, 18, width * 16 - 29, 12, &mountbuttonvalue, togglemount,
		&currentlistitem));
	
	// update the info for the current mount point
	updatefsinfo (currentlistitem->mountpoint, 1);
	
	Epplet_show ();
	Epplet_Loop ();

	return 0;
}
