/*****************************************************************************/
/* td_timeruler.c : Objet Gtk+
 * td_timeruler.c : Gtk+ object
 *
 *
 * ToutDoux : Chtit gestionnaire de projet - A littl' project manager
 * Copyright (c) 2001 Philippe Roy
 * Auteur - Author : Philippe Roy <ph_roy@toutdoux.org>
 *
 *
 * Ce programme est un logiciel libre ; vous pouvez le redistribuer et/ou le modifier
 * sous les termes de la licence publique gnrale GNU telle qu'elle est publie par
 * la Free Software Foundation ; soit la version 2 de la licence, ou
 * (comme vous voulez) toute version ultrieure.
 *
 * Ce programme est distribu dans l'espoir qu'il sera utile,
 * mais SANS AUCUNE GARANTIE ; mme sans la garantie de
 * COMMERCIALIT ou d'ADQUATION A UN BUT PARTICULIER. Voir la
 * licence publique gnrale GNU pour plus de dtails.
 *
 * Vous devriez avoir reu une copie de la licence publique gnrale GNU
 * avec ce programme ; si ce n'est pas le cas, crivez  la Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. */
/*****************************************************************************/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>
#include <math.h>

#include "commons.h"
#include "gtdk.h"

#include "td_app.h"
#include "td_app_palette.h"
#include "td_timeruler.h"
#include "td_mod_gantt.h"

/*****************************************************************************/
/*** Arguments */
/*****************************************************************************/
static GtkObjectClass *parent_class = NULL;

/*****************************************************************************/
/*** Initialisation */
/*****************************************************************************/
static void td_timeruler_init (TdTimeruler *timeruler)
{
  timeruler->widget = NULL;
  timeruler->canvas = NULL;
  timeruler->lower = NULL;
  timeruler->upper = NULL;
  timeruler->width = 0;
  timeruler->zoom_label = NULL;
}

static void td_timeruler_class_init (TdTimerulerClass *klass)
{
  GtkObjectClass *object_class;
  object_class = (GtkObjectClass*) klass;
  parent_class = gtk_type_class (gtk_object_get_type());
  klass->set_lower = td_timeruler_set_lower;
}

GtkType td_timeruler_get_type (void)
{
  static GtkType timeruler_type = 0;
  if (!timeruler_type)
    {
      static const GtkTypeInfo timeruler_info =
      {
  	"TdTimeruler", sizeof (TdTimeruler), sizeof (TdTimerulerClass),
  	(GtkClassInitFunc) td_timeruler_class_init,
	(GtkObjectInitFunc) td_timeruler_init,
	NULL, NULL, (GtkClassInitFunc) NULL,
      };
      timeruler_type = gtk_type_unique (GTK_TYPE_OBJECT, &timeruler_info);
    }
  return timeruler_type;
}

/**
 * td_timeruler_new:
 * 
 * fr: Cr une nouvelle rgle du temps
 *
 * en: Creates a new time ruler
 * 
 * Return value: time ruler
 **/

GtkObject *td_timeruler_new (void)
{
  return GTK_OBJECT (gtk_type_new (td_timeruler_get_type()));
}

/**
 * td_timeruler_destroy:
 * @timeruler: time ruler
 * 
 * fr: Dtruit la rgle du temps
 *
 * en: Destroys the time ruler
 **/

void td_timeruler_destroy (TdTimeruler *timeruler)
{
  g_return_if_fail (timeruler != NULL);
  g_return_if_fail (TD_IS_TIMERULER (timeruler));
  gtk_object_destroy (GTK_OBJECT (timeruler));
  g_date_free (timeruler->lower);
  g_date_free (timeruler->upper);
  if (WINDOW_DATE_SELECTION)
    {
      gtk_widget_destroy (WINDOW_DATE_SELECTION);
      WINDOW_DATE_SELECTION = NULL;
    }
}

/**
 * td_timeruler_hide:
 * @timeruler: time ruler
 * 
 * fr: Cache la rgle du temps
 *
 * en: Hides the time ruler
 **/

void td_timeruler_hide (TdTimeruler *timeruler)
{
  g_return_if_fail (timeruler != NULL);
  g_return_if_fail (TD_IS_TIMERULER (timeruler));
  if (WINDOW_DATE_SELECTION)
    {
      gtk_widget_destroy (WINDOW_DATE_SELECTION);
      WINDOW_DATE_SELECTION = NULL;
    }
}

/*****************************************************************************/
/*** Affectations - Allocations */
/*****************************************************************************/

/**
 * td_timeruler_set_canvas:
 * @timeruler: time ruler
 * @canvas: canvas
 * 
 * fr: Affecte le cavenas  la rgle du temps
 *
 * en: Allocates the canvas to time ruler
 **/

void td_timeruler_set_canvas (TdTimeruler *timeruler, GtkWidget *canvas)
{
  g_return_if_fail (timeruler != NULL);
  g_return_if_fail (TD_IS_TIMERULER (timeruler));
  if (timeruler->canvas != canvas)
    timeruler->canvas = canvas;
}

/**
 * td_timeruler_set_lower:
 * @timeruler: time ruler
 * @lower: lower date
 * 
 * fr: Affecte la date basse  la rgle du temps
 *
 * en: Allocates the lower date to time ruler
 **/

void td_timeruler_set_lower (TdTimeruler *timeruler, GDate *lower)
{
  g_return_if_fail (timeruler != NULL);
  g_return_if_fail (TD_IS_TIMERULER (timeruler));
  if (timeruler->lower != lower)
    timeruler->lower = lower;
}

/*****************************************************************************/
/*** Commandes - Commands */
/*****************************************************************************/

/**
 * td_timeruler_create:
 * @timeruler: time ruler
 * @vbox1: container of slider
 * @vbox2: container of calendar
 * 
 * fr: Cr l'interface de la rgle du temps
 *
 * en: Creates the interface of time ruler
 **/

void td_timeruler_create (TdTimeruler *timeruler, GtkWidget *vbox1, GtkWidget *vbox2)
{
  GdkWindow *window = GTK_WIDGET (TD_APP_PALETTE (TD_APP (APP)->palette)->window)->window;
  GtkAdjustment *adj;
  GtkWidget *hscale;
  GtkWidget *hbox1;
  GtkWidget *frame;
  GtkWidget *calendar_arrow;
  GtkWidget *calendar_button;
  g_return_if_fail (timeruler != NULL);
  g_return_if_fail (TD_IS_TIMERULER (timeruler));

  /*** Zoom */
  hbox1 = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (hbox1);
  gtk_box_pack_start (GTK_BOX (vbox1), hbox1, FALSE, FALSE, 0);
  adj = GTK_ADJUSTMENT (gtk_adjustment_new (13, 1, 19, 1, 1, 0));
  hscale = gtk_hscale_new (GTK_ADJUSTMENT (adj));
  gtk_scale_set_draw_value (GTK_SCALE (hscale), FALSE);
  gtk_widget_show (hscale);
  gtk_box_pack_start (GTK_BOX (hbox1), hscale, TRUE, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT (adj), "value_changed", GTK_SIGNAL_FUNC (td_timeruler_zoom_changed), timeruler);
  timeruler->zoom_label = gtk_label_new (g_strdup_printf ("%d %s ", 1, _("day")));
  gtk_widget_show (timeruler->zoom_label);
  gtk_widget_set_usize (timeruler->zoom_label, td_timeruler_scale_label_maxsize(), 18);
  gtk_box_pack_start (GTK_BOX (hbox1), timeruler->zoom_label, FALSE, FALSE, 5);

  /*** Rgle - Ruler */
  frame = gtk_frame_new (NULL);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_OUT);
  gtk_widget_show (frame);
  gtk_container_add (GTK_CONTAINER (vbox1), frame);
  timeruler->widget = gnome_canvas_new();
  gtk_widget_show (timeruler->widget);
  gtk_container_add (GTK_CONTAINER (frame), timeruler->widget);
  gnome_canvas_set_scroll_region (GNOME_CANVAS (timeruler->widget), 0, 0, 1000, 1000); /* FIXME: personalisable : +tards */
  td_timeruler_set_scale (timeruler, 13);

  /*** Calendrier - Calendar */
  calendar_button = gtk_button_new();
  gtk_button_set_relief (GTK_BUTTON (calendar_button), GTK_RELIEF_NONE);
  gtk_widget_show (calendar_button);
  gtk_widget_set_usize (calendar_button, 18, 18);
  gtk_box_pack_start (GTK_BOX (vbox2), calendar_button, FALSE, FALSE, 0);
  gtk_signal_connect (GTK_OBJECT (calendar_button), "button_press_event", GTK_SIGNAL_FUNC (td_timeruler_calendar_button_press), timeruler);
  calendar_arrow = gtk_arrow_new (GTK_ARROW_DOWN, GTK_SHADOW_IN);
  gtk_widget_show (calendar_arrow);
  gtk_container_add (GTK_CONTAINER (calendar_button), calendar_arrow);
}

/**
 * td_timeruler_zoom_changed:
 * @adj: adjustement of zoom of timeruler
 * @timeruler: time ruler
 * 
 * fr: Evnement 'value_changed' du zoom de la rgle du temps
 *
 * en: Event 'value_changed' of zoom of time ruler
 **/

void td_timeruler_zoom_changed (GtkAdjustment *adj, TdTimeruler *timeruler)
{
  int x1, x2, i, j, num_points, line;
  float width_init, width;
  GnomeCanvasGroup *root;
  GnomeCanvasPoints *points;
  GnomeCanvasGroup *item;
  GnomeCanvasGroup *item2;
  GList *list_tmp;
  int row_height = gdk_string_height (gtk_widget_get_default_style()->font, "0");
  root = gnome_canvas_root (GNOME_CANVAS (timeruler->canvas));

  /*** Rgle - Ruler */
  width_init = timeruler->width;
  td_timeruler_set_scale (timeruler, adj->value);
  width = timeruler->width;
  if (width == width_init)
    return;

  /*** Item */
  for (i=0; i<g_list_length (root->item_list); i++)
    {

      /*** Node */
      if (GNOME_IS_CANVAS_GROUP (g_list_nth_data (root->item_list, i)))
	{
	  list_tmp = gtdk_canvas_item_childs (GNOME_CANVAS_ITEM (g_list_nth_data (root->item_list, i)));
	  for (j=0; j<g_list_length (list_tmp); j++)
	    if ((GNOME_IS_CANVAS_GROUP (g_list_nth_data (list_tmp, j))) && (!strstr ((gchar*) gtk_object_get_user_data (GTK_OBJECT (g_list_nth_data (list_tmp, j))), "handle")))
	      {
		item = g_list_nth_data (list_tmp, j);
		x1 = ceil ((gtdk_canvas_i2w_x (GNOME_CANVAS_ITEM (item), 0)-gtdk_canvas_i2w_x (GNOME_CANVAS_ITEM (item)->parent, 0))/width_init);
		x2 = ceil ((GNOME_CANVAS_RE (g_list_nth_data (item->item_list, 0))->x2)/width_init);
		gnome_canvas_item_move (GNOME_CANVAS_ITEM (item), -(width_init*x1)+(width*x1), 0);
		gnome_canvas_item_set (g_list_nth_data (item->item_list, 0), "x2", width*x2, NULL);
	      }
	  continue;
	}

      /*** Lien - Link */
      if (GNOME_IS_CANVAS_LINE (g_list_nth_data (root->item_list, i)))
	{
	  if (!strcmp ((gchar*) gtk_object_get_user_data (GTK_OBJECT (g_list_nth_data (root->item_list, i))), "link"))
	    continue;
	  num_points = GNOME_CANVAS_LINE (g_list_nth_data (root->item_list, i))->num_points;
	  points = gnome_canvas_points_new (num_points);
	  for (j=0; j<num_points*2; j=j+2)
	    {
	      points->coords[j] = GNOME_CANVAS_LINE (g_list_nth_data (root->item_list, i))->coords[j]*(width/width_init);
  	      points->coords[j+1] = GNOME_CANVAS_LINE (g_list_nth_data (root->item_list, i))->coords[j+1];
	    }
	  gnome_canvas_item_set (g_list_nth_data (root->item_list, i), "points", points, NULL);
	  gnome_canvas_points_free (points);
	  td_mod_gantt_link_correct (GNOME_CANVAS_LINE (g_list_nth_data (root->item_list, i)));
	}
    }
}

/**
 * td_timeruler_calendar_button_press:
 * @widget: button clicked for calendar creation
 * @event: expose event
 * @timeruler: time ruler
 * 
 * fr: Cr l'interface du calendier de la rgle du temps
 *
 * en: Creates the interface of calendar of time ruler
 **/

void td_timeruler_calendar_button_press (GtkWidget *widget, GdkEventButton *event, TdTimeruler *timeruler)
{
  GtkWidget *window;
  GtkWidget *calendar;
  if (WINDOW_DATE_SELECTION)
    {
      gtk_widget_destroy (WINDOW_DATE_SELECTION);
      WINDOW_DATE_SELECTION = NULL;
    }
  WINDOW_DATE_SELECTION = gtk_window_new (GTK_WINDOW_POPUP);
  gtk_window_set_policy (GTK_WINDOW (WINDOW_DATE_SELECTION), FALSE, FALSE, TRUE);
  gtk_window_set_position (GTK_WINDOW (WINDOW_DATE_SELECTION), GTK_WIN_POS_MOUSE);
  calendar = gtk_calendar_new();
  gtk_widget_show (calendar);
  gtk_container_add (GTK_CONTAINER (WINDOW_DATE_SELECTION), calendar);
  gtk_calendar_display_options (GTK_CALENDAR (calendar), GTK_CALENDAR_SHOW_HEADING | GTK_CALENDAR_SHOW_DAY_NAMES | GTK_CALENDAR_SHOW_WEEK_NUMBERS | GTK_CALENDAR_WEEK_START_MONDAY);
  gtdk_calendar_select_date (GTK_CALENDAR (calendar), timeruler->lower);
  gtk_widget_show (WINDOW_DATE_SELECTION);
  gtk_signal_connect (GTK_OBJECT (calendar), "day_selected", GTK_SIGNAL_FUNC (td_timeruler_calendar_day_selected), timeruler);
}

/**
 * td_timeruler_calendar_day_selected:
 * @calendar: calendar widget
 * @timeruler: time ruler
 * 
 * fr: Evnement 'day_selected' du calendrier de la rgle du temps
 *
 * en: Event 'day_selected' of calendar of time ruler
 **/

void td_timeruler_calendar_day_selected (GtkCalendar *calendar, TdTimeruler *timeruler)
{
  int x1, x2, i;
  float width_init, width;
  GDate *lower_init;
  GnomeCanvasGroup *root;
  gchar *txt_tmp;
  gchar *txt_tmp2;
  root = gnome_canvas_root (GNOME_CANVAS (timeruler->canvas));

  /*** Go ! */
  txt_tmp = td_date_print (timeruler->lower, "year-month-day");
  txt_tmp2 = td_date_print (gtdk_calendar_get_date (calendar), "year-month-day");
  if (strcmp (txt_tmp, txt_tmp2))
    {
      /*** Rgle - Ruler */
      width_init = timeruler->width;
      lower_init = g_date_new_dmy (g_date_day (timeruler->lower), g_date_month (timeruler->lower), g_date_year (timeruler->lower));
      td_timeruler_set_lower (timeruler, gtdk_calendar_get_date (calendar));
      td_timeruler_set_scale (timeruler, timeruler->scale);
      width = timeruler->width;

      /*** Items */
      for (i=0; i<g_list_length (root->item_list); i++)
	{
	  x1 = ceil (gtdk_canvas_i2w_x (g_list_nth_data (root->item_list, i), 0)/width_init);
	  x2 = ceil ((GNOME_CANVAS_RE (g_list_nth_data (GNOME_CANVAS_GROUP (g_list_nth_data (root->item_list, i))->item_list, 0))->x2)/width_init);
  	  gnome_canvas_item_move (g_list_nth_data (root->item_list, i), -(width_init*x1)+(width*x1), 0);
	  if (td_date_period_day (timeruler->lower, lower_init) == 0)
	    gnome_canvas_item_move (g_list_nth_data (root->item_list, i), (-width*td_date_period_day (lower_init, timeruler->lower)), 0);
	  else
	    gnome_canvas_item_move (g_list_nth_data (root->item_list, i), (width*td_date_period_day (timeruler->lower, lower_init)), 0);
	  gnome_canvas_item_set (g_list_nth_data (GNOME_CANVAS_GROUP (g_list_nth_data (root->item_list, i))->item_list, 0), "x2", width*x2, NULL);
	}
      g_date_free (lower_init);
    }
  else
    if (WINDOW_DATE_SELECTION)
      {
	gtk_widget_destroy (WINDOW_DATE_SELECTION);
	WINDOW_DATE_SELECTION = NULL;
      }
}

/*****************************************************************************/
/*** Dessin - Drawing */
/*****************************************************************************/

/**
 * td_timeruler_draw_line:
 * @x1: x1 coordinate
 * @y1: y1 coordinate
 * @x2: x2 coordinate
 * @y2: y2 coordinate
 * @timeruler: timeruler
 * 
 * fr: Dessine une ligne
 *
 * en: Draws a line
 **/

void td_timeruler_draw_line (float x1, float y1, float x2, float y2, TdTimeruler *timeruler)
{
  GnomeCanvasPoints *points = NULL;
  points = gnome_canvas_points_new (2);
  points->coords[0] = x1+1;
  points->coords[1] = y1-2;
  points->coords[2] = x2+1;
  points->coords[3] = y2-2;
  gnome_canvas_item_new (gnome_canvas_root (GNOME_CANVAS (timeruler->widget)), gnome_canvas_line_get_type(),
			 "points", points, 
			 "fill_color_gdk", &TD_FACE_TIMERULER_LINE_COLOR,
			 "width_units", TD_FACE_TIMERULER_LINE_WIDTH, NULL);
  gnome_canvas_points_free (points);
}

/**
 * td_timeruler_draw_string:
 * @x: x coordinate
 * @y: y coordinate
 * @string: string
 * @timeruler: timeruler
 * 
 * fr: Dessine une tiquette
 *
 * en: Draws a label
 **/

void td_timeruler_draw_string (float x, float y, gchar *string, TdTimeruler *timeruler)
{
  gnome_canvas_item_new (gnome_canvas_root (GNOME_CANVAS (timeruler->widget)), gnome_canvas_text_get_type(),
			 "text", string,
			 "x", (double) x+1,
			 "y", (double) y+1-2,
			 "font_gdk", gtk_widget_get_default_style()->font,
			 "justification", GTK_JUSTIFY_LEFT,
			 "anchor", GTK_ANCHOR_NORTH_WEST,
			 "fill_color_gdk", &TD_FACE_TIMERULER_TEXT_COLOR, NULL);
}

/**
 * td_timeruler_draw:
 * @timeruler: timeruler
 * 
 * fr: Dessine la rgle du temps
 *
 * en: Draws the timeruler
 **/

void td_timeruler_draw (TdTimeruler *timeruler)
{
  int days;
  float y = 2;
  gchar *tmp_format;
  float height = gdk_char_height (gtk_widget_get_default_style()->font, 'O')+7;
  GtkWidget *widget = timeruler->widget;
  GtkWidget *widget2;
  gtdk_canvas_clear (GNOME_CANVAS (timeruler->widget));

  /*** Anne - Year */
  tmp_format = td_timeruler_year_optimal_format (timeruler);
  if (tmp_format)
    {
      td_timeruler_year_draw (timeruler, y, y+height);
      y=y+height;
      td_timeruler_draw_line (0, y, widget->allocation.width-1, y, timeruler);
    }

  /*** Mois - Month */
  tmp_format = td_timeruler_month_optimal_format (timeruler);
  if (tmp_format)
    {
      td_timeruler_month_draw (timeruler, y, y+height, tmp_format);
      y=y+height;
      td_timeruler_draw_line (0, y, widget->allocation.width-1, y, timeruler);
    }

  /*** Semaine - Week */
  tmp_format = td_timeruler_week_optimal_format (timeruler);
  if (tmp_format)
    {
      td_timeruler_week_draw (timeruler, y, y+height, tmp_format);
      y=y+height;
      td_timeruler_draw_line (0, y, widget->allocation.width-1, y, timeruler);
    }

  /*** Jour - Day */
  tmp_format = td_timeruler_day_optimal_format (timeruler);
  if (tmp_format)
    {
      td_timeruler_day_draw (timeruler, y, y+height, tmp_format);
      y=y+height;
      td_timeruler_draw_line (0, y, widget->allocation.width-1, y, timeruler);
    }

  /*** Cadre - Frame */
  gtk_widget_set_usize (timeruler->widget, -1, y+height);
  widget2 = timeruler->widget->parent->parent;
  gtk_widget_set_usize (widget2, -1, y+height+4);
}

/**
 * td_timeruler_year_optimal_format:
 * @timeruler: time ruler
 * 
 * fr: Retourne le format optimal pour l'affichage des annes 
 *
 * en: Return the optimal format for displaying year
 * 
 * Return value: string
 **/

gchar *td_timeruler_year_optimal_format (TdTimeruler *timeruler)
{
  /*** 28 jours est la priode minimale d'un mois - 28 days is the minimal period of a month */
  if ((timeruler->width*28)>gdk_string_width (gtk_widget_get_default_style()->font, "00/0000"))
    return NULL;
  return "year";
}

/**
 * td_timeruler_year_draw:
 * @timeruler: time ruler
 * @y0: y0 coordinate
 * @y1: y1 coordinate
 * 
 * fr: Affiche les annes sur la rgle du temps 
 *
 * en: Displays years on time ruler
 **/

void td_timeruler_year_draw (TdTimeruler *timeruler, int y0, int y1)
{
  GnomeCanvasItem *label_under;
  GdkWindow *window = timeruler->widget->window;
  int day;
  int i, j;
  g_return_if_fail (timeruler != NULL);
  g_return_if_fail (TD_IS_TIMERULER (timeruler));

  /*** Go ! */
  day = 0;
  for (i=g_date_year (timeruler->lower); i<g_date_year (timeruler->upper)+1; i++)
    {
      label_under = gnome_canvas_get_item_at (GNOME_CANVAS (timeruler->widget), timeruler->width*day, y0);
      if (label_under)
	gtk_object_destroy (GTK_OBJECT (label_under));
      td_timeruler_draw_line (timeruler->width*day, y0, timeruler->width*day, y1, timeruler);
      td_timeruler_draw_string (timeruler->width*day+3, y0, g_strdup_printf ("%d", i), timeruler);

      /*** Premire et dernire anne - First and last year */
      if ((i==g_date_year (timeruler->lower)) && (i==g_date_year (timeruler->upper)))
	for (j=g_date_month (timeruler->lower); j<g_date_month (timeruler->upper)+1; j++)
	  {
	    if ((j==g_date_month (timeruler->lower)) && (j==g_date_month (timeruler->upper))) /*** Premier et dernier mois - First and last month */
	      day = day + g_date_day (timeruler->upper) - (g_date_day (timeruler->lower)-1);
	    if ((j==g_date_month (timeruler->lower)) && (j!=g_date_month (timeruler->upper))) /*** Premier mois - First month */
	      day = day + g_date_days_in_month (j, i) - (g_date_day (timeruler->lower)-1);
	    if ((j!=g_date_month (timeruler->lower)) && (j==g_date_month (timeruler->upper))) /*** Dernier mois - Last month */
	      day = day + g_date_day (timeruler->upper);
	    if ((j!=g_date_month (timeruler->lower)) && (j!=g_date_month (timeruler->upper)))
	      day = day + g_date_days_in_month (j, i);
	  }

      /*** Premire anne - First year */
      if ((i==g_date_year (timeruler->lower)) && (i!=g_date_year (timeruler->upper)))
	for (j=g_date_month (timeruler->lower); j<13; j++)
	  {
	    if (j==g_date_month (timeruler->lower)) /*** Premier mois - First month */
	      day = day + g_date_days_in_month (j, i) - (g_date_day (timeruler->lower)-1);
	    else
	      day = day + g_date_days_in_month (j, i);
	  }

      /*** Dernire anne - Last year */
      if ((i!=g_date_year (timeruler->lower)) && (i==g_date_year (timeruler->upper)))
	for (j=1; j<g_date_month (timeruler->upper)+1; j++)
	  {
	    if (j==g_date_month (timeruler->upper)) /*** Dernier mois - Last month */
	      day = day + g_date_day (timeruler->upper);
	    else
	      day = day + g_date_days_in_month (j, i);
	  }

      /*** Anne intermdiaire - Intermediate year */
      if ((i!=g_date_year (timeruler->lower)) && (i!=g_date_year (timeruler->upper)))
	for (j=1; j<13; j++)
	  day = day + g_date_days_in_month (j, i);
    }
}

/**
 * td_timeruler_month_optimal_format:
 * @timeruler: time ruler
 * 
 * fr: Retourne le format optimal pour l'affichage des mois
 *
 * en: Return the optimal format for displaying months
 * 
 * Return value: string
 **/

gchar *td_timeruler_month_optimal_format (TdTimeruler *timeruler)
{
  if (((timeruler->width)>gdk_string_width (gtk_widget_get_default_style()->font, "00/00/0000 ")) ||
      ((timeruler->width*28)<gdk_string_width (gtk_widget_get_default_style()->font, "00 ")))
    return NULL;
  if ((timeruler->width*28)>gdk_string_width (gtk_widget_get_default_style()->font, "00/0000 "))
    return td_date_intl ("month/year");
  return "month";
}

/**
 * td_timeruler_month_draw:
 * @timeruler: time ruler
 * @y0: y0 coordinate
 * @y1: y1 coordinate
 * @format: date format
 * 
 * fr: Affiche les mois sur la rgle du temps 
 *
 * en: Displays months on time ruler
 **/

void td_timeruler_month_draw (TdTimeruler *timeruler, int y0, int y1, gchar *format)
{
  GnomeCanvasItem *label_under;
  GdkWindow *window = timeruler->widget->window;
  int day;
  int i, j;
  g_return_if_fail (timeruler != NULL);
  g_return_if_fail (TD_IS_TIMERULER (timeruler));
  if (!format)
    format = "month";

  /*** Go ! */
  day = 0;
  for (i=g_date_year (timeruler->lower); i<g_date_year (timeruler->upper)+1; i++)
    {

      /*** Premire et dernire anne - First and last year */
      if ((i==g_date_year (timeruler->lower)) && (i==g_date_year (timeruler->upper)))
	for (j=g_date_month (timeruler->lower); j<g_date_month (timeruler->upper)+1; j++)
	  {
	    label_under = gnome_canvas_get_item_at (GNOME_CANVAS (timeruler->widget), timeruler->width*day, y0);
	    if (label_under)
	      gtk_object_destroy (GTK_OBJECT (label_under));
	    td_timeruler_draw_line (timeruler->width*day, y0, timeruler->width*day, y1, timeruler);
	    td_timeruler_draw_string (timeruler->width*day+3, y0, td_date_month (j, i, format), timeruler);
	    if ((j==g_date_month (timeruler->lower)) && (j==g_date_month (timeruler->upper))) /*** Premier et dernier mois - First and last month */
	      day = day + g_date_day (timeruler->upper) - (g_date_day (timeruler->lower)-1);
	    if ((j==g_date_month (timeruler->lower)) && (j!=g_date_month (timeruler->upper))) /*** Premier mois - First month */
	      day = day + g_date_days_in_month (j, i) - (g_date_day (timeruler->lower)-1);
	    if ((j!=g_date_month (timeruler->lower)) && (j==g_date_month (timeruler->upper))) /*** Dernier mois - Last month */
	      day = day + g_date_day (timeruler->upper);
	    if ((j!=g_date_month (timeruler->lower)) && (j!=g_date_month (timeruler->upper)))
	      day = day + g_date_days_in_month (j, i);
	  }

      /*** Premire anne - Fisrt year */
      if ((i==g_date_year (timeruler->lower)) && (i!=g_date_year (timeruler->upper)))
	for (j=g_date_month (timeruler->lower); j<13; j++)
	  {
	    label_under = gnome_canvas_get_item_at (GNOME_CANVAS (timeruler->widget), timeruler->width*day, y0);
	    if (label_under)
	      gtk_object_destroy (GTK_OBJECT (label_under));
	    td_timeruler_draw_line (timeruler->width*day, y0, timeruler->width*day, y1, timeruler);
	    td_timeruler_draw_string (timeruler->width*day+3, y0, td_date_month (j, i, format), timeruler);
	    if (j==g_date_month (timeruler->lower)) /*** Premier mois - First month */
	      day = day + g_date_days_in_month (j, i) - (g_date_day (timeruler->lower)-1);
	    else
	      day = day + g_date_days_in_month (j, i);
	  }

      /*** Dernire anne - Last year */
      if ((i!=g_date_year (timeruler->lower)) && (i==g_date_year (timeruler->upper)))
	for (j=1; j<g_date_month (timeruler->upper)+1; j++)
	  {
	    label_under = gnome_canvas_get_item_at (GNOME_CANVAS (timeruler->widget), timeruler->width*day, y0);
	    if (label_under)
	      gtk_object_destroy (GTK_OBJECT (label_under));
	    td_timeruler_draw_line (timeruler->width*day, y0, timeruler->width*day, y1, timeruler);
	    td_timeruler_draw_string (timeruler->width*day+3, y0, td_date_month (j, i, format), timeruler);
	    if (j==g_date_month (timeruler->upper)) /*** Dernier mois - Last month */
	      day = day + g_date_day (timeruler->upper);
	    else
	      day = day + g_date_days_in_month (j, i);
	  }

      /*** Anne intermdiaire - Intermediate year */
      if ((i!=g_date_year (timeruler->lower)) && (i!=g_date_year (timeruler->upper)))
	for (j=1; j<13; j++)
	  {
	    label_under = gnome_canvas_get_item_at (GNOME_CANVAS (timeruler->widget), timeruler->width*day, y0);
	    if (label_under)
	      gtk_object_destroy (GTK_OBJECT (label_under));
	    td_timeruler_draw_line (timeruler->width*day, y0, timeruler->width*day, y1, timeruler);
	    td_timeruler_draw_string (timeruler->width*day+3, y0, td_date_month (j, i, format), timeruler);
	    day = day + g_date_days_in_month (j, i);
	  }
    }
}

/**
 * td_timeruler_week_optimal_format:
 * @timeruler: time ruler
 * 
 * fr: Retourne le format optimal pour l'affichage des semaines
 *
 * en: Return the optimal format for displaying weeks
 * 
 * Return value: string
 **/

gchar *td_timeruler_week_optimal_format (TdTimeruler *timeruler)
{
  if (((timeruler->width)>gdk_string_width (gtk_widget_get_default_style()->font, "W00 00/00/0000 ")) ||
      ((timeruler->width*7)<gdk_string_width (gtk_widget_get_default_style()->font, "W00 ")))
    return NULL;
  if ((timeruler->width*7)>gdk_string_width (gtk_widget_get_default_style()->font, "W00/0000 "))
    return td_date_intl ("week/year");
  return "week";
}

/**
 * td_timeruler_week_draw:
 * @timeruler: time ruler
 * @y0: y0 coordinate
 * @y1: y1 coordinate
 * @format: date format
 * 
 * fr: Affiche les semaines sur la rgle du temps 
 *
 * en: Displays weeks on time ruler
 **/

void td_timeruler_week_draw (TdTimeruler *timeruler, int y0, int y1, gchar *format)
{
  GnomeCanvasItem *label_under;
  GdkWindow *window = timeruler->widget->window;
  int day;
  int week;
  int i, j, k;
  g_return_if_fail (timeruler != NULL);
  g_return_if_fail (TD_IS_TIMERULER (timeruler));
  if (!format)
    format = "week";

  /*** Go ! */
  day = 0;
  week = g_date_monday_week_of_year (timeruler->lower);
  for (i=g_date_year (timeruler->lower); i<g_date_year (timeruler->upper)+1; i++)
    {

      /*** Premire et dernire anne - First and last year */
      if ((i==g_date_year (timeruler->lower)) && (i==g_date_year (timeruler->upper)))
	for (j=g_date_monday_week_of_year (timeruler->lower); j<g_date_monday_week_of_year (timeruler->upper)+1; j++)
	  {
	    if (j==g_date_monday_week_of_year (timeruler->lower)) /*** Premire semaine - First week */
	      {
		label_under = gnome_canvas_get_item_at (GNOME_CANVAS (timeruler->widget), timeruler->width*day, y0);
		if (label_under)
		  gtk_object_destroy (GTK_OBJECT (label_under));
		td_timeruler_draw_line (timeruler->width*day, y0, timeruler->width*day, y1, timeruler);
		td_timeruler_draw_string (timeruler->width*day+3, y0, td_date_week (j, i, format), timeruler);
		day = day-g_date_weekday (timeruler->lower)+8;
	      }
	    else /*** Semaine intermdiaire - Intermediate week */
	      {
		label_under = gnome_canvas_get_item_at (GNOME_CANVAS (timeruler->widget), timeruler->width*day, y0);
		if (label_under)
		  gtk_object_destroy (GTK_OBJECT (label_under));
		td_timeruler_draw_line (timeruler->width*day, y0, timeruler->width*day, y1, timeruler);
		td_timeruler_draw_string (timeruler->width*day+3, y0, td_date_week (j, i, format), timeruler);
		day = day+7;
	      }
	  }

      /*** Premire anne - First year */
      if ((i==g_date_year (timeruler->lower)) && (i!=g_date_year (timeruler->upper)))
	for (j=g_date_monday_week_of_year (timeruler->lower); j<g_date_monday_weeks_in_year (i)+1; j++)
	  {
	    if (j==g_date_monday_week_of_year (timeruler->lower)) /*** First week - Premire semaine */
	      {
		label_under = gnome_canvas_get_item_at (GNOME_CANVAS (timeruler->widget), timeruler->width*day, y0);
		if (label_under)
		  gtk_object_destroy (GTK_OBJECT (label_under));
		td_timeruler_draw_line (timeruler->width*day, y0, timeruler->width*day, y1, timeruler);
		td_timeruler_draw_string (timeruler->width*day+3, y0, td_date_week (j, i, format), timeruler);
		day = day-g_date_weekday (timeruler->lower)+7;
	      }
	    else /*** Semaine intermdiaire - Intermediate week */
	      {
		label_under = gnome_canvas_get_item_at (GNOME_CANVAS (timeruler->widget), timeruler->width*day, y0);
		if (label_under)
		  gtk_object_destroy (GTK_OBJECT (label_under));
		td_timeruler_draw_line (timeruler->width*day, y0, timeruler->width*day, y1, timeruler);
		td_timeruler_draw_string (timeruler->width*day+3, y0, td_date_week (j, i, format), timeruler);
		day = day+7;
	      }
	  }

      /*** Dernire anne - Last year */
      if ((i!=g_date_year (timeruler->lower)) && (i==g_date_year (timeruler->upper)))
	for (j=1; j<g_date_monday_week_of_year (timeruler->upper)+1; j++)
	  {
	    label_under = gnome_canvas_get_item_at (GNOME_CANVAS (timeruler->widget), timeruler->width*day, y0);
	    if (label_under)
	      gtk_object_destroy (GTK_OBJECT (label_under));
	    td_timeruler_draw_line (timeruler->width*day, y0, timeruler->width*day, y1, timeruler);
	    td_timeruler_draw_string (timeruler->width*day+3, y0, td_date_week (j, i, format), timeruler);
	    day = day+7;
	  }

      /*** Anne intermdiaire - Intermediate year */
      if ((i!=g_date_year (timeruler->lower)) && (i!=g_date_year (timeruler->upper)))
	for (j=1; j<g_date_monday_weeks_in_year (i)+1; j++)
	  {
	    label_under = gnome_canvas_get_item_at (GNOME_CANVAS (timeruler->widget), timeruler->width*day, y0);
	    if (label_under)
	      gtk_object_destroy (GTK_OBJECT (label_under));
	    td_timeruler_draw_line (timeruler->width*day, y0, timeruler->width*day, y1, timeruler);
	    td_timeruler_draw_string (timeruler->width*day+3, y0, td_date_week (j, i, format), timeruler);
	    day = day+7;
	  }
    }
}

/**
 * td_timeruler_day_optimal_format:
 * @timeruler: time ruler
 * 
 * fr: Retourne le format optimal pour l'affichage des jours
 *
 * en: Return the optimal format for displaying days
 * 
 * Return value: string
 **/

gchar *td_timeruler_day_optimal_format (TdTimeruler *timeruler)
{
  if ((timeruler->width)<gdk_string_width (gtk_widget_get_default_style()->font, "00 "))
    return NULL;
  if ((timeruler->width)>gdk_string_width (gtk_widget_get_default_style()->font, "00/00/0000 "))
    return td_date_intl ("day/month/year");
  if ((timeruler->width)>gdk_string_width (gtk_widget_get_default_style()->font, "W00 00/00/0000 "))
    return td_date_intl ("week day/month/year");
  return "day";
}

/**
 * td_timeruler_day_draw:
 * @timeruler: time ruler
 * @y0: y0 coordinate
 * @y1: y1 coordinate
 * @format: date format
 * 
 * fr: Affiche les jours sur la rgle du temps 
 *
 * en: Displays days on time ruler
 **/

void td_timeruler_day_draw (TdTimeruler *timeruler, int y0, int y1, gchar *format)
{
  GnomeCanvasItem *label_under;
  GdkWindow *window = timeruler->widget->window;
  int day;
  int i, j, k;
  g_return_if_fail (timeruler != NULL);
  g_return_if_fail (TD_IS_TIMERULER (timeruler));
  if (!format)
    format = "day";

  /*** Go ! */
  day = 0;
  for (i=g_date_year (timeruler->lower); i<g_date_year (timeruler->upper)+1; i++)
    {

      /*** Premire et dernire anne - First and last year */
      if ((i==g_date_year (timeruler->lower)) && (i==g_date_year (timeruler->upper)))
	for (j=g_date_month (timeruler->lower); j<g_date_month (timeruler->upper)+1; j++)
	  {

	    /*** Premier et dernier mois - First and last month */
	    if ((j==g_date_month (timeruler->lower)) && (j==g_date_month (timeruler->upper)))
	      for (k=g_date_day (timeruler->lower); k<g_date_day (timeruler->upper)+1; k++)
		{
		  label_under = gnome_canvas_get_item_at (GNOME_CANVAS (timeruler->widget), timeruler->width*day, y0);
		  if (label_under)
		    gtk_object_destroy (GTK_OBJECT (label_under));
		  td_timeruler_draw_line (timeruler->width*day, y0, timeruler->width*day, y1, timeruler);
		  td_timeruler_draw_string (timeruler->width*day+3, y0, td_date_day (k, j, i, format), timeruler);
		  day++;
		}

	    /*** Premier mois - First month */
	    if ((j==g_date_month (timeruler->lower)) && (j!=g_date_month (timeruler->upper)))
	      for (k=g_date_day (timeruler->lower); k<g_date_days_in_month (j, i)+1; k++)
		{
		  label_under = gnome_canvas_get_item_at (GNOME_CANVAS (timeruler->widget), timeruler->width*day, y0);
		  if (label_under)
		    gtk_object_destroy (GTK_OBJECT (label_under));
		  td_timeruler_draw_line (timeruler->width*day, y0, timeruler->width*day, y1, timeruler);
		  td_timeruler_draw_string (timeruler->width*day+3, y0, td_date_day (k, j, i, format), timeruler);
		  day++;
		}

	    /*** Dernier mois - Last month */
	    if ((j!=g_date_month (timeruler->lower)) && (j==g_date_month (timeruler->upper)))
	      for (k=1; k<g_date_day (timeruler->upper)+1; k++)
		{
		  label_under = gnome_canvas_get_item_at (GNOME_CANVAS (timeruler->widget), timeruler->width*day, y0);
		  if (label_under)
		    gtk_object_destroy (GTK_OBJECT (label_under));
		  td_timeruler_draw_line (timeruler->width*day, y0, timeruler->width*day, y1, timeruler);
		  td_timeruler_draw_string (timeruler->width*day+3, y0, td_date_day (k, j, i, format), timeruler);
		  day++;
		}

	    /*** Mois intermdiare - Intermediate month */
	    if ((j!=g_date_month (timeruler->lower)) && (j!=g_date_month (timeruler->upper)))
	      for (k=1; k<g_date_days_in_month (j, i)+1; k++)
		{
		  label_under = gnome_canvas_get_item_at (GNOME_CANVAS (timeruler->widget), timeruler->width*day, y0);
		  if (label_under)
		    gtk_object_destroy (GTK_OBJECT (label_under));
		  td_timeruler_draw_line (timeruler->width*day, y0, timeruler->width*day, y1, timeruler);
		  td_timeruler_draw_string (timeruler->width*day+3, y0, td_date_day (k, j, i, format), timeruler);
		  day++;
		}
	  }

      /*** Premire anne - First year */
      if ((i==g_date_year (timeruler->lower)) && (i!=g_date_year (timeruler->upper)))
	for (j=g_date_month (timeruler->lower); j<13; j++)
	  {
	    if (j==g_date_month (timeruler->lower))
	      for (k=g_date_day (timeruler->lower); k<g_date_days_in_month (j, i)+1; k++)
		{
		  label_under = gnome_canvas_get_item_at (GNOME_CANVAS (timeruler->widget), timeruler->width*day, y0);
		  if (label_under)
		    gtk_object_destroy (GTK_OBJECT (label_under));
		  td_timeruler_draw_line (timeruler->width*day, y0, timeruler->width*day, y1, timeruler);
		  td_timeruler_draw_string (timeruler->width*day+3, y0, td_date_day (k, j, i, format), timeruler);
		  day++;
		}
	    else
	      for (k=1; k<g_date_days_in_month (j, i)+1; k++)
		{
		  label_under = gnome_canvas_get_item_at (GNOME_CANVAS (timeruler->widget), timeruler->width*day, y0);
		  if (label_under)
		    gtk_object_destroy (GTK_OBJECT (label_under));
		  td_timeruler_draw_line (timeruler->width*day, y0, timeruler->width*day, y1, timeruler);
		  td_timeruler_draw_string (timeruler->width*day+3, y0, td_date_day (k, j, i, format), timeruler);
		  day++;
		}
	  }

      /*** Dernire anne - Last year */
      if ((i!=g_date_year (timeruler->lower)) && (i==g_date_year (timeruler->upper)))
	for (j=1; j<g_date_month (timeruler->upper)+1; j++)
	  {
	    if (j==g_date_month (timeruler->upper))
	      for (k=1; k<g_date_day (timeruler->upper)+1; k++)
		{
		  label_under = gnome_canvas_get_item_at (GNOME_CANVAS (timeruler->widget), timeruler->width*day, y0);
		  if (label_under)
		    gtk_object_destroy (GTK_OBJECT (label_under));
		  td_timeruler_draw_line (timeruler->width*day, y0, timeruler->width*day, y1, timeruler);
		  td_timeruler_draw_string (timeruler->width*day+3, y0, td_date_day (k, j, i, format), timeruler);
		  day++;
		}
	    else
	      for (k=1; k<g_date_days_in_month (j, i)+1; k++)
		{
		  label_under = gnome_canvas_get_item_at (GNOME_CANVAS (timeruler->widget), timeruler->width*day, y0);
		  if (label_under)
		    gtk_object_destroy (GTK_OBJECT (label_under));
		  td_timeruler_draw_line (timeruler->width*day, y0, timeruler->width*day, y1, timeruler);
		  td_timeruler_draw_string (timeruler->width*day+3, y0, td_date_day (k, j, i, format), timeruler);
		  day++;
		}
	  }

      /*** Anne intermdiaire - Intermediate year */
      if ((i!=g_date_year (timeruler->lower)) && (i!=g_date_year (timeruler->upper)))
	for (j=1; j<13; j++)
	  for (k=1; k<g_date_days_in_month (j, i)+1; k++)
	    {
	      label_under = gnome_canvas_get_item_at (GNOME_CANVAS (timeruler->widget), timeruler->width*day, y0);
	      if (label_under)
		gtk_object_destroy (GTK_OBJECT (label_under));
	      td_timeruler_draw_line (timeruler->width*day, y0, timeruler->width*day, y1, timeruler);
	      td_timeruler_draw_string (timeruler->width*day+3, y0, td_date_day (k, j, i, format), timeruler);
	      day++;
	    }
    }
}

/**
 * td_timeruler_set_scale:
 * @timeruler: time ruler
 * @scale: zoom scale
 * 
 * fr: Rafrachis la graduation de la rgle du temps
 *
 * en: Refresh the scale of time ruler
 **/

void td_timeruler_set_scale (TdTimeruler *timeruler, int scale)
{
  int txt_num;
  gchar *txt_unit;
  g_return_if_fail (timeruler != NULL);
  g_return_if_fail (TD_IS_TIMERULER (timeruler));
  /*** FIXME: pour les heures : +tards - Hours : this night */

  /*** Go ! */
  if (timeruler->scale != scale)
    timeruler->scale = scale;
  switch (scale) 
    {
    case 1: /*** 5 ans - 5 years */
      timeruler->upper = td_date_add_years (timeruler->lower, 5);
      txt_num = 5;
      txt_unit = _("years");
      break;
    case 2: /*** 4 ans - 4 years */
      timeruler->upper = td_date_add_years (timeruler->lower, 4);
      txt_num = 4;
      txt_unit = _("years");
      break;
    case 3: /*** 3 ans - 3 years */
      timeruler->upper = td_date_add_years (timeruler->lower, 3);
      txt_num = 3;
      txt_unit = _("years");
      break;
    case 4: /*** 2 ans - 2 years */
      timeruler->upper = td_date_add_years (timeruler->lower, 2);
      txt_num = 2;
      txt_unit = _("years");
      break;

    case 5: /*** 1 an - 1 year */
      timeruler->upper = td_date_add_years (timeruler->lower, 1);
      txt_num = 1;
      txt_unit = _("year");
      break;
    case 6: /*** 6 mois - 6 months */
      timeruler->upper = td_date_add_months (timeruler->lower, 6);
      txt_num = 6;
      txt_unit = _("months");
      break;
    case 7: /*** 4 mois - 4 months */
      timeruler->upper = td_date_add_months (timeruler->lower, 4);
      txt_num = 4;
      txt_unit = _("months");
      break;
    case 8: /*** 3 mois - 3 months */
      timeruler->upper = td_date_add_months (timeruler->lower, 3);
      txt_num = 3;
      txt_unit = _("months");
      break;
    case 9: /*** 2 mois - 2 months */
      timeruler->upper = td_date_add_months (timeruler->lower, 2);
      txt_num = 2;
      txt_unit = _("months");
      break;

    case 10: /*** 1 mois - 1 month */
      timeruler->upper = td_date_add_months (timeruler->lower, 1);
      txt_num = 1;
      txt_unit = _("month");
      break;
    case 11: /*** 3 semaines - 3 weeks */
      timeruler->upper = td_date_add_days (timeruler->lower, 7*3);
      txt_num = 3;
      txt_unit = _("weeks");
      break;
    case 12: /*** 2 semaines - 2 weeks */
      timeruler->upper = td_date_add_days (timeruler->lower, 7*2);
      txt_num = 2;
      txt_unit = _("weeks");
      break;

    case 13: /*** 1 semaine - 1 week */
      timeruler->upper = td_date_add_days (timeruler->lower, 7);
      txt_num = 1;
      txt_unit = _("week");
      break;
    case 14: /*** 6 jours - 6 days */
      timeruler->upper = td_date_add_days (timeruler->lower, 6);
      txt_num = 6;
      txt_unit = _("days");
      break;
    case 15: /*** 5 jours - 5 days */
      timeruler->upper = td_date_add_days (timeruler->lower, 5);
      txt_num = 5;
      txt_unit = _("days");
      break;
    case 16: /*** 4 jours - 4 days */
      timeruler->upper = td_date_add_days (timeruler->lower, 4);
      txt_num = 4;
      txt_unit = _("days");
      break;
    case 17: /*** 3 jours - 3 days */
      timeruler->upper = td_date_add_days (timeruler->lower, 3);
      txt_num = 3;
      txt_unit = _("days");
      break;
    case 18: /*** 2 jours - 2 days */
      timeruler->upper = td_date_add_days (timeruler->lower, 2);
      txt_num = 2;
      txt_unit = _("days");
      break;
    case 19: /*** 1 jour - 1 day */
      timeruler->upper = td_date_add_days (timeruler->lower, 1);
      txt_num = 1;
      txt_unit = _("day");
      break;
    default:
      txt_num = 1;
      txt_unit = _("day");
      timeruler->upper = td_date_add_days (timeruler->lower, 1);
      break;
    }
  if (txt_num==1)
    gtk_label_set_text (GTK_LABEL (timeruler->zoom_label), g_strdup_printf ("%d %s ", txt_num, txt_unit));
  else
    gtk_label_set_text (GTK_LABEL (timeruler->zoom_label), g_strdup_printf ("%d %s", txt_num, txt_unit));
  timeruler->width = ((float) timeruler->widget->allocation.width)/td_date_period_day (timeruler->lower, timeruler->upper);
  td_timeruler_draw (timeruler);
}

/**
 * td_timeruler_scale_label_maxsize:
 * 
 * fr: Retourne la longueur de l'tiquette du zoom
 *
 * en: Retourne the lenght of the zoom label
 * 
 * Return value: integer
 **/

int td_timeruler_scale_label_maxsize (void)
{
  int ret = 0;
  if (ret<gdk_string_width (gtk_widget_get_default_style()->font, _("years")))
    ret = gdk_string_width (gtk_widget_get_default_style()->font, _("years"));
  if (ret<gdk_string_width (gtk_widget_get_default_style()->font, _("year")))
    ret = gdk_string_width (gtk_widget_get_default_style()->font, _("year"));
  if (ret<gdk_string_width (gtk_widget_get_default_style()->font, _("months")))
    ret = gdk_string_width (gtk_widget_get_default_style()->font, _("months"));
  if (ret<gdk_string_width (gtk_widget_get_default_style()->font, _("month")))
    ret = gdk_string_width (gtk_widget_get_default_style()->font, _("month"));
  if (ret<gdk_string_width (gtk_widget_get_default_style()->font, _("days")))
    ret = gdk_string_width (gtk_widget_get_default_style()->font, _("days"));
  if (ret<gdk_string_width (gtk_widget_get_default_style()->font, _("day")))
    ret = gdk_string_width (gtk_widget_get_default_style()->font, _("day"));
  ret = ret+gdk_string_width (gtk_widget_get_default_style()->font, " 00 ");
  return ret;
}
