/*****************************************************************************/
/*** td_app.c : Objet Gtk+
 *** td_app.c : Gtk+ object
 *
 *
 * ToutDoux : Chtit gestionnaire de projet - A littl' project manager
 * Copyright (c) 2000-2001 Philippe Roy
 * Auteur - Author : Philippe Roy <ph_roy@toutdoux.org>
 *
 *
 * Ce programme est un logiciel libre ; vous pouvez le redistribuer et/ou le modifier
 * sous les termes de la licence publique gnrale GNU telle qu'elle est publie par
 * la Free Software Foundation ; soit la version 2 de la licence, ou
 * (comme vous voulez) toute version ultrieure.
 *
 * Ce programme est distribu dans l'espoir qu'il sera utile,
 * mais SANS AUCUNE GARANTIE ; mme sans la garantie de
 * COMMERCIALIT ou d'ADQUATION A UN BUT PARTICULIER. Voir la
 * licence publique gnrale GNU pour plus de dtails.
 *
 * Vous devriez avoir reu une copie de la licence publique gnrale GNU
 * avec ce programme ; si ce n'est pas le cas, crivez  la Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. */
/*****************************************************************************/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>
#include <dlfcn.h>
#include <dirent.h>
#include <sys/stat.h>

#include "commons.h"
#include "gtdk.h"
#include "database.h"
#include "td_app.h"
#include "td_app_palette.h"
#include "td_app_bench.h"
#include "td_app_etabliste.h"
#include "td_mod.h"
#include "td_db_base.h"
#include "td_db_table.h"
#include "td_db_connect.h"
#include "td_function.h"
#include "td_flower.h"

/*****************************************************************************/
/*** Arguments */
/*****************************************************************************/
static GtkObjectClass *parent_class = NULL;

enum {
  ARG_0,
  ARG_MODIFIED,
  ARG_MODE_ECHO,
};

static void td_app_set_modified (TdApp *app, gboolean modified)
{
  g_return_if_fail (app != NULL);
  g_return_if_fail (TD_IS_APP (app));
  if (app->modified != modified)
    app->modified = modified;
}

static void td_app_set_mode_echo (TdApp *app, int mode_echo)
{
  g_return_if_fail (app != NULL);
  g_return_if_fail (TD_IS_APP (app));
  if (app->mode_echo != mode_echo)
    app->mode_echo = mode_echo;
}

static void td_app_set_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
  TdApp *app;
  app = TD_APP (object);
  switch (arg_id)
    {
    case ARG_MODIFIED:
      td_app_set_modified (app, GTK_VALUE_BOOL (*arg));
      break;
    case ARG_MODE_ECHO:
      td_app_set_mode_echo (app, GTK_VALUE_INT (*arg));
      break;
    default:
      break;
    }
}

static void td_app_get_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
  TdApp *app;
  app = TD_APP (object);
   switch (arg_id)
    {
    case ARG_MODIFIED:
      GTK_VALUE_BOOL (*arg) = app->modified;
      break;
    case ARG_MODE_ECHO:
      GTK_VALUE_INT (*arg) = app->mode_echo;
    default:
      arg->type = GTK_TYPE_INVALID;
      break;
    }
}

/*****************************************************************************/
/*** Initialisation */
/*****************************************************************************/
static void td_app_init (TdApp *app)
{
  app->pid = 0;
  app->pid_database = NULL;
  app->palette = NULL;
  app->bench = NULL;
  app->etabliste = NULL;
  app->modified = FALSE;
  app->mode_echo = TD_MODE_ECHO_NO;
  app->admin_field = NULL;
  app->connect = NULL;
  app->base = NULL;

  app->accel_group = NULL;
  app->menu_main = NULL;
  app->menu_gnomeui = NULL;
  app->menu_palette = NULL;
  app->menu_favorites = NULL;
  app->menu_templates = NULL;
  app->menu_commands = NULL;
  app->menu_history = NULL;

  app->about = NULL;
  app->presto_browser = NULL;
  app->custom_browser = NULL;
  app->menu_mod = NULL;
  app->mod_core = NULL;
  app->mod_data = NULL;
  app->mod_filter = NULL;
  app->process = NULL;

  app->function = NULL;
}

static void td_app_class_init (TdAppClass *klass)
{
  GtkObjectClass *object_class;
  object_class = (GtkObjectClass*) klass;
  parent_class = gtk_type_class (gtk_object_get_type());
  gtk_object_add_arg_type ("TdApp::modified", GTK_TYPE_BOOL, GTK_ARG_READWRITE, ARG_MODIFIED);
  gtk_object_add_arg_type ("TdApp::mode_echo", GTK_TYPE_BOOL, GTK_ARG_READWRITE, ARG_MODE_ECHO);
  object_class->set_arg = td_app_set_arg;
  object_class->get_arg = td_app_get_arg;

  klass->set_palette = td_app_set_palette;
  klass->set_bench = td_app_set_bench;
  klass->set_etabliste = td_app_set_etabliste;

  klass->set_modified = td_app_set_modified;
  klass->add_admin_field = td_app_add_admin_field;
  klass->set_current = td_app_set_current;

  klass->set_accel_group = td_app_set_accel_group;
  klass->set_menu_main = td_app_set_menu_main;
  klass->set_menu_palette = td_app_set_menu_palette;
  klass->set_menu_favorites = td_app_set_menu_favorites;
  klass->set_menu_templates = td_app_set_menu_templates;
  klass->set_menu_commands = td_app_set_menu_commands;
  klass->set_menu_history = td_app_set_menu_history;

  klass->set_about = td_app_set_about;
  klass->add_menu_mod = td_app_add_menu_mod;

  klass->add_function = td_app_add_function;
  klass->create = td_app_create;
  klass->quit = td_app_quit;
}

GtkType td_app_get_type (void)
{
  static GtkType app_type = 0;
  if (!app_type)
    {
      static const GtkTypeInfo app_info =
      {
  	"TdApp", sizeof (TdApp), sizeof (TdAppClass),
  	(GtkClassInitFunc) td_app_class_init,
	(GtkObjectInitFunc) td_app_init,
	NULL, NULL, (GtkClassInitFunc) NULL,
      };
      app_type = gtk_type_unique (GTK_TYPE_OBJECT, &app_info);
    }
  return app_type;
}

/**
 * td_app_new:
 * 
 * fr: Cr une nouvelle application
 *
 * en: Creates a new application
 * 
 * Return value: application
 **/

GtkObject *td_app_new (void)
{
  return GTK_OBJECT (gtk_type_new (td_app_get_type()));
}

/**
 * td_app_quit:
 * @app: application
 * 
 * fr: Dtruit l'application
 *
 * en: Destroys the application
 **/

void td_app_quit (TdApp *app)
{
  void (*symbol)() = NULL;
  int i;
  g_return_if_fail (app != NULL);
  g_return_if_fail (TD_IS_APP (app));

  /*** Fermeture du fichier en cours - Closing the current file */
  td_app_message (NULL, _("Stopping"), TD_MSG_BEGIN);
  td_app_message (g_strdup_printf (("%s..."), _("Stopping")), NULL, TD_MSG_MESSAGE);
  if (td_db_connect_close (NULL, NULL))
    {

      /*** Historique - History */
      (gpointer*) symbol = td_dlsym (PLUGINS_CORE_BOOKMARKS, NULL, "libbookmarks.so", "plugins_save_history", TRUE);
      symbol();

      /*** Modules noyaux - Core plug-ins */
      TD_FLAG_NO_REFRESH_PALLETTE_MESSAGE = TRUE;
      for (i=0; i<g_list_length (app->mod_core); i++)
	{
	  symbol = dlsym (PLUGINS_CORE[i], "plugins_destroy");
	  if (dlerror() != NULL)
	    td_app_message (_("Executing plug-in"), g_strdup_printf (_("Executing plug-in : file '%s' : function '%s' not find"), (gchar*) g_list_nth_data (app->mod_core, i), "plugins_destroy"), TD_MSG_FAILED);
	  else
	    {
	      symbol (g_list_nth_data (app->mod_core, i));
	      td_app_message (NULL, g_strdup_printf (_("Executing plug-in : file '%s' : function '%s'"), (gchar*) g_list_nth_data (app->mod_core, i), "plugins_destroy"), TD_MSG_PLUGINS);
	    }
	}
      if (PLUGINS_CORE_DATABASE)
	{
	  (gpointer*) symbol = dlsym (PLUGINS_CORE_DATABASE, "plugins_destroy");
	  symbol();
	}
      if (PLUGINS_CORE_MESSAGES)
	{
	  (gpointer*) symbol = dlsym (PLUGINS_CORE_MESSAGES, "plugins_destroy");
	  symbol();
	}
      if (PLUGINS_CORE_XTS)
	{
	  (gpointer*) symbol = dlsym (PLUGINS_CORE_XTS, "plugins_destroy");
	  symbol();
	}
      TD_FLAG_NO_REFRESH_PALLETTE_MESSAGE = FALSE;
      td_app_message (NULL, _("Stopping"), TD_MSG_END);

      /*** Fentres de base - Basis windows */
      TD_FLAG_NO_REFRESH_PALLETTE_MESSAGE = TRUE;
      td_app_lock_remove (app, app->pid);
      td_database_base_logout();
      td_database_base_drop (app->pid_database);
      if (TD_APP_PALETTE (app->palette)->window)
	td_app_palette_destroy (TD_APP_PALETTE (app->palette));
      if (TD_APP_BENCH (app->bench)->window)
	td_app_bench_destroy (TD_APP_BENCH (app->bench));
      if (TD_APP_ETABLISTE (app->etabliste)->window)
	td_app_etabliste_destroy (TD_APP_ETABLISTE (app->etabliste));

      /*** Application */
      app->pid = 0;
      app->pid_database = NULL;
      if (app->palette)
	gtk_object_destroy (app->palette);
      if (app->bench)
	gtk_object_destroy (app->bench);
      if (app->etabliste)
	gtk_object_destroy (app->etabliste);
      app->mode_echo = 0;
      g_list_free (app->admin_field);
      app->modified = 0;
      if (app->connect)
	gtk_object_destroy (app->connect);
      if (app->base)
	gtk_object_destroy (app->base);

      /* gtk_widget_destroy (app->accel_group); */
      gtk_widget_destroy (app->menu_main);
      /* gtk_widget_destroy (app->menu_gnomeui); */
      if (GTK_IS_WIDGET (app->menu_palette))
	gtk_widget_destroy (app->menu_palette);
      if (GTK_IS_WIDGET (app->menu_favorites))
	gtk_widget_destroy (app->menu_favorites);
      if (GTK_IS_WIDGET (app->menu_templates))
	gtk_widget_destroy (app->menu_templates);
      if (GTK_IS_WIDGET (app->menu_commands))
	gtk_widget_destroy (app->menu_commands);
      if (GTK_IS_WIDGET (app->menu_history))
	gtk_widget_destroy (app->menu_history);

      if (app->about)
	gtk_object_destroy (app->about);
      if (app->presto_browser)
	gtk_object_destroy (app->presto_browser);
      if (app->custom_browser)
	gtk_object_destroy (app->custom_browser);
      g_list_free (app->menu_mod);
      g_list_free (app->mod_core);
      g_list_free (app->mod_data);
      g_list_free (app->mod_filter);
      g_list_free (app->process);

      g_list_free (app->function);
      g_list_free (app->function_input);

      gtk_object_destroy (GTK_OBJECT (app));
      td_exit (0);
    }
  td_app_message (NULL, _("Stopping"), TD_MSG_END);
}

/*****************************************************************************/
/*** Affectations - Allocations */
/*****************************************************************************/

/**
 * td_app_set_palette:
 * @app: application
 * @palette: palette
 * 
 * fr: Affecte la palette  l'application
 *
 * en: Allocates the palette to application
 **/

void td_app_set_palette (TdApp *app, GtkObject *palette)
{
  g_return_if_fail (app != NULL);
  g_return_if_fail (TD_IS_APP (app));
  if (app->palette != palette)
    app->palette = palette;
}

/**
 * td_app_set_bench:
 * @app: application
 * @bench: bench
 * 
 * fr: Affecte l'tabli  l'application
 *
 * en: Allocates the bench to application
 **/

void td_app_set_bench (TdApp *app, GtkObject *bench)
{
  g_return_if_fail (app != NULL);
  g_return_if_fail (TD_IS_APP (app));
  if (app->bench != bench)
    app->bench = bench;
}

/**
 * td_app_set_etabliste:
 * @app: application
 * @etabliste: etabliste
 * 
 * fr: Affecte l'tabli des listes  l'application
 *
 * en: Allocates the bench of lists to application
 **/

void td_app_set_etabliste (TdApp *app, GtkObject *etabliste)
{
  g_return_if_fail (app != NULL);
  g_return_if_fail (TD_IS_APP (app));
  if (app->etabliste != etabliste)
    app->etabliste = etabliste;
}

/**
 * td_app_add_admin_field:
 * @app: application
 * @field: field
 * 
 * fr: Ajoute un champs d'administration  l'application
 *
 * en: Adds a admintration field to application
 **/

void td_app_add_admin_field (TdApp *app, gchar *field)
{
  g_return_if_fail (app != NULL);
  g_return_if_fail (TD_IS_APP (app));
  app->admin_field = g_list_append (app->admin_field, field);
}

/**
 * td_app_set_accel_group:
 * @app: application
 * @accel_group: accelerators group
 * 
 * fr: Affecte le groupe d'acclrateurs  l'application
 *
 * en: Allocates the accelerators group to application
 **/

void td_app_set_accel_group (TdApp *app, GtkAccelGroup *accel_group)
{
  g_return_if_fail (app != NULL);
  g_return_if_fail (TD_IS_APP (app));
  if (app->accel_group != accel_group)
    app->accel_group = accel_group;
}

/**
 * td_app_set_menu_main:
 * @app: application
 * @menu_gnomeui: menu
 * 
 * fr: Affecte le menu comme menu principal  l'application
 *
 * en: Allocates the menu like main menu to application
 **/

void td_app_set_menu_main (TdApp *app, GnomeUIInfo *menu_gnomeui)
{
  g_return_if_fail (app != NULL);
  g_return_if_fail (TD_IS_APP (app));
  if (app->menu_gnomeui != menu_gnomeui)
    {
      app->menu_gnomeui = menu_gnomeui;
      app->menu_main = gnome_popup_menu_new (menu_gnomeui);
    }
}

/**
 * td_app_set_menu_palette:
 * @app: application
 * @menu_palette: menu
 * 
 * fr: Affecte le menu comme menu de palette  l'application
 *
 * en: Allocates the menu like palette menu to application
 **/

void td_app_set_menu_palette (TdApp *app, GtkWidget *menu_palette)
{
  g_return_if_fail (app != NULL);
  g_return_if_fail (TD_IS_APP (app));
  if (app->menu_palette != menu_palette)
    app->menu_palette = menu_palette;
}

/**
 * td_app_set_menu_favorites:
 * @app: application
 * @menu_favorites: menu
 * 
 * fr: Affecte le menu comme menu des favoris  l'application
 *
 * en: Allocates the menu like favorites menu to application
 **/

void td_app_set_menu_favorites (TdApp *app, GtkWidget *menu_favorites)
{
  g_return_if_fail (app != NULL);
  g_return_if_fail (TD_IS_APP (app));
  if (app->menu_favorites != menu_favorites)
    app->menu_favorites = menu_favorites;
}

/**
 * td_app_set_menu_templates:
 * @app: application
 * @menu_templates: menu
 * 
 * fr: Affecte le menu comme menu des gabarits  l'application
 *
 * en: Allocates the menu like templates menu to application
 **/

void td_app_set_menu_templates (TdApp *app, GtkWidget *menu_templates)
{
  g_return_if_fail (app != NULL);
  g_return_if_fail (TD_IS_APP (app));
  if (app->menu_templates != menu_templates)
    app->menu_templates = menu_templates;
}

/**
 * td_app_set_menu_commands:
 * @app: application
 * @menu_commands: menu
 * 
 * fr: Affecte le menu comme menu des commandes  l'application
 *
 * en: Allocates the menu like commands menu to application
 **/

void td_app_set_menu_commands (TdApp *app, GtkWidget *menu_commands)
{
  g_return_if_fail (app != NULL);
  g_return_if_fail (TD_IS_APP (app));
  if (app->menu_commands != menu_commands)
    app->menu_commands = menu_commands;
}

/**
 * td_app_set_menu_history:
 * @app: application
 * @menu_history: menu
 * 
 * fr: Affecte le menu comme menu de l'historique  l'application
 *
 * en: Allocates the menu like history menu to application
 **/

void td_app_set_menu_history (TdApp *app, GtkWidget *menu_history)
{
  g_return_if_fail (app != NULL);
  g_return_if_fail (TD_IS_APP (app));
  if (app->menu_history != menu_history)
    app->menu_history = menu_history;
}

/**
 * td_app_set_current:
 * @app: application
 * @connect: connection
 * @base: base
 * 
 * fr: Affecte la connection et la base  l'application
 *
 * en: Allocates the connection and database to application
 **/

void td_app_set_current (TdApp *app, GtkObject *connect, GtkObject *base)
{
  int i, j, k;
  GtkObject *function;
  TdFunction *cur_function;
  g_return_if_fail (app != NULL);
  g_return_if_fail (TD_IS_APP (app));

  /*** Base */
  if (app->connect != connect)
    app->connect = connect;
  if (app->base != base)
    app->base = base;
  if (TD_DB_CONNECT (connect)->type != TD_CONNECT_TYPE_NONE)
    td_app_lock_update (app, TD_DB_CONNECT (connect)->file);

  /*** Fonction - Function */
  if (base)
    for (i=0; i<g_list_length (TD_DB_BASE (base)->table); i++)
      for (j=0; j<g_list_length (TD_DB_TABLE (g_list_nth_data (TD_DB_BASE (base)->table, i))->function); j++)
	{
	  cur_function = TD_FUNCTION (g_list_nth_data (TD_DB_TABLE (g_list_nth_data (TD_DB_BASE (base)->table, i))->function, j));
	  function = td_function_new();
	  gtk_object_set (GTK_OBJECT (function),
			  "name", cur_function->name,
			  "comment", cur_function->comment, NULL);
	  for (k=0; k<g_list_length (cur_function->property_name); k++)
	    {
	      if (!strcmp ((gchar*) g_list_nth_data (cur_function->property_name, k), "input"))
		{
		  td_function_add_input (TD_FUNCTION (function), (gchar*) g_list_nth_data (cur_function->property_value, k));
		  continue;
		}
	      if (!strcmp ((gchar*) g_list_nth_data (cur_function->property_name, k), "output"))
		{
		  td_function_add_output (TD_FUNCTION (function), (gchar*) g_list_nth_data (cur_function->property_value, k));
		  continue;
		}
	      if (!strcmp ((gchar*) g_list_nth_data (cur_function->property_name, k), "language"))
		{
		  gtk_object_set (GTK_OBJECT (function), "language", (gchar*) g_list_nth_data (cur_function->property_value, k), NULL);
		  continue;
		}
	      if (!strcmp ((gchar*) g_list_nth_data (cur_function->property_name, k), "definition"))
		{
		  gtk_object_set (GTK_OBJECT (function), "definition", (gchar*) g_list_nth_data (cur_function->property_value, k), NULL);
		  continue;
		}
	    }
	  td_app_add_function (app, function);
	}
}

/**
 * td_app_set_about:
 * @app: application
 * @about: description
 * 
 * fr: Affecte la description  l'application
 *
 * en: Allocates the description to application
 **/

void td_app_set_about (TdApp *app, GtkObject *about)
{
  g_return_if_fail (app != NULL);
  g_return_if_fail (TD_IS_APP (app));
  if (app->about != about)
    app->about = about;
}

/**
 * td_app_add_mod_core:
 * @app: application
 * @dir: directory location of plug-in
 * @file: file of plug-in
 * 
 * fr: Ajoute un module noyau  l'application
 *
 * en: Adds a core module to application
 *
 * Return value: FALSE on error
 **/

gboolean td_app_add_mod_core (TdApp *app, gchar *dir, gchar *file)
{
  int i, mod_core_id;
  gchar *mod_core;
  void (*symbol)();
  if (!dir)
    dir = TD_PACKAGE_PLUGINS_DIR;
  if (!app)
    app = TD_APP(APP);
  mod_core = g_strdup_printf ("%s%s", dir, file);
  mod_core_id = g_list_length (app->mod_core);

  /*** Chargement - Loading */
  for (i=0; i<g_list_length (app->mod_core); i++)
    if (!strcmp ((gchar*) g_list_nth_data (app->mod_core, i), mod_core))
      return TRUE;
  app->mod_core = g_list_append (app->mod_core, mod_core);
  PLUGINS_CORE[mod_core_id] = dlopen (mod_core, RTLD_LAZY);
  if (dlerror() !=NULL)
    {
      td_app_message (_("Loading plug-in"), g_strdup_printf (_("Loading plug-in : file '%s' unreadable"), mod_core), TD_MSG_FAILED);
      return FALSE;
    }

  /*** Initialisation */
  symbol = dlsym (PLUGINS_CORE[mod_core_id], "plugins_init");
  if (dlerror() != NULL)
    {
      td_app_message (_("Executing plug-in"), g_strdup_printf (_("Executing plug-in : file '%s' : function 'plugins_init' not find"), mod_core), TD_MSG_FAILED);
      return FALSE;
    }
  td_app_message (NULL, g_strdup_printf (_("Executing plug-in : file '%s' : function 'plugins_init'"), mod_core), TD_MSG_PLUGINS);
  symbol();

  /*** Visualisation - Visualization */
  if (PLUGINS_CORE_XTS)
    {
      symbol = dlsym (PLUGINS_CORE_XTS, "plugins_update_all");
      symbol (app);
    }
  return TRUE;
}

/**
 * td_app_add_mod_filter:
 * @app: application
 * @dir: directory location of plug-in
 * @file: file of plug-in
 * @name: name of plug-in
 * @name_intl: international name of plug-in
 * 
 * fr: Ajoute un module filtre  l'application
 *
 * en: Adds a filter module to application
 *
 * Return value: FALSE on error
 **/

gboolean td_app_add_mod_filter (TdApp *app, gchar *dir, gchar *file, gchar *name, gchar *name_intl)
{
  int i, mod_filter_id;
  gchar *mod_filter;
  GtkObject *(*symbol)();
  GtkObject *mod_custom;
  GtkObject *(*symbol2)();
  if (!dir)
    dir = TD_PACKAGE_PLUGINS_DIR;
  if (!app)
    app = TD_APP (APP);
  mod_filter = g_strdup_printf ("%s%s", dir, file);
  mod_filter_id = g_list_length (app->mod_filter);

  /*** Chargement - Loading */
  for (i=0; i<g_list_length (app->mod_filter); i++)
    if (!strcmp (name, TD_MOD (g_list_nth_data (app->mod_filter, i))->name))
      return TRUE;
  PLUGINS_FILTER[mod_filter_id] = dlopen (mod_filter, RTLD_LAZY);
  if (dlerror() !=NULL)
    {
      td_app_message (_("Loading plug-in"), g_strdup_printf (_("Loading plug-in : file '%s' unreadable"), mod_filter), TD_MSG_FAILED);
      return FALSE;
    }

  /*** Initialisation */
  symbol = dlsym (PLUGINS_FILTER[mod_filter_id], "plugins_init");
  if (dlerror() != NULL)
    {
      td_app_message (_("Executing plug-in"), g_strdup_printf (_("Executing plug-in : file '%s' : function 'plugins_init' not find"), mod_filter), TD_MSG_FAILED);
      return FALSE;
    }
  td_app_message (NULL, g_strdup_printf (_("Executing plug-in : file '%s' : function 'plugins_init'"), mod_filter), TD_MSG_PLUGINS);
  mod_custom = symbol (-1);
  app->mod_filter = g_list_append (app->mod_filter, mod_custom);

  /*** Visualisation - Visualization */
  if (PLUGINS_CORE_XTS)
    {
      symbol = dlsym (PLUGINS_CORE_XTS, "plugins_update_all");
      symbol (app);
    }
  return TRUE;
}

/**
 * td_app_add_mod_data:
 * @app: application
 * @dir: directory location of plug-in
 * @file: file of plug-in
 * 
 * fr: Ajoute un module de donne  l'application
 *
 * en: Adds a data module to application
 *
 * Return value: FALSE on error
 **/

gboolean td_app_add_mod_data (TdApp *app, gchar *dir, gchar *file)
{
  int mod_data_id;
  gchar *mod_data;
  GtkObject *(*symbol)();
  if (!dir)
    dir = TD_PACKAGE_PLUGINS_DIR;
  if (!app)
    app = TD_APP (APP);
  mod_data = g_strdup_printf ("%s%s", dir, file);
  mod_data_id = g_list_length (app->mod_data);

  /*** Chargement - Loading */
  PLUGINS_DATA[mod_data_id] = dlopen (mod_data, RTLD_LAZY);
  if (dlerror() !=NULL)
    {
      td_app_message (_("Loading plug-in"), g_strdup_printf (_("Loading plug-in : file '%s' unreadable"), mod_data), TD_MSG_FAILED);
      return FALSE;
    }

  /*** Initialisation */
  symbol = dlsym (PLUGINS_DATA[mod_data_id], "plugins_init");
  if (dlerror() != NULL)
    {
      td_app_message (_("Executing plug-in"), g_strdup_printf (_("Executing plug-in : file '%s' : function 'plugins_init' not find"), mod_data), TD_MSG_FAILED);
      return FALSE;
    }
  td_app_message (NULL, g_strdup_printf (_("Executing plug-in : file '%s' : function 'plugins_init'"), mod_data), TD_MSG_PLUGINS);
  app->mod_data = g_list_append (app->mod_data, symbol (mod_data_id, NULL));

  /*** Visualisation - Visualization */
  if (PLUGINS_CORE_XTS)
    {
      symbol = dlsym (PLUGINS_CORE_XTS, "plugins_update_all");
      symbol (app);
    }
  return TRUE;
}

/**
 * td_app_add_process:
 * @app: application
 * @process: process
 * 
 * fr: Ajoute un processus  l'application
 *
 * en: Adds a process to application
 **/

void td_app_add_process (TdApp *app, GtkObject *process)
{
  g_return_if_fail (app != NULL);
  g_return_if_fail (TD_IS_APP (app));
  app->process = g_list_append (app->process, process);
}

/**
 * td_app_add_menu_mod:
 * @app: application
 * @menu_mod: module menu
 * 
 * fr: Ajoute un menu  module  l'application
 *
 * en: Adds a module menu to application
 **/

void td_app_add_menu_mod (TdApp *app, GtkObject *menu_mod)
{
  g_return_if_fail (app != NULL);
  g_return_if_fail (TD_IS_APP (app));
  app->menu_mod = g_list_append (app->menu_mod, menu_mod);
}

/**
 * td_app_add_function:
 * @app: application
 * @function: function
 * 
 * fr: Ajoute la fonction  l'application
 *
 * en: Adds the function to application
 **/

void td_app_add_function (TdApp *app, GtkObject *function)
{
  int i, j;
  gboolean bool_tmp;
  g_return_if_fail (app != NULL);
  g_return_if_fail (TD_IS_APP (app));

  /*** Fonction - Function */
  app->function = g_list_append (app->function, function);
  
  /*** Champs - Field */
  for (i=0; i<g_list_length (TD_FUNCTION (function)->input); i++)
    {
      bool_tmp = TRUE;
      for (j=0; j<g_list_length (app->function_input); j++)
	if (!strcmp ((gchar*) g_list_nth_data (TD_FUNCTION (function)->input, i), (gchar*) g_list_nth_data (app->function_input, j)))
	  {
	    bool_tmp = FALSE;
	    break;
	  }
      if (bool_tmp)
	app->function_input = g_list_append (app->function_input, (gchar*) g_list_nth_data (TD_FUNCTION (function)->input, i));
    }
}

/*****************************************************************************/
/*** Commandes - Commands */
/*****************************************************************************/

/**
 * td_app_create:
 * @app: application
 * 
 * fr: Cr l'interface de l'application
 *
 * en: Creates the interface of application
 **/

void td_app_create (TdApp *app)
{
  GtkObject *connect;
  void *module_tmp;
  void (*symbol_tmp)();;
  g_return_if_fail (app != NULL);
  g_return_if_fail (TD_IS_APP (app));

  /*** Connexion - Connection */
  connect = td_db_connect_new();
  gtk_object_set (GTK_OBJECT (connect), "name", _("New"), NULL);
  td_db_connect_set_type (TD_DB_CONNECT (connect), TD_CONNECT_TYPE_NONE);
  td_app_set_current (app, connect, NULL);
  td_app_lock_add (app);

  /*** Fentres de base - Basis windows */
  td_app_palette_create (TD_APP_PALETTE (app->palette));
  td_app_bench_create (TD_APP_BENCH (app->bench));
  td_app_etabliste_create (TD_APP_ETABLISTE (app->etabliste));

  /* Champs d'administration */
  td_app_add_admin_field (app, "td_id");
  td_app_add_admin_field (app, "td_obso");
}

/**
 * td_app_custom_show:
 * @app: td_app object
 * 
 * fr: Cr et affiche l'interface de personnalisation de l'application
 *
 * en: Creates and display customize interface of application
 **/

void td_app_custom_show (TdApp *app)
{
  void (*symbol)();
  g_return_if_fail (app != NULL);
  g_return_if_fail (TD_IS_APP (app));
  if (!app->custom_browser)
    {
      app->custom_browser = td_mod_new();
      gtk_object_set (GTK_OBJECT (app->custom_browser), "name", "app", "name_intl", "(null)", NULL);
      (gpointer*) symbol = td_app_mod_core_symbol (NULL, NULL, "libhelp_customize.so", "plugins_create", TRUE);
      if (symbol)
	symbol (app->custom_browser);
    }
  td_mod_show (TD_MOD (TD_MOD (app->custom_browser)->custom_browser));
}

/**
 * td_app_presto_show:
 * @app: td_app object
 * 
 * fr: Cr et affiche l'interface presto de l'application
 *
 * en: Creates and display presto interface of application
 **/

void td_app_presto_show (TdApp *app)
{
  void (*symbol)();
  g_return_if_fail (app != NULL);
  g_return_if_fail (TD_IS_APP (app));
  if (!app->presto_browser)
    {
      app->presto_browser = td_mod_new();
      gtk_object_set (GTK_OBJECT (app->presto_browser), "name", "app", "name_intl", "(null)", NULL);
      (gpointer*) symbol = td_app_mod_core_symbol (NULL, NULL, "libhelp_doc.so", "plugins_create", TRUE);
      if (symbol)
	symbol (app->presto_browser);
    }
  td_mod_show (TD_MOD (TD_MOD (app->presto_browser)->presto_browser));
}

/**
 * td_app_message:
 * @chtit_message: short text for palette's statusbar
 * @message: complete text
 * @type: type of message 
 * 
 * fr: Affiche un message sur la palette et dans la fentre des messages
 *
 * en: Displays message for palette's statusbar and message window
 **/

void td_app_message (gchar *chtit_message, gchar *message, TdMsg type)
{
  void (*symbol)();
  GtkCTreeNode *node;
  GtkCTreeNode *node_parent;
  gchar *txt_tampon = NULL;
  gchar *txt[2];
  GdkColor transparent;
  gchar *type_txt = NULL;
  gchar *type_color = NULL;

  /*** Shunt */
  if ((APP) &&
      ((TD_APP (APP)->mode_echo != TD_MODE_ECHO_ALL_COLORS) && (TD_APP (APP)->mode_echo != TD_MODE_ECHO_ALL_BW)) &&
      ((type == TD_MSG_PLUGINS) || (type == TD_MSG_SQL_CACHE) || (type == TD_MSG_SQL) || (type == TD_MSG_FUNCTION)))
    return;

  /*** Adaptation */
  if (!message)
    message = g_strdup_printf ("%s", chtit_message);
  switch (type)
    {
    case TD_MSG_DONE:
      type_txt = g_strdup_printf ("%s", _("done"));
      type_color = "\033[32m\033[1m";
      break;
    case TD_MSG_TESTED:
      type_txt = g_strdup_printf ("%s", _("tested"));
      type_color = "\033[32m\033[1m";
      break;
    case TD_MSG_PLUGINS:
      type_txt = g_strdup_printf ("%s", _("plugins"));
      type_color = "\033[32m\033[1m";
      break;
    case TD_MSG_COMMAND:
      type_txt = g_strdup_printf ("%s", _("command"));
      type_color = "\033[32m\033[1m";
      break;
    case TD_MSG_SQL:
      type_txt = g_strdup_printf ("%s", _("sql"));
      type_color = "\033[36m\033[1m";
      break;
    case TD_MSG_SQL_CACHE:
      type_txt = g_strdup_printf ("%s", _("sql cache"));
      type_color = "\033[32m\033[1m";
      break;
    case TD_MSG_FUNCTION:
      type_txt = g_strdup_printf ("%s", _("function"));
      type_color = "\033[32m\033[1m";
      break;
    case TD_MSG_FAILED:
      type_txt = g_strdup_printf ("%s", _("failed"));
      type_color = "\033[31m\033[1m";
      break;
    case TD_MSG_STOP:
      type_txt = g_strdup_printf ("%s", _("stop"));
      type_color = "\033[31m\033[1m";
      break;
    case TD_MSG_BEGIN:
      type_txt = g_strdup_printf ("%s", _("begin"));
      type_color = "\033[0m";
      break;
    case TD_MSG_END:
      type_txt = g_strdup_printf ("%s", _("end"));
      type_color = "\033[0m";
      break;
    case TD_MSG_MESSAGE:
      type_txt = g_strdup_printf ("%s", _("message"));
      type_color = "\033[0m";
      break;
    case TD_MSG_YES:
      type_txt = g_strdup_printf ("%s", _("yes"));
      type_color = "\033[35m\033[1m";
      break;
    case TD_MSG_NO:
      type_txt = g_strdup_printf ("%s", _("no"));
      type_color = "\033[35m\033[1m";
      break;
    default:
      type_txt = "(null)"; 
      type_color = "\033[0m";
      break;
    }

  /*** Ecriture dans la palette - Writting in the palette */
  if ((!TD_FLAG_NO_REFRESH_PALLETTE_MESSAGE) && (APP) && (TD_APP (APP)->palette) && (TD_APP_PALETTE (TD_APP (APP)->palette)->status))
    {
      if (type == TD_MSG_END)
	gtk_entry_set_text (GTK_ENTRY (TD_APP_PALETTE (TD_APP (APP)->palette)->status), " ");
      if (chtit_message)
	{
	  if (type == TD_MSG_MESSAGE)
	    txt_tampon = g_strdup_printf ("%s", chtit_message);
	  else
	    txt_tampon = g_strdup_printf ("%s : %s", chtit_message, type_txt);
	  gtk_entry_set_text (GTK_ENTRY (TD_APP_PALETTE (TD_APP (APP)->palette)->status), txt_tampon);
	}
    }

  /*** Console */
  if (APP)
    {
      switch (TD_APP (APP)->mode_echo)
	{
	case TD_MODE_ECHO_NO:
	  break;
	case TD_MODE_ECHO_COLORS:
	  printf ("%s%s : %s\033[0m\n", type_color, message, type_txt);
	  break;
	case TD_MODE_ECHO_ALL_COLORS:
	  printf ("%s%s : %s\033[0m\n", type_color, message, type_txt);
	  break;
	case TD_MODE_ECHO_BW:
	  printf ("%s : %s\n", message, type_txt);
	  break;
	case TD_MODE_ECHO_ALL_BW:
	  printf ("%s : %s\n", message, type_txt);
	  break;
	default:
	  break;
	}
    }
  else
    g_print ("%s : %s\n", message, type_txt);

  /*** Arbre - Tree */
  if ((PLUGINS_CORE_MESSAGES) && (!TD_FLAG_NO_REFRESH_PALLETTE_MESSAGE))
    /* if ((PLUGINS_CORE_MESSAGES) && (!TD_FLAG_NO_REFRESH_PALLETTE_MESSAGE) && (APP) && (TD_APP (APP)->palette)) */
    {
      (gpointer*) symbol = dlsym (PLUGINS_CORE_MESSAGES, "plugins_new_message");
      symbol (message, type, type_txt);
    }
  else
    {
      START_MSG = g_list_append (START_MSG, message);
      START_MSG = g_list_append (START_MSG, GINT_TO_POINTER (type));
      START_MSG = g_list_append (START_MSG, type_txt);
    }

  /*** Bouton d'arrt d'urgence - Big red button */
  if (type == TD_MSG_STOP)
    td_exit (1);

  /*** Rafrachissement GTK+ - Refreshing GTK+ */
  while (gtk_events_pending())
    gtk_main_iteration();
}

/**
 * td_app_mod_core_symbol:
 * @app: td_app object (NULL => default app (APP))
 * @dir: directory location of plug-in (NULL=> default directory (TD_PACKAGE_PLUGINS_DIR))
 * @file: file location of plug-in
 * @symbol: symbol called
 * @warning: warning flag
 * 
 * fr: Retourne le pointer du symbole du greffon charg
 *
 * en: Returns pointer of symbol's plug-in loaded
 * 
 * Return value: pointer
 **/

gpointer *td_app_mod_core_symbol (TdApp *app, gchar *dir, gchar *file, gchar *symbol, gboolean warning)
{
  gpointer *ret;
  gboolean bool_tmp;
  int i;
  gchar *name;

  /*** Go ! */
  if (!dir)
    dir = TD_PACKAGE_PLUGINS_DIR;
  if (!app)
    app = TD_APP(APP);
  name = g_strdup_printf ("%s%s", dir, file);

  /*** Recherche du module - Searching the module */
  bool_tmp = FALSE;
  for (i=0; i<g_list_length (app->mod_core); i++)
    if (!strcmp ((gchar*) g_list_nth_data (app->mod_core, i), name))
      {
	bool_tmp = TRUE;
	break;
      }
  if (!bool_tmp)
    {
      td_app_message (_("Executing plug-in"), g_strdup_printf (_("Executing plug-in : file '%s' not loaded"), name), TD_MSG_FAILED);
      return NULL;
    }

  /*** Activation */
  ret = td_dlsym (PLUGINS_CORE[i], dir, file, symbol, warning);
  return ret;
}

/**
 * td_app_mod_core_activate:
 * @app: td_app object (NULL => default app (APP))
 * @dir: directory location of plug-in (NULL=> default directory (TD_PACKAGE_PLUGINS_DIR))
 * @file: file of plug-in
 * @symbol: symbol called
 * @warning: warning flag
 * 
 * fr: Execute le symbole du greffon charg
 *
 * en: Executes symbol's plug-in loaded
 * 
 * Return value: FALSE on error
 **/

gboolean td_app_mod_core_activate (TdApp *app, gchar *dir, gchar *file, gchar *symbol, gboolean warning)
{
  void (*symbol2)() = NULL;
  (gpointer*) symbol2 = td_app_mod_core_symbol (app, dir, file, symbol, warning);
  if (!symbol2)
    return FALSE;
  symbol2();
  return TRUE;
}

/**
 * td_app_mod_data_symbol:
 * @mod_id: module identifier
 * @symbol: symbol called
 * @warning: warning flag
 * 
 * fr: Retourne le pointer du symbole du greffon charg
 *
 * en: Returns pointer of symbol's plug-in loaded
 * 
 * Return value: pointer
 **/

gpointer *td_app_mod_data_symbol (int mod_id, gchar *symbol, gboolean warning)
{
  gpointer *ret;
  ret = td_dlsym (PLUGINS_DATA[mod_id], "", g_strdup_printf ("%s %d", _("data plugins"), mod_id), symbol, warning);
  return ret;
}

/**
 * td_app_mod_data_activate:
 * @mod_id: module identifier
 * @symbol: symbol called
 * @warning: warning flag
 * 
 * fr: Execute le symbole du greffon charg
 *
 * en: Executes symbol's plug-in loaded
 * 
 * Return value: FALSE on error
 **/

gboolean td_app_mod_data_activate (int mod_id, gchar *symbol, gboolean warning)
{
  void (*symbol2)() = NULL;
  (gpointer*) symbol2 = td_app_mod_data_symbol (mod_id, symbol, warning);
  if (!symbol2)
    return FALSE;
  symbol2();
  return TRUE;
}

/**
 * td_app_action_function:
 * @table: table
 * @mod: module
 * 
 * fr: Execute une modification gnre par une fonction
 *
 * en: Executes a modification generated by function
 **/

void td_app_action_function (gchar *table, GtkObject *mod)
{
  int i, j;
  gboolean bool_tmp;
  TdFunction *function_cur;
  void (*symbol)() = NULL;

  /*** Entre d'une fonction ? - function input ? */
  bool_tmp = FALSE;
  for (i=0; i<g_list_length (TD_APP (APP)->function_input); i++)
    if (!strcmp ((gchar*) g_list_nth_data (TD_APP (APP)->function_input, i), table))
      {
	bool_tmp = TRUE;
	break;
      }
  if (!bool_tmp)
    return;

  /*** Go ! */
  for (i=0; i<g_list_length (TD_APP (APP)->function); i++)
    for (j=0; j<g_list_length (TD_FUNCTION (g_list_nth_data (TD_APP (APP)->function, i))->input); j++)
      if (!strcmp ((gchar*) g_list_nth_data (TD_FUNCTION (g_list_nth_data (TD_APP (APP)->function, i))->input, j), table))
	{
	  if (!td_function_activate (TD_FUNCTION (g_list_nth_data (TD_APP (APP)->function, i))))
	    bool_tmp = FALSE;
	  break;
	}

  /*** Mise  jour - Update */
  if (!bool_tmp)
    return;
  (gpointer*) symbol = td_app_mod_data_symbol (TD_MOD (mod)->id, "plugins_function_action", TRUE);
  if (symbol)
    symbol (mod);
}

/**
 * td_app_mode_echo_parse:
 * @value: string
 * 
 * fr: Retourne le 'TdModeEcho' correspondant  la chane de caractres
 *
 * en: Returns the 'TdModeEcho' corresponding the string
 * 
 * Return value: mode echo
 **/

TdModeEcho td_app_mode_echo_parse (gchar *value)
{
  if (!strcmp (td_app_custom_value ("development", "terminal", "echo"), "no"))
    return TD_MODE_ECHO_NO;
  if (!strcmp (td_app_custom_value ("development", "terminal", "echo"), "colors"))
    return TD_MODE_ECHO_COLORS;
  if (!strcmp (td_app_custom_value ("development", "terminal", "echo"), "all, colors"))
    return TD_MODE_ECHO_ALL_COLORS;
  if (!strcmp (td_app_custom_value ("development", "terminal", "echo"), "black/white"))
    return TD_MODE_ECHO_BW;
  if (!strcmp (td_app_custom_value ("development", "terminal", "echo"), "all, black/white"))
    return TD_MODE_ECHO_ALL_BW;
  return TD_MODE_ECHO_0;
}

/*****************************************************************************/
/*** Personnalisation et paramtrage - Customizing and parameters */
/*****************************************************************************/

/**
 * td_app_custom_value:
 * @category: category
 * @group: group
 * @option: option
 * 
 * fr: Retourne la valeur de l'option de personnalisation de l'application
 *
 * en: Returns the value of the option of the application's customize
 * 
 * Return value: string
 **/

gchar *td_app_custom_value (gchar *category, gchar *group, gchar *option)
{
  return (td_custom_value ("app", category, group, option));
}

/*****************************************************************************/
/*** Verrou - Lock */
/*****************************************************************************/

/**
 * td_app_lock_add:
 * @app: application
 * 
 * fr: Ajoute le verrou de base de donnes
 *
 * en: Adds the database lock
 **/

static void td_app_lock_add (TdApp *app)
{
  int i, j;
  DIR *proc;
  struct dirent *entry;
  struct stat statb;
  GString *cmdline;
  gchar *locks_file;
  GtkObject *flower_new;
  GtkObject *flower_old;
  GList *list_tmp = NULL;
  gboolean bool_tmp;
  int crash = -1;
  int response = 0;
  gchar *txt_tmp;
  locks_file = g_strdup_printf ("%s/var/lock/toutdoux/locks.xml", TD_PACKAGE_DATA_DIR);
  g_return_if_fail (app != NULL);
  g_return_if_fail (TD_IS_APP (app));

  /*** Pid */
  flower_old = td_flower_new();
  gtk_object_set (GTK_OBJECT (flower_old), "root", "locks", "branch", "lock", NULL);
  td_flower_add_branch_attribut (TD_FLOWER (flower_old), "pid");
  td_flower_add_branch_attribut (TD_FLOWER (flower_old), "database");
  td_flower_add_branch_attribut (TD_FLOWER (flower_old), "file");
  td_flower_add_branch_attribut (TD_FLOWER (flower_old), "user");
  if (!td_flower_load_file (TD_FLOWER (flower_old), locks_file))
    {
      td_app_message (NULL, g_strdup_printf (_("Loading locks '%s'"), locks_file), TD_MSG_FAILED);
      app->pid == -1;
      return;
    }
  td_app_message (NULL, g_strdup_printf (_("Loading locks '%s'"), locks_file), TD_MSG_DONE);
  proc = opendir ("/proc");
  if (!proc)
    {
      td_app_message (NULL, g_strdup_printf (_("Loading locks '%s' : processes file system not find"), locks_file), TD_MSG_FAILED);
      app->pid == -1;
      return;
    }
  for (entry = readdir (proc); entry; entry = readdir (proc))
    {
      if ((atoi (entry->d_name) == 0) || (stat (g_strdup_printf ("/proc/%s", entry->d_name), &statb)<0) || (!S_ISDIR (statb.st_mode)))
	continue;
      cmdline = td_string_file (g_strdup_printf ("/proc/%s/cmdline", entry->d_name));
      if (!cmdline)
	continue;
      if (strstr (cmdline->str, "toutdoux"))
	/* if (!strcmp (td_strsplit (cmdline->str, " ", 0), "toutdoux")) */
	list_tmp = g_list_append (list_tmp, entry->d_name);
    }
  closedir (proc);
  for (i=0; i<g_list_length (list_tmp); i++)
    {
      bool_tmp = TRUE;
      for (j=0; j<g_list_length (TD_FLOWER (flower_old)->items_value); j++)
	if (!strcmp (g_list_nth_data (g_list_nth_data (TD_FLOWER (flower_old)->items_value, j), 0), g_list_nth_data (list_tmp, i)))
	  {
	    bool_tmp = FALSE;
	    break;
	  }
      if (bool_tmp)
	{
	  app->pid = atoi (g_list_nth_data (list_tmp, i));
	  break;
	}
    }
  if (app->pid == 0)
    {
      td_app_message (NULL, g_strdup_printf (_("Loading locks '%s' : process not find"), locks_file), TD_MSG_FAILED);
      app->pid == -1;
      return;
    }

  /*** Crash ? */
  for (i=0; i<g_list_length (TD_FLOWER (flower_old)->items_value); i++)
    {
      bool_tmp = TRUE;
      for (j=0; j<g_list_length (list_tmp); j++)
	if (!strcmp (g_list_nth_data (g_list_nth_data (TD_FLOWER (flower_old)->items_value, i), 0), g_list_nth_data (list_tmp, j)))
	  {
	    bool_tmp = FALSE;
	    break;
	  }
      if ((bool_tmp) && (!strcmp (g_list_nth_data (g_list_nth_data (TD_FLOWER (flower_old)->items_value, i), 3), g_get_user_name())))
	{
	  crash = i;
	  break;
	}
    }
  response = 2;
  if (crash > -1)
    {
      td_app_message ("Recovery", g_strdup_printf (_("Crash detected at pid '%s' for the database '%s' with file '%s'"),
						   (gchar*) g_list_nth_data (g_list_nth_data (TD_FLOWER (flower_old)->items_value, crash), 0),
						   (gchar*) g_list_nth_data (g_list_nth_data (TD_FLOWER (flower_old)->items_value, crash), 1),
						   (gchar*) g_list_nth_data (g_list_nth_data (TD_FLOWER (flower_old)->items_value, crash), 2)), TD_MSG_MESSAGE);
      if (!strcmp (td_app_custom_value ("database", "backend", "recovery"), "ask"))
	response = td_question_create_window (g_strdup_printf (_("ToutDoux appears to have crashed or been killed last time it was run.\n Would you like to recover the state of your last session ?\n process %s : database '%s' with file '%s'\n "),
							       (gchar*) g_list_nth_data (g_list_nth_data (TD_FLOWER (flower_old)->items_value, crash), 0),
							       (gchar*) g_list_nth_data (g_list_nth_data (TD_FLOWER (flower_old)->items_value, crash), 1),
							       (gchar*) g_list_nth_data (g_list_nth_data (TD_FLOWER (flower_old)->items_value, crash), 2)), _("Warning"));
      else
	{
	  if ((!strcmp (td_app_custom_value ("database", "backend", "recovery"), "never")) || (TD_FLAG_DEBUG_MODE))
	    response = 1;
	  else
	    response = 0;
	}
    }

  /*** Rcupration - Recovery */
  if (response == 0)
    {
      td_app_message (NULL, _("Recovering the last session"), TD_MSG_MESSAGE);
      app->pid_database = g_strdup_printf ("%s", (gchar*) g_list_nth_data (g_list_nth_data (TD_FLOWER (flower_old)->items_value, crash), 1));
      td_app_lock_remove (app, atoi (g_list_nth_data (g_list_nth_data (TD_FLOWER (flower_old)->items_value, crash), 0)));
      td_database_base_login (app->pid_database);
      gtk_object_set (GTK_OBJECT (app->connect),
		      "name", g_list_nth_data (g_list_nth_data (TD_FLOWER (flower_old)->items_value, crash), 2),
		      "file", g_list_nth_data (g_list_nth_data (TD_FLOWER (flower_old)->items_value, crash), 2), NULL);
      td_db_connect_set_type (TD_DB_CONNECT (app->connect), TD_CONNECT_TYPE_RECOVERY_START);
      td_flower_destroy (TD_FLOWER (flower_old));
      return;
    }

  /*** Destruction - Destroy */
  if (response == 1)
    {
      td_app_message (NULL, _("Destroying the last session"), TD_MSG_MESSAGE);
      app->pid_database = g_strdup_printf ("%s", (gchar*) g_list_nth_data (g_list_nth_data (TD_FLOWER (flower_old)->items_value, crash), 1));
      td_app_lock_remove (app, atoi (g_list_nth_data (g_list_nth_data (TD_FLOWER (flower_old)->items_value, crash), 0)));
      td_database_base_drop (app->pid_database);
      td_database_base_create (app->pid_database);
      td_database_base_login (app->pid_database);
      td_flower_destroy (TD_FLOWER (flower_old));
      return;
    }

  /*** Nouvelle base - New base */
  i=0;
  bool_tmp = FALSE;
  while (!bool_tmp)
    {
      bool_tmp = TRUE;
      txt_tmp = g_strdup_printf ("%s%d", td_app_custom_value ("database", "backend", "prefix"), i++);
      for (j=0; j<g_list_length (TD_FLOWER (flower_old)->items_value); j++)
	if (!strcmp (g_list_nth_data (g_list_nth_data (TD_FLOWER (flower_old)->items_value, j), 1), txt_tmp))
	  {
	    bool_tmp = FALSE;
	    break;
	  }
    }
  td_flower_destroy (TD_FLOWER (flower_old));
  app->pid_database = txt_tmp;
  td_app_lock_update (app, "");
  td_database_base_create (app->pid_database);
  td_database_base_login (app->pid_database);
}

/**
 * td_app_lock_update:
 * @file: file
 * @app: application
 * 
 * fr: Rafrachis le verrou de base de donnes
 *
 * en: Updates the database lock
 **/

static void td_app_lock_update (TdApp *app, gchar *file)
{
  int i;
  GtkObject *flower_new;
  GtkObject *flower_old;
  GList *list_tmp;
  gchar *locks_file;
  locks_file = g_strdup_printf ("%s/var/lock/toutdoux/locks.xml", TD_PACKAGE_DATA_DIR);
  g_return_if_fail (app != NULL);
  g_return_if_fail (TD_IS_APP (app));
  if (app->pid < 0)
    return;
  flower_old = td_flower_new();
  gtk_object_set (GTK_OBJECT (flower_old), "root", "locks", "branch", "lock", NULL);
  td_flower_add_branch_attribut (TD_FLOWER (flower_old), "pid");
  td_flower_add_branch_attribut (TD_FLOWER (flower_old), "database");
  td_flower_add_branch_attribut (TD_FLOWER (flower_old), "file");
  td_flower_add_branch_attribut (TD_FLOWER (flower_old), "user");
  if (!td_flower_load_file (TD_FLOWER (flower_old), locks_file))
    {
      td_app_message (NULL, g_strdup_printf (_("Loading locks '%s'"), locks_file), TD_MSG_FAILED);
      return;
    }
  flower_new = td_flower_new();
  gtk_object_set (GTK_OBJECT (flower_new), "root", "locks", "branch", "lock", NULL);
  td_flower_add_branch_attribut (TD_FLOWER (flower_new), "pid");
  td_flower_add_branch_attribut (TD_FLOWER (flower_new), "database");
  td_flower_add_branch_attribut (TD_FLOWER (flower_new), "file");
  td_flower_add_branch_attribut (TD_FLOWER (flower_new), "user");
  for (i=0; i<g_list_length (TD_FLOWER (flower_old)->items_value); i++)
    if (atoi (g_list_nth_data (g_list_nth_data (TD_FLOWER (flower_old)->items_value, i), 0)) != app->pid)
      td_flower_add_item (TD_FLOWER (flower_new), g_list_nth_data (TD_FLOWER (flower_old)->items_value, i), 0);
  list_tmp = NULL;
  list_tmp = g_list_append (list_tmp, g_strdup_printf ("%d", app->pid));
  list_tmp = g_list_append (list_tmp, app->pid_database);
  if (file)
    list_tmp = g_list_append (list_tmp, file);
  else
    list_tmp = g_list_append (list_tmp, "");
  list_tmp = g_list_append (list_tmp, g_get_user_name());
  td_flower_add_item (TD_FLOWER (flower_new), list_tmp, 0);
  if (!td_flower_save_file (TD_FLOWER (flower_new), locks_file))
    td_app_message (NULL, g_strdup_printf (_("Saving locks '%s'"), locks_file), TD_MSG_FAILED);
  else
    td_app_message (NULL, g_strdup_printf (_("Saving locks '%s'"), locks_file), TD_MSG_DONE);
  td_flower_destroy (TD_FLOWER (flower_old));
  td_flower_destroy (TD_FLOWER (flower_new));
}

/**
 * td_app_lock_remove:
 * @app: application
 * @pid: process identifier
 * 
 * fr: Supprime le verrou de base de donnes
 *
 * en: Removes the database lock
 **/

static void td_app_lock_remove (TdApp *app, int pid)
{
  int i;
  GString *cmdline;
  gchar *locks_file;
  GtkObject *flower_new;
  GtkObject *flower_old;
  locks_file = g_strdup_printf ("%s/var/lock/toutdoux/locks.xml", TD_PACKAGE_DATA_DIR);
  g_return_if_fail (app != NULL);
  g_return_if_fail (TD_IS_APP (app));
  if (pid<0)
    return;
  flower_old = td_flower_new();
  gtk_object_set (GTK_OBJECT (flower_old), "root", "locks", "branch", "lock", NULL);
  td_flower_add_branch_attribut (TD_FLOWER (flower_old), "pid");
  td_flower_add_branch_attribut (TD_FLOWER (flower_old), "database");
  td_flower_add_branch_attribut (TD_FLOWER (flower_old), "file");
  td_flower_add_branch_attribut (TD_FLOWER (flower_old), "user");
  if (!td_flower_load_file (TD_FLOWER (flower_old), locks_file))
    td_app_message (NULL, g_strdup_printf (_("Loading locks '%s'"), locks_file), TD_MSG_FAILED);
  else
    td_app_message (NULL, g_strdup_printf (_("Loading locks '%s'"), locks_file), TD_MSG_DONE);
  flower_new = td_flower_new();
  gtk_object_set (GTK_OBJECT (flower_new), "root", "locks", "branch", "lock", NULL);
  td_flower_add_branch_attribut (TD_FLOWER (flower_new), "pid");
  td_flower_add_branch_attribut (TD_FLOWER (flower_new), "database");
  td_flower_add_branch_attribut (TD_FLOWER (flower_new), "file");
  td_flower_add_branch_attribut (TD_FLOWER (flower_new), "user");
  for (i=0; i<g_list_length (TD_FLOWER (flower_old)->items_value); i++)
    if (atoi (g_list_nth_data (g_list_nth_data (TD_FLOWER (flower_old)->items_value, i), 0)) != pid)
      td_flower_add_item (TD_FLOWER (flower_new), g_list_nth_data (TD_FLOWER (flower_old)->items_value, i), 0);
  if (!td_flower_save_file (TD_FLOWER (flower_new), locks_file))
    td_app_message (NULL, g_strdup_printf (_("Loading locks '%s'"), locks_file), TD_MSG_FAILED);
  else
    td_app_message (NULL, g_strdup_printf (_("Loading locks '%s'"), locks_file), TD_MSG_DONE);
  td_flower_destroy (TD_FLOWER (flower_old));
  td_flower_destroy (TD_FLOWER (flower_new));
}
