/*****************************************************************************/
/*** database.c : Interface de base de donnes
 *** database.c : Database interface
 * 
 *
 * ToutDoux : Chtit gestionnaire de projet - A littl' project manager
 * Copyright (c) 2000-2001 Philippe Roy
 * Auteur - Author : Philippe Roy <ph_roy@toutdoux.org>
 *
 *
 * Ce programme est un logiciel libre ; vous pouvez le redistribuer et/ou le modifier
 * sous les termes de la licence publique gnrale GNU telle qu'elle est publie par
 * la Free Software Foundation ; soit la version 2 de la licence, ou
 * (comme vous voulez) toute version ultrieure.
 *
 * Ce programme est distribu dans l'espoir qu'il sera utile,
 * mais SANS AUCUNE GARANTIE ; mme sans la garantie de
 * COMMERCIALIT ou d'ADQUATION A UN BUT PARTICULIER. Voir la
 * licence publique gnrale GNU pour plus de dtails.
 *
 * Vous devriez avoir reu une copie de la licence publique gnrale GNU
 * avec ce programme ; si ce n'est pas le cas, crivez  la Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. */
/*****************************************************************************/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>
#include <dlfcn.h>

#include "commons.h"
#include "database.h"
#include "td_app.h"
#include "td_db_base.h"
#include "td_db_datatable.h"

/*****************************************************************************/
/*** Connection - Connexion */
/*****************************************************************************/

/**
 * td_database_base_login:
 * @name: name
 * 
 * fr: Connection  la base de donnes
 *
 * en: Logins to the database
 * 
 * Return value: FALSE on error
 **/

gboolean td_database_base_login (gchar *name)
{
  gboolean (*symbol)();
  (gpointer*) symbol = dlsym (PLUGINS_CORE_DATABASE, "plugins_base_login");
  return symbol (name);
}

/**
 * td_database_base_logout:
 * @name: name
 * 
 * fr: Dconnection de la base de donnes
 *
 * en: Logouts to the database
 * 
 * Return value: FALSE on error
 **/

gboolean td_database_base_logout (void)
{
  gboolean (*symbol)();
  (gpointer*) symbol = dlsym (PLUGINS_CORE_DATABASE, "plugins_base_logout");
  return symbol ();
}

/*****************************************************************************/
/*** Crer - Create */
/*****************************************************************************/

/**
 * td_database_base_create:
 * @name: name
 * 
 * fr: Cr la base
 *
 * en: Creates the base
 * 
 * Return value: FALSE on error
 **/

gboolean td_database_base_create (gchar *name)
{
  gboolean (*symbol)();
  (gpointer*) symbol = dlsym (PLUGINS_CORE_DATABASE, "plugins_base_create");
  return symbol (name);
}

/**
 * td_database_base_construct:
 * @base: base
 * 
 * fr: Cr les tables de la base
 *
 * en: Creates the base's tables
 * 
 * Return value: FALSE on error
 **/

gboolean td_database_base_construct (GtkObject *base)
{
  int i;
  for (i=0; i < g_list_length (TD_DB_BASE (base)->table); i++)
    if (!td_database_table_create (g_list_nth_data (TD_DB_BASE (base)->table, i)))
      return FALSE;
  return TRUE;
}

/**
 * td_database_table_create:
 * @table: table
 * 
 * fr: Cr la table
 *
 * en: Creates the table
 * 
 * Return value: FALSE on error
 **/

gboolean td_database_table_create (GtkObject *table)
{
  gboolean (*symbol)();
  (gpointer*) symbol = dlsym (PLUGINS_CORE_DATABASE, "plugins_table_create");
  return symbol (table);
}

/**
 * td_database_rule_create:
 * @rule: rule
 * 
 * fr: Cr la rgle
 *
 * en: Creates the rule
 * 
 * Return value: FALSE on error
 **/

gboolean td_database_rule_create (GtkObject *rule)
{
  gboolean (*symbol)();
  (gpointer*) symbol = dlsym (PLUGINS_CORE_DATABASE, "plugins_rule_create");
  return symbol (rule);
}

/**
 * td_database_function_create:
 * @function: function
 * 
 * fr: Cr la fonction
 *
 * en: Creates the function
 * 
 * Return value: FALSE on error
 **/

gboolean td_database_function_create (GtkObject *function)
{
  gboolean (*symbol)();
  (gpointer*) symbol = dlsym (PLUGINS_CORE_DATABASE, "plugins_function_create");
  return symbol (function);
}

/**
 * td_database_table_exist:
 * @name: name
 * 
 * fr: Retourne TRUE si la table existe
 *
 * en: Return TRUE if table exist
 * 
 * Return value: boolean
 **/

gboolean td_database_table_exist (gchar *name)
{
  gboolean (*symbol)();
  (gpointer*) symbol = dlsym (PLUGINS_CORE_DATABASE, "plugins_table_exist");
  return symbol (name);
}

/*****************************************************************************/
/*** Dtruit - Destroy */
/*****************************************************************************/

/**
 * td_database_base_drop:
 * @name: name
 * 
 * fr: Dtruit les tables de la base
 *
 * en: Destroys the base's tables
 * 
 * Return value: FALSE on error
 **/

gboolean td_database_base_drop (gchar *name)
{
  gboolean (*symbol)();
  (gpointer*) symbol = dlsym (PLUGINS_CORE_DATABASE, "plugins_base_drop");
  return symbol (name);
}

/**
 * td_database_base_raz:
 * 
 * fr: Met  zero la base
 *
 * en: Clears the base
 * 
 * Return value: FALSE on error
 **/

gboolean td_database_base_raz (void)
{
  gboolean (*symbol)();
  (gpointer*) symbol = dlsym (PLUGINS_CORE_DATABASE, "plugins_base_raz");
  return symbol();
}

/**
 * td_database_table_drop:
 * @table: table
 * 
 * fr: Dtruit la table
 *
 * en: Destroys the table
 * 
 * Return value: FALSE on error
 **/

gboolean td_database_table_drop (GtkObject *table)
{
  gboolean (*symbol)();
  (gpointer*) symbol = dlsym (PLUGINS_CORE_DATABASE, "plugins_table_drop");
  return symbol (table);
}

/**
 * td_database_rule_drop:
 * @rule: rule
 * 
 * fr: Dtruit la rgle
 *
 * en: Destroys the rule
 * 
 * Return value: FALSE on error
 **/

gboolean td_database_rule_drop (GtkObject *rule)
{
  gboolean (*symbol)();
  (gpointer*) symbol = dlsym (PLUGINS_CORE_DATABASE, "plugins_rule_drop");
  return symbol (rule);
}

/**
 * td_database_function_drop:
 * @function: function
 * 
 * fr: Dtruit la fonction
 *
 * en: Destroys the function
 * 
 * Return value: FALSE on error
 **/

gboolean td_database_function_drop (GtkObject *function)
{
  gboolean (*symbol)();
  (gpointer*) symbol = dlsym (PLUGINS_CORE_DATABASE, "plugins_function_drop");
  return symbol (function);
}

/*****************************************************************************/
/*** Remplissage de tables - Tables filling */
/*****************************************************************************/

/**
 * td_database_table_copy_in:
 * @datatable: datatable
 * 
 * fr: Peuple la base avec la table de donnes
 *
 * en: Populate the database with datatable
 * 
 * Return value: FALSE on error
 **/

gboolean td_database_table_copy_in (GtkObject *datatable)
{
  gboolean (*symbol)();
  (gpointer*) symbol = dlsym (PLUGINS_CORE_DATABASE, "plugins_table_copy_in");
  return symbol (datatable);
}

/*****************************************************************************/
/*** Requte - Query */
/*****************************************************************************/

/**
 * td_database_test_query:
 * @query: query
 * 
 * fr: Test l'existence de la requette
 *
 * en: Test the query's existence
 * 
 * Return value: boolean
 **/

gboolean td_database_test_query (gchar *query)
{
  if (TD_FLAG_DEBUG_MODE)
    return TRUE;
  if ((!query) || (!strcmp (query, "(null)")))
    return FALSE;
  return TRUE;
}

/**
 * td_database_command:
 * @query: query
 * 
 * fr: Execute la requte commande (sans rsultat)
 *
 * en: Execute the command (without result) query
 * 
 * Return value: FALSE on error
 **/

gboolean td_database_command (gchar *query)
{
  gboolean (*symbol)();
  (gpointer*) symbol = dlsym (PLUGINS_CORE_DATABASE, "plugins_command");
  if (!td_database_test_query (query))
    return TRUE;

  /*** Cache SQL - SQL cache */
  while (strstr (query, "TD_CURRENT_ID"))
    query = td_string_replace (query, "TD_CURRENT_ID", g_strdup_printf ("%d", CURRENT_ID));
  while (strstr (query, "TD_CURRENT_ID_PARENT"))
    query = td_string_replace (query, "TD_CURRENT_ID_PARENT", g_strdup_printf ("%d", CURRENT_ID_PARENT));

  /*** Go ! */
  return symbol (query);
}

/**
 * td_database_insert:
 * @query: query
 * @table: table
 * 
 * fr: Execute la requte d'insertion
 *
 * en: Execute the insert query
 * 
 * Return value: FALSE on error
 **/

gboolean td_database_insert (gchar *query, gchar *table)
{
  gboolean (*symbol)();
  if (!td_database_test_query (query))
    return TRUE;
  (gpointer*) symbol = dlsym (PLUGINS_CORE_DATABASE, "plugins_insert");
  return symbol (query, table);
}

/**
 * td_database_select:
 * @query: query
 * 
 * fr: Execute la requte de slection
 *
 * en: Execute the select query
 * 
 * Return value: array
 **/

GtkObject *td_database_select (gchar *query)
{
  GtkObject *(*symbol)();
  (gpointer*) symbol = dlsym (PLUGINS_CORE_DATABASE, "plugins_select");
  if (!td_database_test_query (query))
    return NULL;

  /*** Cache SQL - SQL cache */
  while (strstr (query, "TD_CURRENT_ID"))
    query = td_string_replace (query, "TD_CURRENT_ID", g_strdup_printf ("%d", CURRENT_ID));
  while (strstr (query, "TD_CURRENT_ID_PARENT"))
    query = td_string_replace (query, "TD_CURRENT_ID_PARENT", g_strdup_printf ("%d", CURRENT_ID_PARENT));

  /*** Go ! */
  return symbol (query);
}

/**
 * td_database_field:
 * @query: query
 * 
 * fr: Retourne la liste des champs de la requte de slection
 *
 * en: Return the field-list of the select query
 * 
 * Return value: list
 **/

GList *td_database_field (gchar *query)
{
  GList *(*symbol)();
  (gpointer*) symbol = dlsym (PLUGINS_CORE_DATABASE, "plugins_field");
  if (!td_database_test_query (query))
    return NULL;

  /*** Cache SQL - SQL cache */
  while (strstr (query, "TD_CURRENT_ID"))
    query = td_string_replace (query, "TD_CURRENT_ID", g_strdup_printf ("%d", CURRENT_ID));
  while (strstr (query, "TD_CURRENT_ID_PARENT"))
    query = td_string_replace (query, "TD_CURRENT_ID_PARENT", g_strdup_printf ("%d", CURRENT_ID_PARENT));

  /*** Go ! */
  return symbol (query);
}

/**
 * td_database_row:
 * @query: query
 * 
 * fr: Retourne la liste des valeur de la premire instance de la requte de slection
 *
 * en: Return the values list of the first row of the select query
 * 
 * Return value: list
 **/

GList *td_database_row (gchar *query)
{
  GList *(*symbol)();
  (gpointer*) symbol = dlsym (PLUGINS_CORE_DATABASE, "plugins_row");
  if (!td_database_test_query (query))
    return NULL;

  /*** Cache SQL - SQL cache */
  while (strstr (query, "TD_CURRENT_ID"))
    query = td_string_replace (query, "TD_CURRENT_ID", g_strdup_printf ("%d", CURRENT_ID));
  while (strstr (query, "TD_CURRENT_ID_PARENT"))
    query = td_string_replace (query, "TD_CURRENT_ID_PARENT", g_strdup_printf ("%d", CURRENT_ID_PARENT));

  /*** Go ! */
  return symbol (query);
}

/**
 * td_database_column:
 * @query: query
 * 
 * fr: Retourne la liste des valeur du premier champs de la requte de slection
 *
 * en: Return the values list of the first field of the select query
 * 
 * Return value: list
 **/

GList *td_database_column (gchar *query)
{
  GList *(*symbol)();
  (gpointer*) symbol = dlsym (PLUGINS_CORE_DATABASE, "plugins_column");
  if (!td_database_test_query (query))
    return NULL;

  /*** Cache SQL - SQL cache */
  while (strstr (query, "TD_CURRENT_ID"))
    query = td_string_replace (query, "TD_CURRENT_ID", g_strdup_printf ("%d", CURRENT_ID));
  while (strstr (query, "TD_CURRENT_ID_PARENT"))
    query = td_string_replace (query, "TD_CURRENT_ID_PARENT", g_strdup_printf ("%d", CURRENT_ID_PARENT));
  
  /*** Go ! */
  return symbol (query);
}

/**
 * td_database_value:
 * @query: query
 * 
 * fr: Retourne la premier valeur de la requte de slection
 *
 * en: Return the fisrt value of the select query
 * 
 * Return value: list
 **/

gchar *td_database_value (gchar *query)
{
  gchar *(*symbol)();
  if (!td_database_test_query (query))
    return NULL;

  /*** Macros SQL interne - Internal SQL macros */
  if (strstr (query, "TD_CURRENT"))
    return td_database_current (query);

  /*** Go ! */
  (gpointer*) symbol = dlsym (PLUGINS_CORE_DATABASE, "plugins_value");
  return symbol (query);
}

/**
 * td_database_set_current:
 * @query: query
 * 
 * fr: Place dans le cache (TD_CURRENT) l'instance de la requte de slection
 *
 * en: Put in cache (TD_CURRENT) the row of the select query
 **/

void td_database_set_current (gchar *query)
{
  void (*symbol)();
  if (!td_database_test_query (query))
    return;
  (gpointer*) symbol = dlsym (PLUGINS_CORE_DATABASE, "plugins_set_current");
  symbol (query);
}

/**
 * td_database_current:
 * @query: query
 * 
 * fr: Retoune une valeur du cache (TD_CURRENT) avec une syntaxe SQL
 *
 * en: Return on value of cache (TD_CURRENT) with SQL syntax
 * 
 * Return value: string
 **/

gchar *td_database_current (gchar *query)
{
  int i, j;
  gchar *query2;
  gchar *field;
  gchar *where;
  gchar *membre1;
  gchar *membre2;

  /*** CURRENT = NULL */
  if ((!strcmp (query, "TD_CURRENT (*);")) ||
      ((!strcmp (query, "TD_CURRENT (null);")) && (!CURRENT_ROW)))
    {
      td_app_message (NULL, g_strdup_printf ("%s '%s'", _("Simulating in local database td_query"), query), TD_MSG_SQL_CACHE);
      return "(ok)";
    }
  if ((!strcmp (query, "TD_CURRENT (null);")) && (CURRENT_ROW))
    {
      td_app_message (NULL, g_strdup_printf ("%s '%s'", _("Simulating in local database td_query"), query), TD_MSG_SQL_CACHE);
      return NULL;
    }
  if (!CURRENT_ROW)
    {
      td_app_message (NULL, g_strdup_printf ("%s '%s'", _("Simulating in local database td_query"), query), TD_MSG_SQL_CACHE);
      return NULL;
    }
    
  /*** Requte sans condition - Query without condition */
  if (!strstr (query, "WHERE"))
    {
      field = strpbrk (query, " ");
      field = g_strndup (field+1, strlen (field)-2);
      for (i=0; i< g_list_length (CURRENT_FIELD); i++)
	if (!strcmp ((gchar*) g_list_nth_data (CURRENT_FIELD, i), field))
	  {
	    td_app_message (NULL, g_strdup_printf ("%s '%s'", _("Simulating in local database td_query"), query), TD_MSG_SQL_CACHE);
	    return g_list_nth_data (CURRENT_ROW, i);
	  }
    }

  /*** Requte avec une condition ngative - Query with negative condition */
  if (strstr (query, "WHERE NOT"))
    {
      query2 = strpbrk (query, " ");
      query2 = g_strndup (query2+1, strlen (query2)-2);
      field = g_strdup_printf ("%s", query2);
      where = strtok (field, "WHERE");
      where = strtok (NULL, "NOT");
      where = strtok (NULL, "NOT");
      field = g_strndup (field, strlen (field)-1);
      where = g_strndup (where+1, strlen (where)-1);
      membre1 = g_strdup_printf ("%s", where);
      membre2 = strtok (membre1, "=");
      membre2 = strtok (NULL, "=");
      membre1 = g_strndup (membre1+1, strlen (membre1)-2);
      membre2 = g_strndup (membre2+2, strlen (membre2)-4);
      for (i=0; i< g_list_length (CURRENT_FIELD); i++)
	if (!strcmp ((gchar*) g_list_nth_data (CURRENT_FIELD, i), field))
	  for (j=0; j< g_list_length (CURRENT_FIELD); j++)
	    if (!strcmp ((gchar*) g_list_nth_data (CURRENT_FIELD, j), membre1))
	      {
		if (strcmp ((gchar*)g_list_nth_data (CURRENT_ROW, j), membre2))
		  {
		    td_app_message (NULL, g_strdup_printf ("%s '%s'", _("Simulating in local database td_query"), query), TD_MSG_SQL_CACHE);
		    return g_list_nth_data (CURRENT_ROW, i);
		  }
		else
		  {
		    td_app_message (NULL, g_strdup_printf ("%s '%s'", _("Simulating in local database td_query"), query), TD_MSG_SQL_CACHE);
		    return NULL;
		  }
	      }
      td_app_message (NULL, g_strdup_printf ("%s '%s'", _("Simulating in local database td_query"), query), TD_MSG_SQL_CACHE);
      return NULL;
    }

  /*** Requte avec une condition positive - Query with positive condition */
  if (strstr (query, "WHERE"))
    {
      query2 = strpbrk (query, " ");
      query2 = g_strndup (query2+1, strlen (query2)-2);
      field = g_strdup_printf ("%s", query2);
      where = strtok (field, "WHERE");
      where = strtok (NULL, "WHERE");
      field = g_strndup (field, strlen (field)-1);
      where = g_strndup (where+1, strlen (where)-1);
      membre1 = g_strdup_printf ("%s", where);
      membre2 = strtok (membre1, "=");
      membre2 = strtok (NULL, "=");
      membre1 = g_strndup (membre1+1, strlen (membre1)-2);
      membre2 = g_strndup (membre2+2, strlen (membre2)-4);
      for (i=0; i< g_list_length (CURRENT_FIELD); i++)
	if (!strcmp ((gchar*) g_list_nth_data (CURRENT_FIELD, i), field))
	  for (j=0; j< g_list_length (CURRENT_FIELD); j++)
	    if (!strcmp ((gchar*) g_list_nth_data (CURRENT_FIELD, j), membre1))
	      {
		if (!strcmp ((gchar*) g_list_nth_data (CURRENT_ROW, j), membre2))
		  {
		    td_app_message (NULL, g_strdup_printf ("%s '%s'", _("Simulating in local database td_query"), query), TD_MSG_SQL_CACHE);
		    return g_list_nth_data (CURRENT_ROW, i);
		  }
		else
		  {
		    td_app_message (NULL, g_strdup_printf ("%s '%s'", _("Simulating in local database td_query"), query), TD_MSG_SQL_CACHE);
		    return NULL;
		  }
	      }
      td_app_message (NULL, g_strdup_printf ("%s '%s'", _("Simulating in local database td_query"), query), TD_MSG_SQL_CACHE);
      return NULL;
    }
  return NULL;
}

/*****************************************************************************/
/*** Chane de caractre - String */
/*****************************************************************************/

/**
 * td_database_strescape
 * @source: original string
 * 
 * fr: Ajoute un '\' par '\' ou ''' ou '"'
 *
 * en: Adds a '\' per '\' or ''' or '"'
 * 
 * Return value: string adjusted
 **/

gchar *td_database_strescape (gchar *source)
{
  /*** THANKS: fonction rcrite par - function rewritten by
       Fichier - File: patch pour hOpla - patch for hOpla
       Copyright : Copyright (C) 1998 The Free Software Foundation
       Auteur - Author: Guy Zrcher <gzuercher@raptus.com> */

  gchar *srcpos = (guchar *) source;
  gchar *dest = g_malloc (strlen (source) * 4 + 1);
  gchar *dstpos = dest;
  
  while (*srcpos)
    {
      if ((*srcpos == '\'') || /*** guillemets simples - simple quotes */
	  (*srcpos == '\"') || /*** guillemets doubles - double quotes */
	  (*srcpos == '\\'))  /*** antislashes - backslashes */
	*dstpos++ = '\\';
      
      *dstpos++ = *srcpos;
      srcpos++;
    }
  *dstpos++ = 0;
  return dest;
}

/**
 * td_database_adaptvalue
 * @value: original string
 * @type: type
 * 
 * fr: Ajuste la valeur pour former un requte SQL valide
 *
 * en: Ajusts the value for making a valid SQL query
 * 
 * Return value: string adjusted
 **/

gchar *td_database_adaptvalue (gchar *value, gchar *type)
{
  /*** THANKS: fonction rcrite par - function rewritten by
       Fichier - File: patch pour hOpla - patch for hOpla
       Copyright : Copyright (C) 1998 The Free Software Foundation
       Auteur - Author: Guy Zrcher <gzuercher@raptus.com> */

  int day, month, year;

  /*** boolean, intervale */
  if ((strstr (type, "bool")) ||
      (strstr (type, "interval")))
    {
      value = g_strdup_printf ("'%s'", value);
      return (value);
    }

  /*** date */
  if (strstr (type, "date"))
    {
      sscanf (value, "%d-%d-%d", &year, &month, &day);
      value = g_strdup_printf ("'%04d-%02d-%02d'", year, month, day);
      return (value);
    }

  /*** timestamp */
  if (strstr (type, "timestamp"))
    {
      if (strlen (value) == 0)
	value="null";
      else
	value = g_strdup_printf ("'%s'", value);
      return (value);
    }

  /*** char(n), varchar(n), text */
  if ((strstr (type, "char")) ||
      (strstr (type, "text")))
    {
      value = td_database_strescape (value);
      value = g_strdup_printf ("'%s'", value);
      return (value);
   }

  return (value);
}

/*****************************************************************************/
/*** Arborescence - Tree structure */
/*****************************************************************************/

/**
 * td_datatable_tree_sorted
 * @table: table's name
 * @table_tree: table_tree's name
 * @query: query for datatable
 * @query_tree: query for tree structure
 * @column_oid: column identifier of datatable
 * 
 * fr: Retourne le table des valeurs classe dans l'ordre de l'arborescence
 *
 * en: Returns the datatable sorted with the tree order
 * 
 * Return value: datatable
 **/

GtkObject *td_datatable_tree_sorted (gchar *table, gchar *table_tree, gchar *query, gchar *query_tree, int column_oid)
{
  GtkObject *ret;
  GtkObject *datatable;
  GtkObject *datatable_tree;
  GList *order = NULL;
  gchar *txt_tmp;
  gboolean bool_tmp;
  int i,j;

  /*** Classement - Sorting */
  ret = td_db_datatable_new();
  datatable = td_database_select (query);
  datatable_tree = td_database_select (query_tree);
  txt_tmp = "0";
  order = NULL;
  bool_tmp = TRUE;
  while (bool_tmp)
    {
      bool_tmp = FALSE;
      for (i=0; i<g_list_length (TD_DB_DATATABLE (datatable_tree)->item); i++)
	if (!strcmp ((gchar*) g_list_nth_data (g_list_nth_data (TD_DB_DATATABLE (datatable_tree)->item, i), 2), txt_tmp))
	  {
	    order = g_list_append (order, g_list_nth_data (g_list_nth_data (TD_DB_DATATABLE (datatable_tree)->item, i), 0));
	    txt_tmp =  g_list_nth_data (g_list_nth_data (TD_DB_DATATABLE (datatable_tree)->item, i), 0);
	    bool_tmp = TRUE;
	    break;
	  }
    }

  /*** Inversion - Invert */
  for (i=g_list_length (order)-1; i!=-1; i--)
    for (j=0; j<g_list_length (TD_DB_DATATABLE (datatable)->item); j++)
      if (!strcmp ((gchar*) g_list_nth_data (g_list_nth_data (TD_DB_DATATABLE (datatable)->item, j), column_oid), (gchar*) g_list_nth_data (order, i)))
	{
	  td_db_datatable_add_item (TD_DB_DATATABLE (ret), g_list_nth_data (TD_DB_DATATABLE (datatable)->item, j));
	  break;
	}
  return ret;
}

/**
 * td_datatable_tree_childs
 * @fields: fields
 * @start_id: first node's identifier
 * @table: table's name
 * @table_tree: table_tree's name
 * 
 * fr: Retourne les valeurs des enfants
 *
 * en: Returns the values of childs
 * 
 * Return value: list
 **/

GList *td_datatable_tree_childs (gchar *fields, int start_id, gchar *table, gchar *table_tree)
{
  int i, j;
  GList *ret = NULL;
  GtkObject *datatable_tmp;
  GList *id_parent = NULL;
  GList *id_parent2;

  /*** Enfants - Childs */
  id_parent = g_list_append (id_parent, GINT_TO_POINTER (start_id));
  i = 0;
  while (g_list_nth_data (id_parent, i))
    {
      id_parent2 = td_database_column (g_strdup_printf ("SELECT td_id FROM %s WHERE td_id_parent = %d;", 
							table_tree, GPOINTER_TO_INT (g_list_nth_data (id_parent, i))));
      for (j=0; j<g_list_length (id_parent2); j++)
	id_parent = g_list_append (id_parent, GINT_TO_POINTER (atoi (g_list_nth_data (id_parent2, j))));
      i++;
    }

  /*** Donnes - Data */
  if (strstr (fields, ","))
    {
      for (i=0; i<g_list_length (id_parent); i++)
	ret = g_list_append (ret, td_database_row (g_strdup_printf ("SELECT %s FROM %s WHERE td_id = %d;", 
								    fields, table, GPOINTER_TO_INT (g_list_nth_data (id_parent, i)))));
      return ret;
    }
  for (i=0; i<g_list_length (id_parent); i++)
    ret = g_list_append (ret, td_database_value (g_strdup_printf ("SELECT %s FROM %s WHERE td_id = %d;", 
								  fields, table, GPOINTER_TO_INT (g_list_nth_data (id_parent, i)))));
  return ret;
}
