/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */

package org.gnu.gtkhtml;

import java.util.Vector;
import org.gnu.glib.EventMap;
import org.gnu.glib.Handle;
import org.gnu.gtk.Layout;
import org.gnu.gtkhtml.event.HTMLViewEvent;
import org.gnu.gtkhtml.event.HTMLViewListener;

public class HTMLView extends Layout {
	
    /**
     * Listeners for handling document events
     */
    private Vector docListeners = null;

    /**
     * Map to collect the events.
     */
    private static EventMap evtMap = new EventMap();
    static {
        addEvents(evtMap);
    }

    public HTMLView() {
        super(html_view_new());
    }
	
    public void setDocument(HTMLDocument document) {
        if (document == null) 
            html_view_set_document(getHandle(), null);
        else
            html_view_set_document(getHandle(), document.getHandle());
    }
	
    public void jumpToAnchor(String anchor) {
        html_view_jump_to_anchor(getHandle(), anchor);
    }
	
    public double getMagnification() {
        return html_view_get_magnification(getHandle());
    }
	
    public void setMagnification(double magnification) {
        html_view_set_magnification(getHandle(), magnification);
    }
	
    public void zoomIn() {
        html_view_zoom_in(getHandle());
    }
	
    public void zoomOut() {
        html_view_zoom_out(getHandle());
    }
	
    public void zoomReset() {
        html_view_zoom_reset(getHandle());
    }

    public String getSelectedText() {
        return html_selection_get_text(getHandle());
    }

    public void addListener( HTMLViewListener listener ) {
        int i = findListener(docListeners, listener);
        if (i == -1) {
            if (null == docListeners) {
                evtMap.initialize(this, HTMLViewEvent.Type.ON_URL);
                docListeners = new Vector();
            }
            docListeners.addElement( listener );
        }
    }

    public void removeListener(HTMLViewListener listener) {
        int i = findListener(docListeners, listener);
        if (i > -1) {
            docListeners.remove(i);
        }
        if (0 == docListeners.size()) {
            evtMap.uninitialize(this, HTMLViewEvent.Type.ON_URL);
            docListeners = null;
        }
    }

    /**
     * Give us a way to locate a specific listener in a Vector.
     * @param list The Vector of listeners to search.
     * @param listener The object that is to be located in the Vector.
     * @return Returns the index of the listener in the Vector, or -1 if
     *                 the listener is not contained in the Vector.
     */
    protected static int findListener(Vector list, Object listener) {
        if (null == list || null == listener)
            return -1;
        return list.indexOf(listener);
    }

    /**
     * Implementation method to build an EventMap for this widget class.
     * Not useful (or supported) for application use.
     */
    private static void addEvents(EventMap anEvtMap) {
        anEvtMap.addEvent("on_url", "handleOnURL", 
                          HTMLViewEvent.Type.ON_URL, 
                          HTMLViewListener.class);
    }

    protected void fireViewEvent(HTMLViewEvent event) {
        if (null == docListeners) {
            return;
        }
        int size = docListeners.size();
        int i = 0;
        while (i < size) {
            HTMLViewListener vl = 
                (HTMLViewListener)docListeners.elementAt(i);
            vl.viewEvent(event);
            i++;
        }
    }

    protected void handleOnURL(String url) {
        HTMLViewEvent anEvent = new HTMLViewEvent(this, HTMLViewEvent.Type.ON_URL);
        anEvent.setURL(url);
        fireViewEvent(anEvent);
    }

    native static final protected int html_view_get_type();
    native static final protected Handle html_view_new();
    native static final protected void html_view_set_document(Handle view, Handle document);
    native static final protected void html_view_jump_to_anchor(Handle view, String anchor);
    native static final protected double html_view_get_magnification(Handle view);
    native static final protected void html_view_set_magnification(Handle view, double magnification);
    native static final protected void html_view_zoom_in(Handle view);
    native static final protected void html_view_zoom_out(Handle view);
    native static final protected void html_view_zoom_reset(Handle view);
    native static final protected Handle html_view_find_layout_box(Handle view, Handle node, boolean findParent);
    native static final protected void html_view_scroll_to_node(Handle view, Handle node, int scrollType);
    native static final protected String html_selection_get_text(Handle view);

    static {
        System.loadLibrary("gtkhtmljni-2.6");
    }
}
