/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */

package org.gnu.gtk;

import org.gnu.glib.GObject;
import org.gnu.glib.Handle;
import org.gnu.glib.Type;

/**
 * The FileSelection widget lists all the files and directories in
 * the current working directory and enables the user to select one of
 * the file names. There are a number of navigation controls that
 * enable the user to browse through the entire file system. There also
 * are buttons that create directories, delete files, and rename files.
 */
public class FileSelection extends Dialog {
	
	/**
	 * construct a new FileSelection dialog.
	 * 
	 * @param title The string that is displayed in the title bar of the dialog.
	 */
	public FileSelection(String title) {
		super(gtk_file_selection_new(title));
	}
	
	/**
	 * Construct a FileSelection using a handle to a native resource.
	 */
	public FileSelection(Handle handle) {
	    super(handle);
	}

	/**
	 * Sets the default path for the file requestor.  If filename includes a
	 * directory path the requestor will open with that path set as its 
	 * current working directory.
	 * 
	 * @param filename The default path for the widget.
	 */
	public void setFilename(String filename) {
		FileSelection.gtk_file_selection_set_filename(getHandle(), filename);
	}
	
	/**
	 * Returns the selected filename.
	 * 
	 * @return The file that is selected in the dialog.
	 */
	public String getFilename() {
		return FileSelection.gtk_file_selection_get_filename(getHandle());
	}
	
	/**
	 * Will attempt to match <code>pattern</code> to a valid filename or subdirectory
	 * in the current directory.  If a match can be made the matched filename will appear
	 * in the text entry field in the file selection dialog.  If a partial match can be made the
	 * "Files" list will contain those file names which have been partially matched and the
	 * "Directories" list will list those directories which have been partially matched.
	 * 
	 * @param pattern The pattern to use for matching.
	 */
	public void complete(String pattern) {
		FileSelection.gtk_file_selection_complete(getHandle(), pattern);
	}
	
	/**
	 * Shows the file operation buttons, if they have previously been hidden.  The rest
	 * of the widgets in the dialog will be resized accordingly.
	 */
	public void showFileopButtons() {
		FileSelection.gtk_file_selection_show_fileop_buttons(getHandle());
	}
	
	/**
	 * Hides the file operation buttons that normally appear at the top of the dialog.
	 */
	public void hideFileopButtons() {
		FileSelection.gtk_file_selection_hide_fileop_buttons(getHandle());
	}
	
	/**
	 * Return the OK Button widget for this Dialog.
	 * 
	 * @return The OK Button.
	 */
	public Button getOKButton() {
	    Handle hndl = FileSelection.getOkButton(getHandle());
		GObject obj = getGObjectFromHandle(hndl);
		if (null != obj)
			return (Button)obj;
		return new Button(hndl);
	}
	
	/**
	 * Return the Cancel Button widget for this dialog.
	 * 
	 * @return The Cancel Button.
	 */
	public Button getCancelButton() {
	    Handle hndl = FileSelection.getCancelButton(getHandle());
		GObject obj = getGObjectFromHandle(hndl);
		if (null != obj)
			return (Button)obj;
		return new Button(hndl);
	}
	
	/**
	 * Return the Help Button widget for this dialog.
	 * 
	 * @return The Help Button.
	 */
	public Button getHelpButton() {
	    Handle hndl = FileSelection.getHelpButton(getHandle());
		GObject obj = getGObjectFromHandle(hndl);
		if (null != obj)
			return (Button)obj;
		return new Button(hndl);
	}
	

	/**
	 * Return the directory list for this widget
	 *
	 * @return The Directory List
	 */
	public Widget getDirList(){
	    Handle hndl = getDirList(getHandle());
		if (null == hndl)
			return null;
		GObject obj = getGObjectFromHandle(hndl);
		if (null != obj)
			return (Widget)obj;
		return new Widget(hndl);
	}

	/**
	 * Returns the the file list for this widget
	 *
	 * @return The File List
	 */
	public Widget getFileList(){
	    Handle hndl = getFileList(getHandle());
		if (null == hndl)
			return null;
		GObject obj = getGObjectFromHandle(hndl);
		if (null != obj)
			return (Widget)obj;
		return new Widget(hndl);
	}

	/**
	 * Returns the selection entry
	 *
	 * @return selection entry
	 */
	public Widget getSelectionEntry(){
	    Handle hndl = getSelectionEntry(getHandle());
		if (null == hndl)
			return null;
		GObject obj = getGObjectFromHandle(hndl);
		if (null != obj)
			return (Widget)obj;
		return new Widget(hndl);
	}

    /**
     * Retrieves the list of file selections the user has made in the
     * dialog box. This function is intended for use when the user can
     * select multiple files in the file list.
     */
    public String[] getSelections() {
        return gtk_file_selection_get_selections( getHandle() );
    }
    /**
     * Sets whether the user is allowed to select multiple files in
     * the file list. Use {@link #getSelections} to get
     * the list of selected files.
     */
    public void setSelectMultiple( boolean selectMultiple ) {
        gtk_file_selection_set_select_multiple( getHandle(), selectMultiple );
    }
    /**
     * Determines whether or not the user is allowed to select
     * multiple files in the file list.
     * @see #setSelectMultiple
     */
    public boolean getSelectMultiple() {
        return gtk_file_selection_get_select_multiple( getHandle() );
    }

	/**
	 * Retrieve the runtime type used by the GLib library.
	 */
	public static Type getType() {
		return new Type(gtk_file_selection_get_type());
	}


    native static final protected Handle getDirList (Handle cptr);
    native static final protected Handle getFileList (Handle cptr);
    native static final protected Handle getSelectionEntry (Handle cptr);
    native static final protected Handle getSelectionText (Handle cptr);
    native static final protected Handle getMainVbox (Handle cptr);
    native static final protected Handle getOkButton (Handle cptr);
    native static final protected Handle getCancelButton (Handle cptr);
    native static final protected Handle getHelpButton (Handle cptr);
    native static final protected Handle getHistoryPulldown (Handle cptr);
    native static final protected Handle getHistoryMenu (Handle cptr);
    native static final protected Handle getButtonArea (Handle cptr);
    native static final protected int gtk_file_selection_get_type ();
    native static final protected Handle gtk_file_selection_new (String title);
    native static final protected void gtk_file_selection_set_filename (Handle filesel, String filename);
    native static final protected String gtk_file_selection_get_filename (Handle filesel);
    native static final protected void gtk_file_selection_complete (Handle filesel, String pattern);
    native static final protected void gtk_file_selection_show_fileop_buttons (Handle filesel);
    native static final protected void gtk_file_selection_hide_fileop_buttons (Handle filesel);

    native static final private String[] gtk_file_selection_get_selections(Handle filesel);
    native static final private void gtk_file_selection_set_select_multiple(Handle filesel, boolean select_multiple);
    native static final private boolean gtk_file_selection_get_select_multiple(Handle filesel);

}

